import struct
import six

from ..pickle import dumps, loads


__all__ = [
    "StreamBase",
]

sFloat = struct.Struct("f")
sDouble = struct.Struct("d")
sInt = struct.Struct("<I")
sBEInt = struct.Struct(">I")


class StreamBase(object):
    def __init__(self, slave):
        self.setSlave(slave)

    def setSlave(self, slave):
        self.slave = slave
        if slave:
            self.write = slave.send if hasattr(slave, "send") else slave.write if hasattr(slave, "write") else None
            self.read = slave.recv if hasattr(slave, "recv") else slave.read if hasattr(slave, "read") else None
            self.readBuf = slave.readBuf if hasattr(slave, "readBuf") else self.read
        else:
            self.write = None
            self.read = None
            self.readBuf = None

    def writeBool(self, b):
        self.write(b"\x01" if b else b"\x00")

    def readBool(self):
        return ord(self.readBuf(1)[:1]) != 0

    def readNFloat(self):
        """Reads native float"""
        return sFloat.unpack(self.readBuf(sFloat.size))[0]

    def readNFloats(self, l):
        s = struct.Struct("f" * l)
        return s.unpack(self.readBuf(s.size))

    def writeNFloat(self, f):
        """Writes native float"""
        self.write(sFloat.pack(f))

    def readNDouble(self):
        """Reads native double"""
        return sDouble.unpack(self.readBuf(sDouble.size))[0]

    def writeNDouble(self, d):
        """Writes native double"""
        self.write(sDouble.pack(d))

    def readNInt(self):
        """Reads native unsigned int"""
        return sInt.unpack(self.readBuf(sInt.size))[0]

    def writeNInt(self, i):
        """Writes native unsigned int"""
        self.write(sInt.pack(i))

    def readBEInt(self):
        """Reads native unsigned int"""
        return sBEInt.unpack(self.readBuf(sBEInt.size))[0]

    def readBELInt(self):
        """Effectively reads very long ints in direct byte order"""
        bytes = self.read(self.readBEInt())
        res = 0
        for byte in six.moves.xrange(len(bytes)):
            res = (res << 8) + ord(bytes[byte:byte + 1])
        return res

    def writeBEInt(self, i):
        """Writes native unsigned int"""
        self.write(sBEInt.pack(i))

    def writeBELInt(self, i):
        """Effectively writes very long ints in direct byte order"""
        bytes = []
        while i:
            bytes.append(six.int2byte(i % 256))
            i //= 256
        # network cap
        self.writeBEInt(len(bytes))
        self.write(b"".join(reversed(bytes)))

    def writeBELIntNetCap(self, i):
        bytes = []
        while i:
            bytes.append(six.int2byte(i % 256))
            i //= 256
        # network cap
        if bytes and (ord(bytes[-1]) & 128):
            bytes.append(six.int2byte(0))
        self.writeBEInt(len(bytes))
        self.write(b"".join(reversed(bytes)))

    def readLInt(self):
        """
        Effective writes very long ints (more than 64 bits)
        No size limitation
        """
        return self.readFInt(self.readSInt())

    def writeLInt(self, i):
        """
        Effective reads very long ints (more than 64 bits)
        No size limitation
        """
        bytes = []
        while i:
            bytes.append(six.int2byte(i % 256))
            i //= 256
        self.writeSInt(len(bytes))
        self.write(b"".join(bytes))

    def readFInt(self, l=4):
        """
        Reads fixed length ints
        Up to 2 ^ (8 * l)
        When l == 4 equal to readNInt on 64bit platform
        """
        s = self.readBuf(l)
        result = 0
        l -= 1
        while l >= 0:
            result *= 256
            result += ord(s[l:l+1])
            l -= 1
        return result

    def writeFInt(self, i, l=4):
        """
        Writes fixed length ints
        Up to 2 ^ (8 * l)
        When l == 4 equal to writeNInt on 64bit platform
        """
        while l > 0:
            self.write(six.int2byte(i % 256))
            i //= 256
            l -= 1

    def writeSInt(self, i):
        """
        Effectively writes small ints
        1 byte when < 128, 2 bytes when less then 128 ^ 2 and so on
        No size limitation
        """
        while True:
            byte = i & 127
            i >>= 7
            if i:
                byte |= 128
            self.write(six.int2byte(byte))
            if not i:
                break

    def readSInt(self):
        """
        Effectively reads small ints
        1 byte when < 128, 2 bytes when less then 128 ^ 2 and so on
        No size limitation
        """
        result = 0
        factor = 1
        while True:
            byte = ord(self.readBuf(1)[:1])
            result += factor * (byte & 127)
            if not byte & 128:
                break
            factor <<= 7
        return result

    def readStr(self):
        """
        Reads string
        Up to 2^32 characters
        """
        l = self.readNInt()
        result = self.readBuf(l) if l else b""
        return result if isinstance(result, six.binary_type) else six.b(result)

    def writeStr(self, s):
        """
        Writes string
        Up to 2^32 characters
        """
        self.writeNInt(len(s))
        if s:
            self.write(s)

    def writeObj(self, obj):
        """
        Write object to stream

        :param obj: any picklable object
        """
        self.writeStr(dumps(obj))

    def readObj(self):
        """
        Reads object previously stored to stream

        :return: read object
        """
        return loads(self.readStr())

    def readBEStr(self):
        """
        Reads string with big endian len
        Up to 2^32 characters
        """
        l = self.readBEInt()
        result = self.readBuf(l) if l else b""
        return result if isinstance(result, six.binary_type) else six.b(result)

    def writeBEStr(self, s):
        """
        Writes string with big endian len
        Up to 2^32 characters
        """
        self.writeBEInt(len(s))
        if s:
            if isinstance(s, six.binary_type):
                self.write(s)
            else:
                self.write(s.encode('utf-8'))

    def readSStr(self):
        """
        Effectively reads small strings
        No size limitations
        """
        l = self.readSInt()
        result = self.readBuf(l) if l else b""
        return result if isinstance(result, six.binary_type) else six.b(result)

    def writeSStr(self, s):
        """
        Effectively writes small strings
        No size limitations
        """
        self.writeSInt(len(s))
        if s:
            self.write(s)
