"""
Processor affinity control implementation for python
"""

__all__ = ['setProcessAffinity', 'getProcessAffinity', 'getCpuList']

from .syslibs import libc

from ctypes import POINTER, Structure, pointer, sizeof, get_errno
from ctypes import c_int, c_long, c_ulong, c_int64, c_size_t

import os

import six


LIBC = libc(useErrNo=True)
_uname = os.uname()[0].lower()

if _uname == 'freebsd':
    class CpuSetMask(Structure):
        MaxSize = 128
        LongLengthBits = sizeof(c_long) * 8
        LengthWords = (MaxSize + LongLengthBits - 1) / LongLengthBits

        _fields_ = [
            ('__bits', c_long * LengthWords)
        ]

        def clear(self):
            for i in six.moves.xrange(CpuSetMask.LengthWords):
                getattr(self, '__bits')[i] = 0

        @staticmethod
        def __word(cpuNum):
            return 0 if CpuSetMask.LengthWords == 1 else cpuNum / CpuSetMask.LongLengthBits

        @staticmethod
        def __pos(cpuNum):
            return 1 << (cpuNum if CpuSetMask.LengthWords == 1 else cpuNum % CpuSetMask.LongLengthBits)

        def setAll(self):
            for i in six.moves.xrange(CpuSetMask.LengthWords):
                getattr(self, '__bits')[i] = -1

        def set(self, cpuNum):
            getattr(self, '__bits')[self.__word(cpuNum)] |= self.__pos(cpuNum)

        def unset(self, cpuNum):
            getattr(self, '__bits')[self.__word(cpuNum)] &= ~self.__pos(cpuNum)

        def isset(self, cpuNum):
            return (getattr(self, '__bits')[self.__word(cpuNum)] & self.__pos(cpuNum)) != 0

        def __init__(self, cpus=None):
            Structure.__init__(self)
            if cpus is None:
                self.setAll()
            else:
                self.clear()
                for i in cpus:
                    self.set(i)

        def __repr__(self):
            return '<CpuSetMask ({0})>'.format(', '.join([str(i) for i in six.moves.xrange(CpuSetMask.MaxSize) if self.isset(i)]))

        def __iter__(self):
            return iter([i for i in six.moves.xrange(CpuSetMask.MaxSize) if self.isset(i)])

    class CpuLevel(object):
        Root = 1  # All system CPUs
        CpuSet = 2  # Available CPUs for which
        Which = 3  # Actual mask/id for which

    class CpuWhich(object):
        Tid = 1  # Thread ID
        Pid = 2  # Process ID
        CpuSet = 3  # Set ID
        Irq = 4  # IRQ No.
        Jail = 5  # Jail ID

    # int cpuset_setaffinity(cpulevel_t level,
    #                        cpuwhich_t which,
    #                        id_t id,
    #                        size_t setsize,
    #                        const cpuset_t *mask)
    CPUSET_SETAFFINITY = LIBC.cpuset_setaffinity
    CPUSET_SETAFFINITY.restype = c_int
    CPUSET_SETAFFINITY.argtypes = [c_int, c_int, c_int64, c_size_t, POINTER(CpuSetMask)]

    # int cpuset_getaffinity(cpulevel_t level,
    #                        cpuwhich_t which,
    #                        id_t id,
    #                        size_t setsize,
    #                        cpuset_t *mask)
    CPUSET_GETAFFINITY = LIBC.cpuset_getaffinity
    CPUSET_GETAFFINITY.restype = c_int
    CPUSET_GETAFFINITY.argtypes = [c_int, c_int, c_int64, c_size_t, POINTER(CpuSetMask)]

    # int cpuset(cpusetid_t *setid)
    CPUSET = LIBC.cpuset
    CPUSET.restype = c_int
    CPUSET.argtypes = [POINTER(c_int)]

    # int cpuset_getid(cpulevel_t level,
    #                  cpuwhich_t which,
    #                  id_t id,
    #                  cpusetid_t *setid)
    CPUSET_GETID = LIBC.cpuset_getid
    CPUSET_GETID.restype = c_int
    CPUSET_GETID.argtypes = [c_int, c_int, c_int64, POINTER(c_int)]

    # int cpuset_setid(cpuwhich_t which,
    #                  id_t id,
    #                  cpusetid_t setid)
    CPUSET_SETID = LIBC.cpuset_setid
    CPUSET_SETID.restype = c_int
    CPUSET_SETID.argtypes = [c_int, c_int64, c_int]

    def _getAffinity(level, which, objId):
        mask = CpuSetMask([])
        retval = CPUSET_GETAFFINITY(level, which, objId, sizeof(mask), pointer(mask))
        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))
        return list(mask)

    def _setAffinity(level, which, objId, cpus):
        if cpus is None:
            cpus = getCpuList()
        mask = CpuSetMask(cpus)
        retval = CPUSET_SETAFFINITY(level, which, objId, sizeof(mask), pointer(mask))
        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))

    def getProcessAffinity(pid=None):
        if pid is None:
            pid = -1

        return _getAffinity(CpuLevel.Which, CpuWhich.Pid, pid)

    def setProcessAffinity(pid=None, cpus=None):
        if pid is None:
            pid = -1

        _setAffinity(CpuLevel.Which, CpuWhich.Pid, pid, cpus)

    def getCpuSetAffinity(cpuSet=None):
        if cpuSet is None:
            cpuSet = -1

        return _getAffinity(CpuLevel.CpuSet, CpuWhich.CpuSet, cpuSet)

    def setCpuSetAffinity(cpuSet=None, cpus=None):
        if cpuSet is None:
            cpuSet = -1

        _setAffinity(CpuLevel.CpuSet, CpuWhich.CpuSet, cpuSet, cpus)

    def createCpuSet():
        cpuset = c_int()
        retval = CPUSET(pointer(cpuset))

        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))

        return cpuset.value

    def setProcessCpuSet(cpuSet, pid=None):
        if pid is None:
            pid = -1

        retval = CPUSET_SETID(CpuWhich.Pid, pid, cpuSet)
        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))

    def getProcessCpuSet(pid=None, rootId=False):
        if pid is None:
            pid = -1

        cpuset = c_int()

        retval = CPUSET_GETID(CpuLevel.Root if rootId else CpuLevel.CpuSet, CpuWhich.Pid, pid, pointer(cpuset))
        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))
        return cpuset.value

    def getCpuList():
        return range(os.sysconf('SC_NPROCESSORS_CONF'))
elif _uname == 'linux':
    class CpuSetMask(Structure):
        MaxSize = 1024
        LongLengthBits = sizeof(c_ulong) * 8
        LengthWords = (MaxSize + LongLengthBits - 1) / LongLengthBits

        _fields_ = [
            ('__bits', c_ulong * LengthWords)
        ]

        def clear(self):
            for i in six.moves.xrange(CpuSetMask.LengthWords):
                getattr(self, '__bits')[i] = 0

        @staticmethod
        def __word(cpuNum):
            return 0 if CpuSetMask.LengthWords == 1 else cpuNum / CpuSetMask.LongLengthBits

        @staticmethod
        def __pos(cpuNum):
            return 1 << (cpuNum if CpuSetMask.LengthWords == 1 else cpuNum % CpuSetMask.LongLengthBits)

        def setAll(self):
            for i in six.moves.xrange(CpuSetMask.LengthWords):
                getattr(self, '__bits')[i] = -1

        def set(self, cpuNum):
            getattr(self, '__bits')[self.__word(cpuNum)] |= self.__pos(cpuNum)

        def unset(self, cpuNum):
            getattr(self, '__bits')[self.__word(cpuNum)] &= ~self.__pos(cpuNum)

        def isset(self, cpuNum):
            return (getattr(self, '__bits')[self.__word(cpuNum)] & self.__pos(cpuNum)) != 0

        def __init__(self, cpus=None):
            Structure.__init__(self)
            if cpus is None:
                self.setAll()
            else:
                self.clear()
                for i in cpus:
                    self.set(i)

        def __repr__(self):
            return '<CpuSetMask ({0})>'.format(', '.join([str(i) for i in six.moves.xrange(CpuSetMask.MaxSize) if self.isset(i)]))

        def __iter__(self):
            return iter([i for i in six.moves.xrange(CpuSetMask.MaxSize) if self.isset(i)])

    # int sched_setaffinity(pid_t pid,
    #                       size_t cpusetsize,
    #                       cpu_set_t *mask)
    SCHED_SETAFFINITY = LIBC.sched_setaffinity
    SCHED_SETAFFINITY.restype = c_int
    SCHED_SETAFFINITY.argtypes = [c_int, c_size_t, POINTER(CpuSetMask)]

    # int sched_getaffinity(pid_t pid,
    #                       size_t cpusetsize,
    #                       cpu_set_t *mask)
    SCHED_GETAFFINITY = LIBC.sched_getaffinity
    SCHED_GETAFFINITY.restype = c_int
    SCHED_GETAFFINITY.argtypes = [c_int, c_size_t, POINTER(CpuSetMask)]

    def getProcessAffinity(pid=None):
        if pid is None:
            pid = 0

        mask = CpuSetMask([])
        retval = SCHED_GETAFFINITY(pid, sizeof(mask), pointer(mask))
        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))
        return list(mask)

    def setProcessAffinity(pid=None, cpus=None):
        if pid is None:
            pid = 0
        if cpus is None:
            cpus = getCpuList()

        mask = CpuSetMask(cpus)
        retval = SCHED_SETAFFINITY(pid, sizeof(mask), pointer(mask))
        if retval != 0:
            err = get_errno()
            raise OSError(err, os.strerror(err))

    def getCpuList():
        return range(os.sysconf('SC_NPROCESSORS_CONF'))

    def createCpuSet(*args, **kwargs):
        raise NotImplementedError

    def setProcessCpuSet(*args, **kwargs):
        raise NotImplementedError

    def getProcessCpuSet(*args, **kwargs):
        raise NotImplementedError

else:
    def getProcessAffinity(*args, **kwargs):
        raise NotImplementedError

    def setProcessAffinity(*args, **kwargs):
        raise NotImplementedError

    def createCpuSet(*args, **kwargs):
        raise NotImplementedError

    def setProcessCpuSet(*args, **kwargs):
        raise NotImplementedError

    def getProcessCpuSet(*args, **kwargs):
        raise NotImplementedError

    def getCpuList():
        raise NotImplementedError
