from __future__ import print_function

import ctypes
import os
import signal
import thread  # noqa
import threading
import errno
import Queue

from ..errors import sFail
from ..sys import libc
from ..functional import singleton

import six

if os.name == 'posix':
    def _handleSigUsr2(sigNum, frame):
        pass

    if os.uname()[0].lower() == "freebsd":
        class _AIOCB_private(ctypes.Structure):
            _fields_ = [
                ('status', ctypes.c_long),
                ('error', ctypes.c_long),
                ('kernelinfo', ctypes.c_void_p),
            ]

        class _sigevent(ctypes.Structure):
            _fields_ = [
                ('sigev_notify', ctypes.c_int),
                ('sigev_signo', ctypes.c_int),
                ('sigev_value', ctypes.c_void_p),
                ('_sigev_un', ctypes.c_long * 8)
            ]

        class _AIOCB(ctypes.Structure):
            _fields_ = [
                ('aio_fildes', ctypes.c_int),
                ('aio_offset', ctypes.c_size_t),
                ('aio_buf', ctypes.c_char_p),
                ('aio_nbytes', ctypes.c_size_t),
                ('spare', ctypes.c_void_p),
                ('aio_lio_opcode', ctypes.c_int),
                ('aio_reqprio', ctypes.c_int),
                ('_aio_private', _AIOCB_private),
                ('_aio_sigevent', _sigevent)
            ]

            def __hash__(self):
                return ctypes.addressof(self)

    class _AIOReader_(object):
        _aioSuspend = libc(True).aio_suspend
        _aioRead = libc(True).aio_read
        _aioError = libc(True).aio_error
        _aioReturn = libc(True).aio_return
        _pthreadSelf = libc(True).pthread_self
        _pthreadKill = libc(True).pthread_kill

        def __init__(self):
            self._WaitQueue = Queue.Queue()
            self._readyQueue = Queue.Queue()

            self._waitThread = threading.Thread(target=self._waitMain)

            self._waitThread.daemon = True
            self._waitThread.start()

            self._readyThread = threading.Thread(target=self._readyMain)

            self._readyThread.daemon = True
            self._readyThread.start()

            self._callBacks = {}

        def _waitMain(self):
            aiocbOwner = []

            aiocbsPointersReserved = [4]
            aiocbsPointers = [(ctypes.c_void_p * aiocbsPointersReserved[0])()]

            def processAIOCB(aiocb):
                aiocbOwner.append(aiocb)
                if len(aiocbOwner) > aiocbsPointersReserved[0]:
                    while len(aiocbOwner) > aiocbsPointersReserved[0]:
                        aiocbsPointersReserved[0] *= 2

                    # Doesnt work:
                    # ctypes.resize(aiocbsPointers, ctypes.sizeof(ctypes.c_void_p) * aiocbsPointersReserved[0])

                    aiocbsPointers[0] = (ctypes.c_void_p * aiocbsPointersReserved[0])()
                    for i in six.moves.xrange(len(aiocbOwner)):
                        aiocbsPointers[0][i] = ctypes.addressof(aiocbOwner[i])

                aiocbsPointers[0][len(aiocbOwner) - 1] = ctypes.addressof(aiocb)

            while True:
                try:
                    while True:
                        processAIOCB(self._WaitQueue.get_nowait())
                except Queue.Empty:
                    if aiocbOwner:
                        if self._aioSuspend(ctypes.addressof(aiocbsPointers[0]), len(aiocbOwner), 0) == 0:
                            finished = {}
                            aiocbOwnerLen = len(aiocbOwner)
                            i = 0
                            while i < aiocbOwnerLen:
                                errno_ = self._aioError(aiocbsPointers[0][i])
                                if errno_ != errno.EINPROGRESS:
                                    finished[aiocbsPointers[0][i]] = errno_, aiocbOwner[i]
                                    aiocbOwnerLen -= 1
                                    aiocbsPointers[0][i] = aiocbsPointers[0][aiocbOwnerLen]
                                    aiocbOwner[i] = aiocbOwner[aiocbOwnerLen]
                                else:
                                    i += 1
                            del aiocbOwner[aiocbOwnerLen:]

                            for address, (errno_, aiocb) in finished.iteritems():
                                cb, buffer, fdForClose = self._callBacks.pop(address)
                                if errno_ == 0:
                                    self._readyQueue.put((cb, buffer[:self._aioReturn(address)], None))
                                else:
                                    self._readyQueue.put((cb, None, OSError(errno_, os.strerror(errno_))))
                                if fdForClose is not None:
                                    try:
                                        os.close(fdForClose)
                                    except EnvironmentError:
                                        pass

                        else:
                            _errno_ = ctypes.get_errno()
                            if _errno_ == errno.EINTR:
                                continue
                            else:
                                print('Error in aio.waitMain: {0}'.format(_errno_))
                    else:
                        processAIOCB(self._WaitQueue.get())

        def _readyMain(self):
            while True:
                cb, buffer, errno_ = self._readyQueue.get()
                try:
                    cb(buffer, errno_)
                except Exception as err:
                    print('Error in aio.readyMain: {0}'.format(err))

        def aioRead(self, fd, offset, bytes, cb):
            fdForClose = None
            if hasattr(fd, 'fileno'):
                fd = fd.fileno()
            elif isinstance(fd, int):
                pass
            elif isinstance(fd, six.string_types):
                fd = os.open(fd, os.O_RDONLY | os.O_DIRECT)
                fdForClose = fd
            else:
                sFail('Unexpected type of fd: {0}'.format(type(fd)))

            aiocb = _AIOCB()
            aiocb.aio_fildes = fd
            aiocb.aio_offset = offset
            buffer = ctypes.create_string_buffer(bytes)
            aiocb.aio_buf = ctypes.addressof(buffer)
            aiocb.aio_nbytes = bytes
            self._callBacks[ctypes.addressof(aiocb)] = cb, buffer, fdForClose

            if self._aioRead(ctypes.addressof(aiocb)) != 0:
                errno_ = ctypes.get_errno()
                raise OSError(errno_, os.strerror(errno_))

            self._WaitQueue.put(aiocb)

            self._pthreadKill(self._waitThread.ident, signal.SIGIO)

    @singleton
    def _AIOReader():
        signal.signal(signal.SIGIO, _handleSigUsr2)

        return _AIOReader_()

    aioRead = _AIOReader().aioRead
