import os
import six
import tempfile
import errno

from ..misc import cachingFunction

from .user import getUserName


@cachingFunction
def getSkynet():
    result = os.path.realpath(__file__)
    for _i in range(4):
        result = os.path.dirname(result)
    return result


class TempDir:
    def __str__(self):
        return self.dir()

    def __init__(self, *arg, **kwarg):
        self.tempDir = None
        self.arg = arg
        self.kwarg = kwarg

    def dir(self):
        if self.tempDir:
            return self.tempDir
        else:
            raise RuntimeError("Not opened", errno.ENOTDIR)

    def open(self):
        self.close()
        self.tempDir = tempfile.mkdtemp(*self.arg, **self.kwarg)

    def close(self):
        if self.tempDir:
            try:
                removeDirs(self.tempDir)
            except EnvironmentError:
                pass
        self.tempDir = None

    def __enter__(self):
        self.open()
        return self.dir()

    def __exit__(self, _excType, _excValue, _traceback):
        self.close()


def setSkynetTemp():
    tmp = os.path.join("/var/tmp", getUserName())
    try:
        ensureDirs(tmp)
        tempfile.tempdir = tmp
    except OSError:
        pass


def ensureDirs(dirNames, create=True):
    result = True
    if isinstance(dirNames, six.string_types):
        dirNames = (dirNames,)
    for dirName in dirNames:
        if not os.path.exists(dirName):
            result = False
            if create:
                try:
                    os.makedirs(dirName)  # Race condition here
                except OSError as err:
                    if err.args[0] not in (errno.EEXIST,):
                        raise

    return result


def removeDirs(dirNames):
    if isinstance(dirNames, six.string_types):
        dirNames = (dirNames,)
    for dirName in dirNames:
        for root, dirs, files in os.walk(dirName, topdown=False):
            for file in files:
                os.remove(os.path.join(root, file))
            for dir in dirs:
                path = os.path.join(root, dir)
                if os.path.islink(path):
                    os.remove(path)
                else:
                    os.rmdir(os.path.join(root, dir))
        os.rmdir(dirName)

removeDirsWithContent = removeDirs


def rotateFile(fileName, count=5):
    if not fileName:
        return
    try:
        os.remove(fileName + ".%i" % count)
    except OSError:
        pass

    for i in reversed(range(1, count)):
        try:
            os.rename(fileName + ".%i" % i, fileName + ".%i" % (i + 1))
        except OSError:
            pass
    try:
        os.rename(fileName, fileName + ".1")
    except OSError:
        pass
