"""
Darwin version
"""

import os
import struct
import socket
import errno

__all__ = [
    'getpeerid'
]

_LOCAL_PEERCRED = 0x001
_NGROUPS = 16

# NOTE: struct xucred is similar to freebsd one except last "unused" field missing

# #define NGROUPS NGROUPS_MAX
# #define NGROUPS_MAX 16
# struct xucred {
#         u_int   cr_version;             /* structure layout version */
#         uid_t   cr_uid;                 /* effective user id */
#         short   cr_ngroups;             /* number of advisory groups */
#         gid_t   cr_groups[NGROUPS];     /* advisory group list */
# };
_structxucred = struct.Struct('2ih{0}i'.format(_NGROUPS))


def getpeerid(sock, getpid=False):
    """
    Returns peer uid and peer gid of unix domain socket.
    :param sock: Unix domain socket of interest. Or its fd.
    :type sock: socket.SocketType or int
    :return: (uid, list of gid)
    :rtype: (int, list of int)
    :raise: OSError upon syscall errors
    """
    if isinstance(sock, int):
        sock = socket.fromfd(sock, socket.AF_UNIX, socket.SOCK_STREAM)

    if getpid:
        raise NotImplementedError('getpid is not supported on Darwin')

    res = sock.getsockopt(socket.SOL_IP, _LOCAL_PEERCRED, _structxucred.size)
    if len(res) < _structxucred.size:
        raise OSError(errno.EINVAL, os.strerror(errno.EINVAL))

    res = _structxucred.unpack(res)

    uid = res[1]
    groupsLen = res[2]
    guids = res[3:3 + groupsLen]
    return uid, guids
