"""
Linux version via getsockopt.
"""

import os
import sys
import struct
import socket
import errno

__all__ = [
    'getpeerid'
]

# hopefully they don't break ABI
# SO_PEERCRED is defined only in python3.3
_SO_PEERCRED = 0x200 if sys.platform == 'cygwin' else 17

_struct3i = struct.Struct('3i')


def getpeerid(sock, getpid=False):
    """
    Returns peer uid and peer gid of unix domain socket.
    :param sock: Unix domain socket of interest. Or its fd.
    :type sock: socket.SocketType or int
    :return: (uid, list of gid)
    :rtype: (int, list of int)
    :raise: OSError upon syscall errors
    """
    if isinstance(sock, int):
        sock = socket.fromfd(sock, socket.AF_UNIX, socket.SOCK_STREAM)

    try:
        res = sock.getsockopt(socket.SOL_SOCKET, _SO_PEERCRED, _struct3i.size)
    except socket.error as e:
        if sys.platform == 'cygwin':
            # cygwin throws socket.error exception in case of invalid socket type
            # so we try to emulate linux behavior and throw OSError instead
            raise OSError(errno.EINVAL, os.strerror(errno.EINVAL))
        else:
            raise e

    pid, uid, gid = _struct3i.unpack(res)
    # seems that in case of bad socket (e.g. AF_INET)
    # we are getting 0, -1, -1 as pid, uid, gid accordingly
    # so let's manually raise EINVAL
    if -1 in (uid, gid):
        raise OSError(errno.EINVAL, os.strerror(errno.EINVAL))

    if getpid:
        return uid, (gid, ), pid

    return uid, (gid,)
