from __future__ import print_function
import ctypes
import os


if os.name == "posix" and os.uname()[0].lower() != 'darwin' and not os.uname()[0].lower().startswith('cygwin'):
    if os.uname()[0].lower() == "freebsd":
        CLOCK_MONOTONIC = 4
        CLOCK_THREAD_CPUTIME_ID = 14
    else:
        CLOCK_MONOTONIC = 1
        CLOCK_THREAD_CPUTIME_ID = 3

    class timespec(ctypes.Structure):
        _fields_ = [
            ('tv_sec', ctypes.c_long),
            ('tv_nsec', ctypes.c_long)
        ]

    from .syslibs import librt
    clock_gettime = librt(True).clock_gettime
    clock_gettime.argtypes = [ctypes.c_int, ctypes.POINTER(timespec)]

    def clockGetTime(timerType):
        t = timespec()
        if clock_gettime(timerType, ctypes.pointer(t)):
            errno_ = ctypes.get_errno()
            raise OSError(errno_, os.strerror(errno_))
        return t.tv_sec + t.tv_nsec * 1e-9

    def monoTime():
        return clockGetTime(CLOCK_MONOTONIC)

    def threadCPUTime():
        return clockGetTime(CLOCK_THREAD_CPUTIME_ID)

else:
    import time
    monoTime = time.time
    threadCPUTime = time.time

if __name__ == "__main__":
    import threading
    import time
    import six

    def test1():
        print(threading.currentThread().name, threadCPUTime())
        f = 1
        for i in six.moves.xrange(10000000):
            f *= i
        print(threading.currentThread().name, threadCPUTime())

    def test2():
        print(threading.currentThread().name, threadCPUTime())
        time.sleep(1)
        print(threading.currentThread().name, threadCPUTime())

    print("Main thread", threadCPUTime())
    t1 = threading.Thread(target=test1, name="Fact thread")
    t2 = threading.Thread(target=test2, name="Sleep thread")
    t2.start()
    t1.start()
    t2.join()
    t1.join()
    print("Main thread", threadCPUTime())
