import ctypes
import ctypes.util
import resource

UTIL = ctypes.CDLL(ctypes.util.find_library('util'), use_errno=True)
LIBC = ctypes.CDLL(ctypes.util.find_library('c'), use_errno=True)


class LoginCapT(ctypes.Structure):
    pass


class Passwd(ctypes.Structure):
    pass


GETPWNAM = LIBC.getpwnam
GETPWNAM.restype = ctypes.POINTER(Passwd)
GETPWNAM.argtypes = [ctypes.c_char_p]

GETPWCLASS = UTIL.login_getpwclass
GETPWCLASS.restype = ctypes.POINTER(LoginCapT)
GETPWCLASS.argtypes = [ctypes.POINTER(Passwd)]

GETUSERCLASS = UTIL.login_getuserclass
GETUSERCLASS.restype = ctypes.POINTER(LoginCapT)
GETUSERCLASS.argtypes = [ctypes.POINTER(Passwd)]

LOGINCLOSE = UTIL.login_close
LOGINCLOSE.restype = None
LOGINCLOSE.argtypes = [ctypes.POINTER(LoginCapT)]

RLIM_INFINITY = -1

GETCAPSIZE = UTIL.login_getcapsize
GETCAPSIZE.restype = ctypes.c_int64
GETCAPSIZE.argtypes = [ctypes.POINTER(LoginCapT), ctypes.c_char_p, ctypes.c_int64, ctypes.c_int64]

GETCAPNUM = UTIL.login_getcapnum
GETCAPNUM.restype = ctypes.c_int64
GETCAPNUM.argtypes = [ctypes.POINTER(LoginCapT), ctypes.c_char_p, ctypes.c_int64, ctypes.c_int64]


def _getLimit(pwdClass, limitName, method, pos):
    value = method(pwdClass, limitName, -2, -3)
    if value == -3:  # error
        raise AttributeError("Unknown limit: {0}".format(limitName))
    if value == -2:  # default value
        return None, None  # limit not set

    if value == RLIM_INFINITY:
        value = resource.RLIM_INFINITY

    if pos == 'cur':
        return value, None
    elif pos == 'max':
        return None, value
    else:
        return value, value


def getUmask(userclass, pwclass):
    umask = GETCAPNUM(pwclass, 'umask', -2, -3)
    if umask in (-2, -3):
        umask = 0

    _umaskUser = GETCAPNUM(userclass, 'umask', -2, -3)
    if _umaskUser not in (-2, -3):
        umask = _umaskUser

    return umask


def getUserResourceLimits(user, umask=False, defaults=None):
    userpwd = GETPWNAM(user)

    userclass = GETUSERCLASS(userpwd)
    pwclass = GETPWCLASS(userpwd)

    result = {}
    try:
        for limitId, limitName, method in (
            ('CORE', "coredumpsize", GETCAPSIZE),
            ('CPU', 'cputime', GETCAPNUM),
            ('FSIZE', 'filesize', GETCAPSIZE),
            ('DATA', 'datasize', GETCAPSIZE),
            ('STACK', 'stacksize', GETCAPSIZE),
            ('RSS', 'memoryuse', GETCAPSIZE),
            ('NPROC', 'maxproc', GETCAPNUM),
            ('NOFILE', 'openfiles', GETCAPNUM),
            ('MEMLOCK', 'memorylocked', GETCAPSIZE),
            ('VMEM', 'vmemoryuse', GETCAPSIZE),
        ):
            try:
                limitKey = getattr(resource, 'RLIMIT_{0}'.format(limitId))
            except AttributeError:
                continue

            valueCur, valueMax = None, None

            try:
                for pwdClass in (pwclass, userclass):
                    for limit, pos in (
                        (limitName, 'both'),
                        ('{0}-max'.format(limitName), 'max'),
                        ('{0}-cur'.format(limitName), 'cur'),
                    ):
                        _valueCur, _valueMax = _getLimit(pwdClass, limit, method, pos)

                        if (
                            _valueMax is not None
                            and (valueMax is None or _valueMax <= valueMax)
                        ):
                            valueMax = _valueMax

                        if (
                            _valueCur is not None
                            and (valueMax is None or _valueCur <= valueMax)
                            and (valueCur is None or _valueCur <= valueCur)
                        ):
                            valueCur = _valueCur
            except AttributeError:
                continue

            if valueCur is None:
                continue

            result[limitKey] = (valueCur, valueMax)

        if umask:
            result = (result, getUmask(userclass, pwclass))
    finally:
        LOGINCLOSE(userclass)
        LOGINCLOSE(pwclass)

    return result
