import os
import re
import pwd
import grp
import resource


def _parse_system_limits():
    lim_names = {
        'core': resource.RLIMIT_CORE,
        'data': resource.RLIMIT_DATA,
        'fsize': resource.RLIMIT_FSIZE,
        'memlock': resource.RLIMIT_MEMLOCK,
        'nofile': resource.RLIMIT_NOFILE,
        'rss': resource.RLIMIT_RSS,
        'stack': resource.RLIMIT_STACK,
        'cpu': resource.RLIMIT_CPU,
        'nproc': resource.RLIMIT_NPROC,
        'as': resource.RLIMIT_AS,
    }

    basedir = '/etc/security'

    files = [os.path.join(basedir, 'limits.conf')]

    confd = os.path.join(basedir, 'limits.d')

    if os.path.exists(confd) and os.path.isdir(confd):
        for fn in filter(lambda fn: fn.endswith('.conf'), sorted(os.listdir(confd))):
            files.append(os.path.join(confd, fn))

    all_raw = []
    for fn in files:
        for line in open(fn, 'rb'):
            line = line.strip()

            if '#' in line:
                line = line.split('#', 1)[0]

            if not line:
                continue

            all_raw.append(line)

    user_limits, group_limits = {}, {}

    for line in all_raw:
        line = line.strip()
        items = line.split()
        if len(items) != 4:
            continue

        who, typ, lim, value = items

        if lim not in lim_names:
            continue

        try:
            value = int(value)
        except:
            continue

        lim = lim_names[lim]

        if who.startswith('@'):
            who = who.lstrip('@')
            limits = group_limits
        else:
            limits = user_limits

        limits.setdefault(who, {}).setdefault(lim, [None, None])[0 if typ == 'soft' else 1] = value

    return user_limits, group_limits


UMASK_RE = re.compile('^\s*umask\s+(\d+)', re.IGNORECASE)
USERGROUPS_RE = re.compile('^\s*usergroups_enab\s+(yes|no)', re.IGNORECASE)
UMASK_GECOS_RE = re.compile('(^|,\s*)umask=(\d+)', re.IGNORECASE)


def get_umask(user):
    umask = 0o022
    usergroups = False
    for filename in ('/etc/default/login', '/etc/login.defs'):
        try:
            with open(filename) as f:
                for line in f:
                    result = UMASK_RE.match(line)
                    if result:
                        umask = int(result.group(1), 8) & 0o777
                        continue
                    result = USERGROUPS_RE.match(line)
                    if result:
                        usergroups = result.group(1).lower() == 'yes'
        except (EnvironmentError, ValueError):
            pass

    try:
        gecos = pwd.getpwnam(user).pw_gecos
        result = UMASK_GECOS_RE.search(gecos)
        if result:
            umask = int(result.group(2), 8) & 0o777
    except Exception:
        pass

    if usergroups:
        umask = (umask & 0o707) | ((umask & 0o700) >> 3)

    return umask


def get_user_resource_limits(user, umask=False, defaults=None):
    kernel_defaults = {
        resource.RLIMIT_CPU: [resource.RLIM_INFINITY] * 2,
        resource.RLIMIT_FSIZE: [resource.RLIM_INFINITY] * 2,
        resource.RLIMIT_DATA: [resource.RLIM_INFINITY] * 2,
        resource.RLIMIT_STACK: [8388608, resource.RLIM_INFINITY],
        resource.RLIMIT_CORE: [resource.RLIM_INFINITY] * 2,
        resource.RLIMIT_RSS: [resource.RLIM_INFINITY] * 2,
        resource.RLIMIT_NPROC: [47018, 47018],
        resource.RLIMIT_NOFILE: [1024, 4096],
        resource.RLIMIT_MEMLOCK: [65536, 65536],
        resource.RLIMIT_AS: [resource.RLIM_INFINITY] * 2,
    }

    # Importing at runtime to avoid cyclic imports
    from .posix import getUserGIDs, getUserUID

    uid = getUserUID(user)
    gids = getUserGIDs(uid)

    groups = [grp.getgrgid(_).gr_name for _ in gids]

    system_user_limits, system_group_limits = _parse_system_limits()

    user_limits = kernel_defaults if defaults is None else defaults

    for group in groups:
        for lim, (soft, hard) in system_group_limits.get(group, {}).items():
            this_lim = user_limits.setdefault(lim, [None, None])
            if soft is not None:
                this_lim[0] = soft
            if hard is not None:
                this_lim[1] = hard

    for lim, (soft, hard) in system_user_limits.get(user, {}).items():
        this_lim = user_limits.setdefault(lim, [None, None])
        if soft is not None:
            this_lim[0] = soft
        if hard is not None:
            this_lim[1] = hard

    for key in user_limits:
        user_limits[key] = tuple(user_limits[key])

    if umask:
        return user_limits, get_umask(user)

    return user_limits


# Map old CamelCase function names
getUserResourceLimits = get_user_resource_limits
