import os
import six
import pwd
import grp
import errno
try:
    import resource
except ImportError:
    resource = None


__all__ = [
    'getUserName',
    'getUserHome',
    'getUserUID',
    'getUserGIDs',
    'getResourceLimits',
    'setResourceLimits',
    'getUserResourceLimits',
    'UserPrivileges',
    'userPrivileges',
]

sysname = os.uname()[0].lower()

if resource:
    if sysname == 'freebsd':
        from ._freebsd import getUserResourceLimits as _getUserResourceLimits
    elif sysname.startswith('linux'):
        from ._linux import getUserResourceLimits as _getUserResourceLimits
    elif sysname == 'darwin':
        from ._darwin import getUserResourceLimits as _getUserResourceLimits
    else:
        _getUserResourceLimits = None
else:
    _getUserResourceLimits = None


def getUserName():
    try:
        return pwd.getpwuid(os.geteuid()).pw_name
    except KeyError:
        return os.getenv("USER")


def getUserHome(username=None):
    if username is not None:
        try:
            return pwd.getpwnam(username).pw_dir
        except KeyError:
            return "/var/tmp/%s" % username
    try:
        return pwd.getpwuid(os.geteuid()).pw_dir
    except KeyError:
        result = os.getenv("HOME")

    if result is not None and os.path.isdir(result) and os.path.normpath(result) != "/":
        return result
    else:
        return "/var/tmp/%s" % getUserName()


def getUserUID(user=None):
    if user is None:
        return os.geteuid()
    try:
        return pwd.getpwnam(user).pw_uid
    except (TypeError, KeyError):
        return None


def getUserGIDs(user=None):
    # FIXME getgrall() can return incomplete set of groups with LDAP
    # so the function should be rewritten with getgrouplist(user, gid, &buffer, &max_groups) extension
    if user is None:
        pwRec = pwd.getpwuid(os.geteuid())
    elif isinstance(user, six.integer_types):
        pwRec = pwd.getpwuid(user)
    elif isinstance(user, six.string_types):
        pwRec = pwd.getpwnam(user)
    elif isinstance(user, pwd.struct_passwd):
        pwRec = user
    else:
        raise AssertionError('Unknown user type')

    gids = set()

    for group in grp.getgrall():
        if pwRec.pw_name in group.gr_mem:
            gids.add(group.gr_gid)

    gids.discard(pwRec.pw_gid)

    result = [pwRec.pw_gid]
    result.extend(gids)

    return result


def getResourceLimits():
    result = {}
    for key in dir(resource):
        if key.startswith('RLIMIT_'):
            lim = getattr(resource, key)
            result[lim] = resource.getrlimit(lim)
    return result


def setResourceLimits(limits, minimalLimits=None, raiseFrom=None, tolerateErrors=False):
    if os.getuid() == 0:
        # ignore ALL errors if we are root, changing limits upper may be prohibited by
        # porto
        tolerateErrors = True

    if isinstance(limits, tuple):
        _limits, _ = limits
    else:
        _limits, _ = limits, None

    if minimalLimits is None:
        minimalLimits = {}
    for lim, (soft, hard) in _limits.items():
        minSoft, minHard = minimalLimits.get(lim, (0, 0))
        if hard is None:
            _, hard = resource.getrlimit(lim)
        soft = (resource.RLIM_INFINITY
                if resource.RLIM_INFINITY in (soft, minSoft)
                else max(soft, minSoft))
        hard = (resource.RLIM_INFINITY
                if resource.RLIM_INFINITY in (hard, minHard)
                else max(hard, minHard))

        if raiseFrom is not None:
            initialMax = raiseFrom.get(lim, (resource.RLIM_INFINITY, resource.RLIM_INFINITY))[1]
            if initialMax == resource.RLIM_INFINITY or initialMax > hard:
                continue

        try:
            resource.setrlimit(lim, (soft, hard))
        except ValueError:
            if not tolerateErrors:
                raise


def getUserResourceLimits(user=None, umask=False, defaults=None):
    """
    Get user resource limits.

    Caveats:
        linux:
            this is done using parsing data in /etc/security/limits.conf,
            /etc/security/limits.d/, /etc/default/login and pwd database.

        freebsd:
            this is done by login_getpwclass and login_getuserclass
            from libutil.
    """

    if user is None:
        response = getResourceLimits()
        if umask:
            response = (response, os.umask(0))
            os.umask(response[1])
        return response

    try:
        if isinstance(user, six.integer_types):
            rec = pwd.getpwuid(user)
        else:
            rec = pwd.getpwnam(user)
    except KeyError:
        raise EnvironmentError('No user "{0}" in system'.format(user))

    if _getUserResourceLimits is None:
        raise RuntimeError("Unsupported OS: {0}".format(sysname))

    return _getUserResourceLimits(rec.pw_name, umask=umask, defaults=defaults)


class UserPrivileges(object):
    """
    Switch priveleges to different user and possibly store root for
    later changes

    Example:
        # we are root

        with userPrivileges('skynet'):
            # we are skynet, but root is stored

            with userPrivileges():
                # we are root again

                with userPrivileges('skynet', store=False):
                    # we are skynet again, but cant become root anymore

                    with userPrivileges():  # raises EnvironmentError with EPERM errno.
                        # nether executed
                        pass

    Note that this functionality is *NOT* threadsafe. Thus, switching priveleges
    in one thread/greenlet, will simultaneously change effective and real user of *ALL*
    threads/greenlets in program. You have been warned.
    """

    def __init__(self,
                 user=None,
                 store=True,
                 limit=False,
                 modifyGreenlet=True,
                 minimalLimits=None,
                 onlyRaiseLimits=False):
        self.user = user
        self.store = store
        self.limit = limit
        self.modifyGreenlet = modifyGreenlet
        self.minimalLimits = minimalLimits or {}
        self.onlyRaiseLimits = onlyRaiseLimits

    def _get_ids(self):
        if sysname == 'darwin' or sysname.startswith('cygwin'):
            # no stored uid in OS X
            self.euid = os.geteuid()
            self.egid = os.getegid()
            self.ruid = os.getuid()
            self.rgid = os.getgid()
            self.suid, self.sgid = None, None
        else:
            self.ruid, self.euid, self.suid = os.getresuid()
            self.rgid, self.egid, self.sgid = os.getresgid()

    def _get_pwinfo(self):
        try:
            if isinstance(self.user, six.integer_types):
                rec = pwd.getpwuid(self.user)
            else:
                rec = pwd.getpwnam(self.user)

            uid = rec.pw_uid
            gid = rec.pw_gid
        except KeyError:
            raise EnvironmentError('No user "{0}" in system'.format(self.user))

        return rec, uid, gid

    def _raise_privileges(self):
        if sysname == 'darwin':
            os.setreuid(0, 0)
            os.setregid(0, 0)
        elif sysname.startswith('cygwin'):
            os.setreuid(0, 0)
            os.setregid(11, 11)
        else:
            os.setresuid(0, 0, 0)
            os.setresgid(0, 0, 0)

    def _drop_privileges(self, uid, gid):
        if sysname == 'darwin':
            os.setregid(0 if self.store else gid, gid)
            os.setreuid(0 if self.store else uid, uid)
        elif sysname.startswith('cygwin'):
            os.setregid(11 if self.store else gid, gid)
            os.setreuid(0 if self.store else uid, uid)
        else:
            os.setresgid(gid, gid, 0 if self.store else gid)
            os.setresuid(uid, uid, 0 if self.store else uid)

    def _restore_privileges(self):
        if sysname == 'darwin' or sysname.startswith('cygwin'):
            os.setregid(self.rgid, self.egid)
            os.setreuid(self.ruid, self.euid)
        else:
            os.setresgid(self.rgid, self.egid, self.sgid)
            os.setresuid(self.ruid, self.euid, self.suid)

    def __enter__(self):
        rec, uid, gid = (None, ) * 3

        if self.user:
            rec, uid, gid = self._get_pwinfo()

        self._get_ids()

        # Check what we have stored root
        if self.suid != 0 and self.ruid != 0:
            raise EnvironmentError(errno.EPERM, 'Dont have root privileges')

        self.nonroot = rec is not None and uid != 0

        if self.store:
            self.groups = os.getgroups()

        # Raise privileges to root.
        self._raise_privileges()

        if self.limit:
            self.oldLimits = getResourceLimits()

        # Drop all groups
        if not sysname.startswith('cygwin'):
            os.initgroups(rec.pw_name if rec else "root", rec.pw_gid if rec else 0)

            # Apply user limits if needed
            if self.limit:
                userLimits, umask = getUserResourceLimits(rec.pw_name if rec else 'root', True)
                setResourceLimits(
                    userLimits,
                    minimalLimits=self.minimalLimits,
                    raiseFrom=self.oldLimits if self.onlyRaiseLimits else None,
                    tolerateErrors=True
                )
                os.umask(umask)

        # If we want to change effective and real uids to some user -- do that now
        if self.nonroot:
            self._drop_privileges(uid, gid)

        if self.modifyGreenlet and self.store:
            import gevent
            greenlet = gevent.getcurrent()
            if hasattr(greenlet, 'registerContext'):
                greenlet.registerContext(self)

    def __exit__(self, exc_type, exc_value, tb):
        if self.modifyGreenlet and self.store:
            import gevent
            greenlet = gevent.getcurrent()
            if hasattr(greenlet, 'unregisterContext'):
                greenlet.unregisterContext(self)

        # If we didnt stored root -- we cant regreet privileges back
        if self.store:
            # If we changed suid to some user -- restore back to root first
            if self.nonroot:
                self._raise_privileges()

            if not sysname.startswith('cygwin'):
                if self.limit:
                    setResourceLimits(self.oldLimits, tolerateErrors=True)

                # Restore original user groups
                os.setgroups(self.groups[:os.sysconf('SC_NGROUPS_MAX')])

            # And restore original gids/uids
            self._restore_privileges()


def userPrivileges(user=None,
                   store=True,
                   limit=False,
                   modifyGreenlet=True,
                   minimalLimits=None,
                   onlyRaiseLimits=False):
    return UserPrivileges(user=user,
                          store=store,
                          limit=limit,
                          modifyGreenlet=modifyGreenlet,
                          minimalLimits=minimalLimits,
                          onlyRaiseLimits=onlyRaiseLimits)
