"""
Helps for extended exception handling.

Tests are in tests/kernel/util/test_errors.py
"""

import textwrap
import inspect
import pickle
import sys

# noinspection PyUnresolvedReferences
from pytest import raises

from api.copier import errors
from kernel.util.errors import raiseEx, formatException, saveTraceback, SkynetBaseError


def test_format_simple_exception():
    try:
        raise IOError('auch!')
    except Exception:
        import traceback
        assert traceback.format_exc() == formatException()


def test_traceback_chaining():
    srcLine = inspect.currentframe().f_lineno

    def _doSomeBadStuff():
        try:
            try:
                def _doSomeReallyBadStuff():
                    try:
                        raise errors.CopierError('inner error')
                    except Exception as err:
                        raiseEx(errors.CopierError('outer error'), err, 'catching inner')
                _doSomeReallyBadStuff()
            except Exception as err:
                raiseEx(errors.CopierError('third error'), err, 'catching inner+outer')
        except Exception as err:
            raiseEx(errors.CopierError('final error'), err, 'catching ALL')
    try:
        _doSomeBadStuff()
    except errors.CopierError:
        print(formatException())
        assert formatException() == textwrap.dedent(
            '''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_traceback_chaining
                _doSomeBadStuff()
              File "{file}", line {line2}, in _doSomeBadStuff
                raiseEx(errors.CopierError('final error'), err, 'catching ALL')
              --------------------------- catching ALL ---------------------------
              File "{file}", line {line3}, in _doSomeBadStuff
                raiseEx(errors.CopierError('third error'), err, 'catching inner+outer')
              ----------------------- catching inner+outer -----------------------
              File "{file}", line {line4}, in _doSomeBadStuff
                _doSomeReallyBadStuff()
              File "{file}", line {line5}, in _doSomeReallyBadStuff
                raiseEx(errors.CopierError('outer error'), err, 'catching inner')
              -------------------------- catching inner --------------------------
              File "{file}", line {line6}, in _doSomeReallyBadStuff
                raise errors.CopierError('inner error')
            CopierError: final error
            '''
        ).format(
            file=__file__.rstrip('c'),
            line1=srcLine + 16, line2=srcLine + 14, line3=srcLine + 12,
            line4=srcLine + 10, line5=srcLine + 9, line6=srcLine + 7
        )


def test_traceback_formatting_outside_except_block():
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('inner error')
    except Exception:
        pass

    print(formatException())
    assert formatException() == textwrap.dedent('''\
       Traceback (most recent call last):
         File "{file}", line {line1}, in test_traceback_formatting_outside_except_block
           raise errors.CopierError('inner error')
       CopierError: inner error
       ''').format(file=__file__.rstrip('c'), line1=srcLine + 2)


def test_traceback_formatting_outside_except_block_if_something_was_raised_after():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('test error')
    except errors.CopierError as err:
        saveTraceback(err)
        e = err

    try:
        raise Exception('something other')
    except:
        pass

    print(formatException(e))
    assert formatException(e) == textwrap.dedent('''\
        Traceback (most recent call last):
          File "{file}", line {line1}, in test_traceback_formatting_outside_except_block_if_something_was_raised_after
            raise errors.CopierError('test error')
        CopierError: test error
        ''').format(file=__file__.rstrip('c'), line1=srcLine + 2)

    print(formatException())
    assert formatException() == textwrap.dedent('''\
        Traceback (most recent call last):
          File "{file}", line {line1}, in test_traceback_formatting_outside_except_block_if_something_was_raised_after
            raise Exception('something other')
        Exception: something other
        ''').format(file=__file__.rstrip('c'), line1=srcLine + 8)


def test_chain_different_exceptions():
    try:
        try:
            raise errors.ApiError('api error')
        except Exception as err:
            raiseEx(errors.FilesystemError('confusing but okay'), err)
    except errors.FilesystemError:
        assert formatException().endswith('FilesystemError: confusing but okay\n')


def test_traceback_chaining_outside_except_block():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('error A')
    except Exception as err:
        e = err

    try:
        raiseEx(errors.CopierError('error B'), e, 'next chain')
    except errors.CopierError:
        print(formatException())
        assert formatException() == textwrap.dedent('''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_traceback_chaining_outside_except_block
                raiseEx(errors.CopierError('error B'), e, 'next chain')
              ---------------------------- next chain ----------------------------
              File "{file}", line {line2}, in test_traceback_chaining_outside_except_block
                raise errors.CopierError('error A')
            CopierError: error B
            ''').format(file=__file__.rstrip('c'), line1=srcLine + 7, line2=srcLine + 2)


def test_traceback_reraising():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('error here')
    except Exception as err:
        saveTraceback(err, 'catching remote run')
        e = err

    def _inner():
        raise e

    try:
        _inner()
    except errors.CopierError:
        print(formatException())
        assert formatException() == textwrap.dedent('''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_traceback_reraising
                _inner()
              File "{file}", line {line2}, in _inner
                raise e
              ----------------------- catching remote run ------------------------
              File "{file}", line {line3}, in test_traceback_reraising
                raise errors.CopierError('error here')
            CopierError: error here
            ''').format(file=__file__.rstrip('c'), line1=srcLine + 11, line2=srcLine + 8, line3=srcLine + 2)


def test_traceback_reraising_after_other_error():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('error here')
    except Exception as err:
        saveTraceback(err)
        e = err

    try:
        raise Exception('something other here')
    except:
        pass

    def _inner():
        raise e

    try:
        _inner()
    except errors.CopierError:
        print(formatException())
        assert formatException() == textwrap.dedent('''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_traceback_reraising_after_other_error
                _inner()
              File "{file}", line {line2}, in _inner
                raise e
              --------------------------------------------------------------------
              File "{file}", line {line3}, in test_traceback_reraising_after_other_error
                raise errors.CopierError('error here')
            CopierError: error here
            ''').format(file=__file__.rstrip('c'), line1=srcLine + 16, line2=srcLine + 13, line3=srcLine + 2)


def test_traceback_reraising_multiple_times():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('error here')
    except Exception as err:
        saveTraceback(err, 'something useful')
        e = err

    try:
        raise e
    except:
        print(formatException())
        assert formatException() == textwrap.dedent('''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_traceback_reraising_multiple_times
                raise e
              ------------------------- something useful -------------------------
              File "{file}", line {line2}, in test_traceback_reraising_multiple_times
                raise errors.CopierError('error here')
            CopierError: error here
            ''').format(file=__file__.rstrip('c'), line1=srcLine + 8, line2=srcLine + 2)

    try:
        raise e
    except errors.CopierError:
        print(formatException())
        assert formatException() == textwrap.dedent('''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_traceback_reraising_multiple_times
                raise e
              ------------------------- something useful -------------------------
              File "{file}", line {line2}, in test_traceback_reraising_multiple_times
                raise errors.CopierError('error here')
            CopierError: error here
            ''').format(file=__file__.rstrip('c'), line1=srcLine + 22, line2=srcLine + 3)


def test_format_not_available_without_exception_raised():
    try:
        raise errors.CopierError('error here')
    except Exception:
        pass

    sys.exc_clear()

    with raises(AssertionError):
        formatException()


def test_format_not_raised_exception():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        raise errors.CopierError('error here')
    except Exception as err:
        saveTraceback(err)
        e = err

    sys.exc_clear()

    with raises(AssertionError):
        formatException()

    print(formatException(e))
    assert formatException(e) == textwrap.dedent('''\
        Traceback (most recent call last):
          File "{file}", line {line}, in test_format_not_raised_exception
            raise errors.CopierError('error here')
        CopierError: error here
        ''').format(file=__file__.rstrip('c'), line=srcLine + 2)

    sys.exc_clear()

    print(formatException(e))
    assert formatException(e) == textwrap.dedent('''\
        Traceback (most recent call last):
          File "{file}", line {line}, in test_format_not_raised_exception
            raise errors.CopierError('error here')
        CopierError: error here
        ''').format(file=__file__.rstrip('c'), line=srcLine + 2)


def test_hiding_middle_traceback():
    e = None
    srcLine = inspect.currentframe().f_lineno
    try:
        try:
            try:
                raise errors.CopierError('deep one')
            except Exception as err:
                e = err
                raiseEx(errors.CopierError('middle one'), err, 'middle block')
        except Exception as err:
            e = err
            raise err
    except Exception:
        print(formatException())
        assert formatException(e) == textwrap.dedent('''\
            Traceback (most recent call last):
              File "{file}", line {line1}, in test_hiding_middle_traceback
                raise err
              --------------------------- middle block ---------------------------
              File "{file}", line {line2}, in test_hiding_middle_traceback
                raise errors.CopierError(\'deep one\')
            CopierError: middle one
            ''').format(file=__file__.rstrip('c'), line1=srcLine + 10, line2=srcLine + 4)


def test_raising_old_exception():
    def _doErr():
        raise errors.CopierError('done err')

    e = None
    try:
        _doErr()
    except errors.CopierError as err:
        saveTraceback(err)
        e = err

    error = [None]

    def _putErr(err):
        error[0] = err

    def _reraiseErr(err):
        raise error[0]

    try:
        _putErr(e)
        _reraiseErr(e)
    except errors.CopierError as err:
        print(formatException())


class TestSkynetBaseError1(SkynetBaseError):
    def __init__(self, val=None, val2=123):
        SkynetBaseError.__init__(self, val2)
        self.val = val


def test_skynet_base_error():
    err = TestSkynetBaseError1(val='xxx', val2='yyy')
    assert str(err) == "args: ('yyy',)"
    assert repr(err) == "TestSkynetBaseError1('yyy', message='')"

    err2 = pickle.loads(pickle.dumps(err))

    assert str(err2) == str(err)
    assert repr(err2) == repr(err)

    assert err2.args == ('yyy', )
    assert err2.val == 'xxx'


class TestSkynetBaseError2(SkynetBaseError):
    def __init__(self, msg, **extra):
        SkynetBaseError.__init__(self, message=msg, **extra)


def test_skynet_base_error_with_kwargs():
    err = TestSkynetBaseError2('some message', val1='xxx', val2='yyy', val3=None, val4=5)
    err2 = pickle.loads(pickle.dumps(err))

    assert str(err) == 'some message' == str(err2)
    assert repr(err) == "TestSkynetBaseError2(message='some message')" == repr(err2)

    assert err.args == err2.args == ()
    assert err.val1 == err2.val1 == 'xxx'
    assert err.val2 == err2.val2 == 'yyy'
    assert err.val3 is err2.val3 is None
    assert err.val4 == err2.val4 == 5


class TestSkynetBaseError3(SkynetBaseError):
    def __init__(self, message, *args, **kwargs):
        SkynetBaseError.__init__(self, *args, message=message, **kwargs)


def test_skynet_base_error_auto_formatting():
    err = TestSkynetBaseError3('Some message for user %s from %s', 'mocksoul', 'pg')
    err2 = pickle.loads(pickle.dumps(err))

    assert str(err) == str(err2) == 'Some message for user mocksoul from pg'
    assert repr(err) == repr(err2) == "TestSkynetBaseError3('mocksoul', 'pg', message='Some message for user %s from %s')"

    err = TestSkynetBaseError3('Some message for user {0} from {what!r} {1!s} times', 'mocksoul', 5, what='ziliboba')
    err2 = pickle.loads(pickle.dumps(err))

    assert str(err) == str(err2) == "Some message for user mocksoul from 'ziliboba' 5 times"
    assert repr(err) == repr(err2) == "TestSkynetBaseError3('mocksoul', 5, message='Some message for user {0} from {what!r} {1!s} times')"
