import gevent
import select


# class ThreadingMixIn:
#     """Mix-in class to handle each request in a new thread."""

#     # Decides how threads will act upon termination of the
#     # main process
#     daemon_threads = False

#     def process_request_thread(self, request, client_address):
#         """Same as in BaseServer but as a thread.

#         In addition, exception handling is done here.

#         """
#         try:
#             self.finish_request(request, client_address)
#             self.close_request(request)
#         except:
#             self.handle_error(request, client_address)
#             self.close_request(request)

#     def process_request(self, request, client_address):
#         """Start a new thread to process the request."""
#         t = threading.Thread(target = self.process_request_thread,
#                              args = (request, client_address))
#         if self.daemon_threads:
#             t.setDaemon (1)
#         t.start()


class GreenletMixIn(object):
    """Mix-in class to handle each request in a new greenlet."""
    def __init__(self, *args, **kwargs):
        super(GreenletMixIn, self).__init__(*args, **kwargs)
        self.__shutdownRequested = False
        self.__isShutDown = gevent.event.Event()

    def _process_request(self, request, client_address):
        """Same as in BaseServer but as a greenlet.

        In addition, exception handling is done here.

        """
        try:
            self.finish_request(request, client_address)
            self.close_request(request)
        except:
            self.handle_error(request, client_address)
            self.close_request(request)

    def process_request(self, request, client_address):
        """Start a new greenlet to process the request."""
        gevent.Greenlet.spawn(self._process_request, request, client_address)

    def serve_forever(self, poll_interval=0.5):
        """Handle one request at a time until shutdown.

        Polls for shutdown every poll_interval seconds. Ignores
        self.timeout. If you need to do periodic tasks, do them in
        another thread.
        """
        self.__isShutDown.clear()
        try:
            while not self.__shutdownRequested:
                # XXX: Consider using another file descriptor or
                # connecting to the socket to wake this up instead of
                # polling. Polling reduces our responsiveness to a
                # shutdown request and wastes cpu at all other times.
                r, w, e = select.select([self], [], [], poll_interval)
                if self in r:
                    self._handle_request_noblock()
        finally:
            self.__shutdownRequested = False
            self.__isShutDown.set()

    def shutdown(self):
        """Stops the serve_forever loop.
        """
        self.__shutdownRequested = True
        self.__isShutDown.wait()
