from __future__ import absolute_import, print_function, division
import sys
import xmlrpclib
import traceback
from SimpleXMLRPCServer import SimpleXMLRPCRequestHandler, SimpleXMLRPCServer
__import__("pkg_resources").require("simplejson")
try:
    __import__('pkg_resources').require('msgpack')
except:
    __import__('pkg_resources').require('msgpack-python')
import simplejson as json
import msgpack


class MarshallerXMLRPC(object):
    @staticmethod
    def loads(*args, **kwargs):
        params, method = xmlrpclib.loads(*args, **kwargs)
        return method, params

    @staticmethod
    def dumps(*args, **kwargs):
        return xmlrpclib.dumps(*args, **kwargs)


class MarshallerJSON(object):
    @staticmethod
    def loads(s):
        return json.loads(s)

    @staticmethod
    def dumps(obj, error=False):
        if error:
            return json.dumps(["err", str(obj)])
        return json.dumps(["ok", obj])


class MarshallerMSGPACK(object):
    @staticmethod
    def loads(s):
        return msgpack.loads(s)

    @staticmethod
    def dumps(obj, error=False):
        if error:
            return msgpack.dumps(["err", str(obj)])
        return msgpack.dumps(["ok", obj])


class MarshallerNotFoundError(Exception):
    pass


class RequestHandler(SimpleXMLRPCRequestHandler):
    rpc_paths = ('/xmlrpc', '/msgpack')

    def do_POST(self):
        """Handles the HTTP POST request.

        Attempts to interpret all HTTP POST requests as XML-RPC calls,
        which are forwarded to the server's _dispatch method for handling.
        """

        # Check that the path is legal
        if not self.is_rpc_path_valid():
            self.report_404()
            return

        try:
            # Get arguments by reading body of request.
            # We read this in chunks to avoid straining
            # socket.read(); around the 10 or 15Mb mark, some platforms
            # begin to have problems (bug #792570).
            max_chunk_size = 10*1024*1024
            size_remaining = int(self.headers["content-length"])
            L = []
            while size_remaining:
                chunk_size = min(size_remaining, max_chunk_size)
                L.append(self.rfile.read(chunk_size))
                size_remaining -= len(L[-1])
            data = ''.join(L)

            # In previous versions of SimpleXMLRPCServer, _dispatch
            # could be overridden in this class, instead of in
            # SimpleXMLRPCDispatcher. To maintain backwards compatibility,
            # check to see if a subclass implements _dispatch and dispatch
            # using that method if present.
            response = self.server._marshaled_dispatch(
                data, getattr(self, '_dispatch', None), self.path
            )
        except MarshallerNotFoundError:
            self.send_error(400, 'Marshaller Not Found')

        except Exception as e:  # This should only happen if the module is buggy
            # internal error, report as HTTP server error
            self.send_response(500)

            # Send information about the exception if requested
            if hasattr(self.server, '_send_traceback_header') and \
                    self.server._send_traceback_header:
                self.send_header("X-exception", str(e))
                self.send_header("X-traceback", traceback.format_exc())

            self.end_headers()

        else:
            # got a valid XML RPC response
            self.send_response(200)
            self.send_header("Content-type", "text/xml")
            self.send_header("Content-length", str(len(response)))
            self.end_headers()
            self.wfile.write(response)

            # shut down the connection
            self.wfile.flush()
            self.connection.shutdown(1)


class MultiMarshallerXMLRPCServer(SimpleXMLRPCServer):
    _send_traceback_header = True

    def __init__(self, addr, requestHandler=RequestHandler,
                 logRequests=True, allow_none=False, encoding=None):
        SimpleXMLRPCServer.__init__(self, addr, requestHandler, logRequests, allow_none,
                                    encoding)
        self.marshallers = {}
        self.allow_none = allow_none
        self.encoding = encoding

    def add_marshaller(self, path, marshaller):
        self.marshallers[path] = marshaller
        return marshaller

    def get_marshaller(self, path):
        return self.marshallers[path]

    def _marshaled_dispatch(self, data, dispatch_method, path):
        try:
            marshaller = self.marshallers[path]

        except KeyError:
            raise MarshallerNotFoundError

        try:
            method, params = marshaller.loads(data)

            # generate response
            if dispatch_method is not None:
                response = dispatch_method(method, params)
            else:
                response = self._dispatch(method, params)

            # wrap response in a singleton tuple
            response = (response,)
            response = marshaller.dumps(response, methodresponse=1,
                                        allow_none=self.allow_none, encoding=self.encoding)

        except xmlrpclib.Fault as fault:
            response = marshaller.dumps(fault, allow_none=self.allow_none,
                                        encoding=self.encoding)
        except:
            # report exception back to server
            exc_type, exc_value, exc_tb = sys.exc_info()
            response = marshaller.dumps(
                xmlrpclib.Fault(1, "%s:%s" % (exc_type, exc_value)),
                encoding=self.encoding, allow_none=self.allow_none,
            )

        return response
