from __future__ import print_function

import random
import socket
import sys
import gzip
import six
from six.moves import http_client as httplib
from six.moves import xmlrpc_client as xmlrpclib
from six.moves import cStringIO as StringIO
from six.moves.urllib_parse import urlsplit, urlunsplit
from kernel.util.errors import raiseEx, formatException


try:
    try:
        __import__('pkg_resources').require('simplejson')
    except:
        pass
    import simplejson as json
except ImportError:
    import json


try:
    __import__('pkg_resources').require('msgpack')
except:
    __import__('pkg_resources').require('msgpack-python')
import msgpack


class ByPassParser(object):
    def __init__(self, target):
        self.target = target
        self.rawData = []

    def feed(self, data):
        self.rawData.append(data)

    def close(self):
        self.target.feed(''.join(self.rawData))
        del self.target  # get rid of circular references


class UnmarshallSimple(object):
    def __init__(self):
        self.data = None

    def feed(self, data):
        assert self.data is None
        self.data = data

    def close(self):
        assert self.data is not None
        answ, data = self.unmarshall(self.data)
        if answ.lower() == 'ok':
            return data
        else:
            raise Exception('{0}: {1}'.format(answ, str(data)))

    def unmarshall(self, data):
        raise NotImplementedError


class UnmarshallJson(UnmarshallSimple):
    def unmarshall(self, data):
        return json.loads(data)


class UnmarshallMsgpack(UnmarshallSimple):
    def unmarshall(self, data):
        return msgpack.loads(data)


class TimeoutHTTPConnection(httplib.HTTPConnection):
    def connect(self):
        httplib.HTTPConnection.connect(self)
        self.sock.settimeout(self.timeout)


if sys.hexversion < 0x02070000:
    class TimeoutHTTP(httplib.HTTP):
        _connection_class = TimeoutHTTPConnection

        def set_timeout(self, timeout):
            self._conn.timeout = timeout


class TimeoutTransport(xmlrpclib.Transport):
    def __init__(self, timeout=10, protocol='xmlrpc', headersHandler=None, *l, **kw):
        xmlrpclib.Transport.__init__(self, *l, **kw)
        self.timeout = timeout
        self.protocol = protocol
        self.headersHandler = headersHandler

    if sys.hexversion < 0x02070000:
        def make_connection(self, host):
            conn = TimeoutHTTP(host)
            conn.set_timeout(self.timeout)
            return conn
    else:
        def make_connection(self, host):  # noqa
            if self._connection and host == self._connection[0]:
                return self._connection[1]
            chost, self._extra_headers, x509 = self.get_host_info(host)
            self._connection = host, TimeoutHTTPConnection(chost)
            self._connection[1].timeout = self.timeout
            return self._connection[1]

    def getparser(self):
        if self.protocol == 'xmlrpc':
            return xmlrpclib.Transport.getparser(self)

        if self.protocol == 'json':
            target = UnmarshallJson()
        elif self.protocol == 'msgpack':
            target = UnmarshallMsgpack()

        return ByPassParser(target), target

    def request(self, host, handler, request_body, verbose=0):
        # issue XML-RPC request

        h = self.make_connection(host)
        if verbose:
            h.set_debuglevel(1)

        self.send_request(h, handler, request_body)
        self.send_host(h, host)
        self.send_user_agent(h)
        h.putheader('Accept-Encoding', 'gzip')
        self.send_content(h, request_body)

        if sys.hexversion < 0x02070000:
            errcode, errmsg, headers = h.getreply()
        else:
            response = h.getresponse()
            errcode = response.status
            errmsg = response.reason
            headers = response.msg

        if errcode != 200:
            raise xmlrpclib.ProtocolError(
                host + handler,
                errcode, errmsg,
                headers
            )

        if (self.headersHandler is not None and callable(self.headersHandler)):
            self.headersHandler(headers)

        self.verbose = verbose

        if sys.hexversion < 0x02070000:
            try:
                sock = h._conn.sock
            except AttributeError:
                sock = None

            if headers.get('Content-Encoding', None) == 'gzip':
                return self._parse_gzipped_response(h.getfile(), sock)

            return self._parse_response(h.getfile(), sock)
        else:
            return self.parse_response(response)

    def _parse_gzipped_response(self, file, sock):
        p, u = self.getparser()
        response = StringIO()
        while 1:
            if sock:
                r = sock.recv(1024)
            else:
                r = file.read(1024)
            if not r:
                break
            response.write(r)

        response.seek(0)
        gzf = gzip.GzipFile(fileobj=response)
        responseDecompressed = gzf.read()
        gzf.close()

        if self.verbose:
            print("body:", repr(responseDecompressed))

        p.feed(responseDecompressed)
        file.close()
        p.close()
        return u.close()


class TimeoutServerProxy(xmlrpclib.ServerProxy):
    def __init__(self, uri, timeout=10, protocol='xmlrpc', headersHandler=None, *l, **kw):
        self.uri = uri
        self.timeout = timeout
        assert protocol in ('xmlrpc', 'json', 'msgpack')
        self.protocol = protocol
        kw['transport'] = TimeoutTransport(
            timeout=timeout, protocol=protocol,
            headersHandler=headersHandler,
            use_datetime=kw.get('use_datetime', 0)
        )
        xmlrpclib.ServerProxy.__init__(self, uri, *l, **kw)

    def __request(self, methodname, params):
        if self.protocol == 'json':
            dumps = json.dumps
        elif self.protocol == 'msgpack':
            dumps = msgpack.dumps

        request = dumps((methodname, params))
        response = self._ServerProxy__transport.request(
            self._ServerProxy__host,
            self._ServerProxy__handler,
            request,
            verbose=self._ServerProxy__verbose
        )

        return response

    def __getattr__(self, name):
        if self.protocol == 'xmlrpc':
            return xmlrpclib.ServerProxy.__getattr__(self, name)
        elif self.protocol in ('json', 'msgpack'):
            return xmlrpclib._Method(self.__request, name)


class RobustServerProxy:
    def __init__(self, uris, timeouts, *l, **kw):
        self.attempts = []

        if isinstance(uris, six.string_types):
            uris = [uris]

        for tout in timeouts:
            c = []

            for uri in uris:
                c.append(TimeoutServerProxy(uri, tout, *l, **kw))

            self.attempts.append(c)

    def getProxy(self, attempt, n):
        v = self.attempts[attempt]
        return v[(n + attempt) % len(v)]

    def __getattr__(self, name):
        def wrapper(*lst):
            n = random.randint(0, 100000)
            l = len(self.attempts)

            for attempt in range(0, l):
                proxy = self.getProxy(attempt, n)
                try:
                    return getattr(proxy, name)(*lst)
                except (
                    socket.error, xmlrpclib.ProtocolError,
                    httplib.BadStatusLine, httplib.ImproperConnectionState
                ) as e:
                    if attempt == l - 1:
                        message = (
                            'call "{func}{params}" failed after {n} attempts, '
                            'timeout={timeout}, original error="{error}"'
                        ).format(
                            error=e,
                            timeout=proxy.timeout, func=name,
                            params=str(lst), n=attempt + 1
                        )
                        raiseEx(socket.error(message), e)

        return wrapper


class StickyServerProxy:
    def __init__(self, headerName, balancerUri, timeouts, *l, **kw):
        self.headerName = headerName
        self.parsedUrl = urlsplit(balancerUri)
        self.timeouts = timeouts
        self.stickToHost = None
        self.l = l
        self.kw = kw

    def __getUrl(self):
        if self.stickToHost is not None:
            port = ""
            if self.parsedUrl.port:
                port = ":" + str(self.parsedUrl.port)
            return urlunsplit([
                self.parsedUrl.scheme,
                "".join([self.stickToHost, port]),
                self.parsedUrl.path,
                self.parsedUrl.query,
                self.parsedUrl.fragment])
        else:
            return urlunsplit(self.parsedUrl)

    def __attempts(self):
        return [
            TimeoutServerProxy(self.__getUrl(), tout, headersHandler=self.__headersHandler, *self.l, **self.kw)
            for tout in self.timeouts]

    def __headersHandler(self, headers):
        if self.stickToHost is None and self.headerName in headers:
            self.stickToHost = headers[self.headerName]

    def __getattr__(self, name):
        def wrapper(*lst):
            attempts = self.__attempts()
            l = len(attempts)
            for attempt in range(0, l):
                proxy = attempts[attempt]
                try:
                    return getattr(proxy, name)(*lst)
                except socket.error as e:
                    if attempt == l - 1:
                        message = (
                            'call "{func}{params}" failed after {n} attempts, '
                            'timeout={timeout}, original error="{error}"'
                        ).format(
                            error=e,
                            timeout=proxy.timeout, func=name,
                            params=str(lst), n=attempt + 1
                        )
                        raiseEx(socket.error(message), e)

        return wrapper


if __name__ == "__main__":
    # s = RobustServerProxy(['http://127.0.0.1:9090/msgpack'], [0.05, 0.2, 0.2, 1.2, 3], allow_none=True, protocol='msgpack')
    s = StickyServerProxy(
        "CMS-InstanceHost",
        'http://localhost:8000/msgpack/bs',
        [0.05, 0.2, 0.2, 1.2, 3],
        allow_none=True, protocol='msgpack')
    try:
        s.ping()
        s.ping()
    except:
        print(formatException())
