import zmq
import os

from .context import Context
from .fixpath import fix, pathForHB
from .errors import WireError
from .nointr import noIntr


class Func(object):
    def __init__(self, name, parent):
        self.parent = parent
        self.name = name

    def __call__(self, *args, **kwargs):
        self.parent.s.send_pyobj((self.name, args, kwargs))

        while True:
            ready = self.parent.p.poll(timeout=self.parent.hbcount * 1000)

            if not ready:
                self.parent.reconnect()
                raise WireError()

            for (s, events) in ready:
                if s == self.parent.s:
                    (res, err) = noIntr(s.recv_pyobj)

                    if err:
                        raise err

                    return res
                else:
                    noIntr(s.recv_pyobj)


class Client(object):
    def __init__(self, path, heartbeats=3):
        self.hbcount = heartbeats
        self.path = fix(path)
        self.reconnect()

    def maySendRequest(self):
        path = self.path

        if path.startswith('ipc://'):
            return os.path.exists(path[6:])

        return True

    def reconnect(self):
        path = self.path
        ctx = Context()

        p = zmq.Poller()

        s = ctx.socket(zmq.REQ)
        s.setsockopt(zmq.LINGER, 0)
        s.connect(path)

        p.register(s, zmq.POLLIN)

        h = ctx.socket(zmq.SUB)
        h.connect(pathForHB(path))
        h.setsockopt(zmq.SUBSCRIBE, '')

        p.register(h, zmq.POLLIN)

        self.s = s
        self.h = h
        self.p = p

    def __getattr__(self, name):
        if name.startswith('__'):
            raise AttributeError(name)

        return Func(name, self)


class CheckedFunc(object):
    def __init__(self, slave):
        self.slave = slave

    def __call__(self, *args, **kwargs):
        if self.slave.parent.maySendRequest():
            return self.slave(*args, **kwargs)

        raise WireError()


class CtlClient(object):
    def __init__(self, *args, **kwargs):
        self.client = Client(*args, **kwargs)

    def __getattr__(self, name):
        return CheckedFunc(getattr(self.client, name))
