from __future__ import print_function

import fcntl
import os
import sys
import Queue
import threading
import zmq
import socket
import errno

from kernel.util.misc import safeOpen
from kernel.util.errors import formatException

from .stop import Stop
from .nointr import noIntr


class HeartBeat(object):
    def __init__(self, sock):
        self.s = sock
        self.q = Queue.Queue()

    def run(self):
        while True:
            try:
                self.q.get(timeout=1)

                return
            except Queue.Empty:
                pass

            self.s.send_pyobj('pg - the best')

    def stop(self):
        self.q.put('stop')

    def __enter__(self):
        threading.Thread(target=self.run).start()

        return self

    def __exit__(self, *args):
        self.stop()


def runZeroMQRequestServer(socket, server):
    stop = False

    def defaultFunc():
        raise Exception('method %r not implemented' % attr)

    def stopFunc():
        raise Stop()

    while not stop:
        (attr, args, kwargs) = noIntr(socket.recv_pyobj)

        try:
            ret = (getattr(server, attr, stopFunc if attr == 'stop' else defaultFunc)(*args, **kwargs), None)
        except Stop:
            ret = (None, None)
            stop = True
        except (KeyboardInterrupt, SystemExit):
            raise
        except Exception as e:
            ret = (None, e)

        try:
            socket.send_pyobj(ret)
        except Exception as err:
            socket.send_pyobj((None, err))
            print(repr(ret), file=sys.stderr)
            print(formatException(), file=sys.stderr)


class Lock(object):
    def __init__(self, path, addr):
        self.path = path
        self.addr = addr
        self.fd = None
        self.good = False

    def __enter__(self):
        prefix = 'ipc://'

        if self.path.startswith(prefix):
            self.good = True
            self.path = self.path[len(prefix):] + '.lock'

        if self.good:
            self.fd = safeOpen(self.path, 'wb')

            try:
                fcntl.flock(self.fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
            except IOError:
                raise Exception('address %s already binded' % self.addr)

        return self

    def __exit__(self, *args):
        if self.fd:
            os.unlink(self.path)

    def check(self):
        self.__enter__()
        return self.good


class SockWrapper(object):
    def __init__(self, ctx, sockType, path):
        self.__sockType = sockType
        self.__ctx = ctx
        self.__sock = None
        self.__path = path
        assert self.__path.startswith('ipc://')
        self.__rpath = self.__path[len('ipc://'):]

        self.reopen()

    def reopen(self):
        if self.__sock is not None:
            try:
                self.__sock.close()
            except:
                pass

        self.__sock = self.__ctx.socket(self.__sockType)

    def __getattr__(self, attr):
        return getattr(self.__sock, attr)

    def bind(self):
        return self.__sock.bind(self.__path)

    def tryBind(self):
        if os.path.exists(self.__rpath):
            sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            refused = False
            try:
                sock.connect(self.__rpath)
            except socket.error as err:
                if err.errno == errno.ECONNREFUSED:
                    refused = True
            except:
                pass
            finally:
                try:
                    sock.close()
                except:
                    pass

            if not refused:
                return

        self.__sock.bind(self.__path)


class Sockets(object):
    def __init__(self, sockets):
        self.sockets = sockets

    def __enter__(self):
        return self.sockets

    def __exit__(self, *args):
        for s in self.sockets:
            s.close()


def loop(addr, server):
    from . import context, fixpath

    path = fixpath.fix(addr)

    with Lock(path, addr) as lock:
        ctx = context.Context()

        with Sockets((SockWrapper(ctx, zmq.REP, path), SockWrapper(ctx, zmq.PUB, fixpath.pathForHB(path)))) as (ss, hb):
            ss.bind()
            hb.bind()
            with HeartBeat(hb):
                thread = threading.Thread(target=runZeroMQRequestServer, args=(ss, server))
                thread.start()

                try:
                    while thread.isAlive():
                        thread.join(30)
                        if lock.check():
                            ss.tryBind()
                            hb.tryBind()
                except (KeyboardInterrupt, SystemExit):
                    with Sockets((ctx.socket(zmq.REQ), )) as (stopSock, ):
                        stopSock.connect(path)
                        stopSock.send_pyobj(('stop', (), {}))
                        stopSock.recv_pyobj()
                        thread.join()
                    raise
