"""Sign routines"""

import stat
import six

from kernel.util.functional import singleton

from .base import FileKeysAuthManager, BaseAuthManager, ChainAuthManager


class SignManager(BaseAuthManager):
    """Sign manager"""

    def filterKey(self, key):
        """
        :type key: .key.RSAKey
        """
        return key.hasPrivate()

    def sign(self, hash, fingerprints=None):
        """
        :param string hash: hash received with hashFunc().digest()
        :param fingerprints: iterable of interested fingerprints or None if all needed
        :return: iterable of (fingerprint, sign) pairs
        """
        if isinstance(fingerprints, six.binary_type):
            fingerprints = (fingerprints,)

        if not fingerprints:
            fingerprints = self.fingerprints()

        for fingerprint in fingerprints:
            key = self._keys.get(fingerprint)
            """:type: services.cqueue.src.auth.key.RSAKey"""
            if key:
                signature = key.sign(key.correctedHash(hash))
                signature = key.encodeSign(signature)

                self.log.debug('{0} signed with key {1}'.format(self, key))
                yield (fingerprint, signature)

    def _checkPermissions(self, filePath, uid, st):
        if st.st_mode & 0o77 != 0:
            self.log.warning('Key `{}` has invalid permissions: {:04o} (at max 0700 expected)'.format(filePath, stat.S_IMODE(st.st_mode)))
            return False

        if st.st_uid not in (0, uid):
            self.log.warning('Key `{}` owner is not valid: {} (0 or {} expected)'.format(filePath, st.st_uid, uid))
            return False

        return True


class FileKeysSignManager(SignManager, FileKeysAuthManager):
    pass


class ChainSignManager(ChainAuthManager):
    def sign(self, hash, fingerprints=None):
        if isinstance(fingerprints, six.string_types):
            fingerprints = (fingerprints,)

        if not fingerprints:
            fingerprints = self.fingerprints()

        if not isinstance(fingerprints, set):
            fingerprints = set(fingerprints)

        for manager in self.managers:
            if not fingerprints:
                return

            intersection = fingerprints & manager.fingerprints()

            if intersection:
                for result in manager.sign(hash, intersection):
                    yield result

            fingerprints -= intersection


@singleton
def gSignManager():
    """Global default sign manager"""
    return FileKeysSignManager()
