"""
Ssh agent signer implementation
"""

import os
import six

import socket
import threading

from kernel.util.sys.user import getUserName
from kernel.util.net.socketstream import SocketStream

from .base import BaseAuthManager
from .key import CryptoKey, Certificate
from . import sshagent


class SignAgentClient(BaseAuthManager):
    def __init__(self, address=None, log=None):
        super(SignAgentClient, self).__init__(log=log)

        self.sck = None
        self.stream = None
        self.remoteKeys = []

        if address is None:
            if os.name != 'posix':
                return

            address = os.getenv('SSH_AUTH_SOCK')

            if not isinstance(address, six.string_types) or not address:
                self.log.debug('{} no auth sock found'.format(self))
                return

        self.sck = socket.socket(socket.AF_UNIX)
        self.sck.setblocking(True)

        self.lock = threading.Lock()

        try:
            self.log.debug('{0} connecting to auth sock {1}'.format(self, address))
            self.sck.connect(address)
        except EnvironmentError:
            self.log.debug('{0} connection to auth sock failed'.format(self))
            return

        self.stream = SocketStream(self.sck)

    def running(self):
        return bool(self.stream)

    def load(self, username=None):
        if not self.running():
            return

        with self.lock:
            try:
                # with gevent.Timeout(10, EnvironmentError(errno.ETIMEDOUT, 'Timeout when loading agent keys')):
                self.log.debug('{0} reading keys'.format(self))
                sshagent.RequestIdentitiesMessage().write(self.stream)
                reply = sshagent.SshMessage.Load(self.stream, self.log)
                for keyData, comment in reply.keys:
                    key = CryptoKey.fromNetworkRepresentation(keyData, log=self.log)
                    if key is not None:
                        key.comment = comment
                        key.userNames.add(getUserName())
                        self.addKey(key)
                self.log.debug('{0} got {1} keys'.format(self, len(self._keys)))
            except (EnvironmentError, RuntimeError) as err:
                self.log.warning('{0} error {1}'.format(self, err))
                self.shutdown()

    def sign(self, hash, fingerprints=None):
        """
        :param string hash: hash received with hashFunc().digest()
        :param fingerprints: iterable of interested fingerprints or None if all needed
        :return: iterable of (fingerprint, sign) pairs
        """
        if not self.running():
            return

        if isinstance(fingerprints, six.binary_type):
            fingerprints = (fingerprints,)

        if not fingerprints:
            fingerprints = self.fingerprints()

        with self.lock:
            try:
                for fingerprint in fingerprints:
                    key = self._keys.get(fingerprint)
                    """:type: .key.CryptoKey"""
                    if key:
                        # with gevent.Timeout(
                        #   10,
                        #   EnvironmentError(errno.ETIMEDOUT, 'Timeout when signing with key {0}'.format(key))
                        # ):
                        msg = sshagent.SignRequestMessage()
                        msg.key_blob = key.publicKey().networkRepresentation()
                        msg.data = hash
                        msg.flags = 0
                        msg.write(self.stream)
                        reply = sshagent.SshMessage.Load(self.stream, self.log)
                        if not isinstance(reply, sshagent.SignResponseMessage):
                            self.log.info(
                                '{0} agent reports failure to sign with key {1}'.format(self, key)
                            )
                            continue

                        self.log.debug('{0} signed with key {1}'.format(self, key))

                        if isinstance(key, Certificate):
                            # We have to do this to maintain backward compatibility for some time.
                            # 1. Key contents ain't normally transferred to remote side, so there
                            #    would be no key to verify signature, and no way to attribute it
                            #    to CA, and no way to validate the certificate itself.
                            # 2. We cannot just add certificate contents as third member in the sign
                            #    tuple like (fingerprint, signature, certificate), because current
                            #    code in production expects exactly 2-member tuples and ain't ready
                            #    for extension. And signature is expected to be a decodable string,
                            #    so it leaves only a fingerprint used as an index to be safe.
                            fingerprint = (fingerprint, key.exportKey('OpenSSH'))

                        yield fingerprint, reply.signature

            except (EnvironmentError, RuntimeError) as err:
                self.log.warning('{0} error {1}'.format(self, err))
                self.shutdown()
                return

    def shutdown(self):
        if not self.sck:
            return
        self.log.debug('{0} shutdown'.format(self))
        self.remoteKeys = []
        try:
            self.sck.close()
        except EnvironmentError:
            pass
        self.sck = None
        self.stream = None


class _Consts(object):
    failure = (5, 30, 102)
    requestIdentities = 11
    identitiesAnswer = 12
    signRequest = 13
    signResponse = 14
