"""Tests for the ssh-agent features"""

import os
import logging

import pytest

from library.auth import sshagent
from library.auth.signagent import SignAgentClient
from library.auth.tempkey import TempKey
from library.auth.key import RSAKey, DSSKey, ECDSAKey
from library.auth.sign import FileKeysSignManager
from library.auth.verify import FileKeysVerifyManager


key_classes = (RSAKey, DSSKey,) if ECDSAKey is None else (RSAKey, DSSKey, ECDSAKey,)


@pytest.fixture(scope='function', autouse=True)
def setup_logging():
    logging.getLogger().setLevel(logging.INFO)
    logging.getLogger().addHandler(logging.StreamHandler())


@pytest.fixture
def managers():
    return FileKeysSignManager(), FileKeysVerifyManager()


@pytest.yield_fixture
def tempkeys(managers):
    temp_keys = [TempKey(*managers, keyClass=key_class) for key_class in key_classes]
    keys = [key.__enter__() for key in temp_keys]
    try:
        yield keys
    finally:
        for key in temp_keys:
            key.__exit__()


@pytest.yield_fixture
def agent(managers):
    ssh_agent = sshagent.SshAgent(managers[0])
    try:
        ssh_agent.start()
        yield ssh_agent
    finally:
        ssh_agent.stop()


@pytest.mark.skipif(os.name != 'posix', reason='Posix only')
def test_list_keys(managers, tempkeys, agent):
    """Tests the retrieval of keys from the remote agent"""
    sign_agent = SignAgentClient(agent.socketPath)
    sign_agent.load()
    assert sign_agent.fingerprints() == managers[0].fingerprints()
    for key in managers[0]:
        key_hash = key.fingerprint()
        sig1 = list(sign_agent.sign(key_hash, key.fingerprint()))
        sig2 = list(managers[0].sign(key_hash, key.fingerprint()))
        assert len(sig1) == len(sig2)
        assert 1 == len(sig1)
        # We can't just compare signs, since they're always different
        assert managers[1].verify(key_hash, sig1), '\n'.join((
            'Private key: %r' % (key.exportKey('PEM'),),
            'Data: %r' % (key_hash,),
            'Signature: %r' % (sig1,),
        ))
        assert managers[1].verify(key_hash, sig2), '\n'.join((
            'Private key: %r' % (key.exportKey('PEM'),),
            'Data: %r' % (key_hash,),
            'Signature: %r' % (sig2,),
        ))
