import os
import sys
import logging
import textwrap
from hashlib import sha1

from kernel.util.sys.user import getUserName
from kernel.util.sys.dirut import TempDir

from library.auth.tempkey import TempKey
from library.auth.key import RSAKey, DSSKey, ECDSAKey, CryptoKey
from library.auth.sign import (
    gSignManager,
    FileKeysSignManager,
    ChainSignManager,
    SignManager
)

from library.auth.verify import (
    gVerifyManager,
    FileKeysVerifyManager,
    ChainVerifyManager,
    VerifyManager
)

import six
import pytest

try:
    import paramiko
except ImportError:
    paramiko = None


keyClasses = (RSAKey, DSSKey,) if ECDSAKey is None else (RSAKey, DSSKey, ECDSAKey,)


@pytest.fixture(scope='function', autouse=True)
def setup_logging():
    logger = logging.getLogger()
    logger.setLevel(logging.INFO)
    if not any(
        isinstance(handler, logging.StreamHandler)
        for handler in logger.handlers
    ):
        logger.addHandler(logging.StreamHandler())


@pytest.mark.parametrize('keyClass', keyClasses)
def test_gsign_verify_pair(keyClass):
    """Test for paired signer verifier"""
    with TempKey(keyClass=keyClass) as key:
        dataHash = key.hash().digest()
        signs = list(gSignManager().sign(dataHash, [key.fingerprint()]))
        assert len(signs) == 1
        assert signs[0][0] == key.fingerprint()
        assert gVerifyManager().verify(dataHash, signs), '\n'.join((
            'Private Key: %r' % (key.exportKey('PEM'),),
            'Data Hash: %r' % (dataHash,),
            'Signatures: %r' % (signs,),
        ))


@pytest.mark.parametrize('keyClass', keyClasses)
def test_key_generate(keyClass):
    """Test key serialization/deserialization"""
    with TempKey(keyClass=keyClass) as key:
        assert key.__class__ is keyClass
        assert key.fingerprint() == CryptoKey.fromNetworkRepresentation(key.networkRepresentation()).fingerprint()
        assert key.fingerprint() == next(CryptoKey.loads(key.exportKey())).fingerprint()


@pytest.mark.parametrize('keyClass', keyClasses)
def test_sign_verify_pair(keyClass):
    """Test just created sign / verify managers"""
    managers = FileKeysSignManager(), FileKeysVerifyManager()
    with TempKey(*managers, keyClass=keyClass) as key:
        dataHash = key.hash().digest()
        signs = list(managers[0].sign(dataHash, [key.fingerprint()]))
        assert len(signs) == 1
        assert signs[0][0] == key.fingerprint()
        assert managers[1].verify(dataHash, signs), '\n'.join((
            'Private Key: %r' % (key.exportKey('PEM'),),
            'Data Hash: %r' % (dataHash,),
            'Signatures: %r' % (signs,),
        ))


@pytest.mark.parametrize('keyClass', keyClasses)
def test_failed_verification(keyClass):
    """Test failed verification"""
    with TempKey(verifyManager=None, keyClass=keyClass) as key:
        dataHash = key.hash().digest()
        signs = list(gSignManager().sign(dataHash, [key.fingerprint()]))
        len(signs) == 1
        signs[0][0] == key.fingerprint()
        assert not gVerifyManager().verify(dataHash, signs)


@pytest.mark.parametrize('keyClass', keyClasses)
def test_common_file_authorization(keyClass):
    """Load of common keys"""
    with TempDir() as tmpDir:
        sManager = FileKeysSignManager()

        vManager = FileKeysVerifyManager(
            commonKeyDirs=[tmpDir],
            keyFiles=['key.public$']
        )

        with TempKey(signManager=sManager, verifyManager=None, keyClass=keyClass) as key:
            dataHash = key.hash().digest()  # hash for empty string

            fileName = os.path.join(tmpDir, 'key.public')

            with open(fileName, 'wb') as f:
                f.write(key.publicKey().exportKey())

            # Permissions not mater
            os.chmod(fileName, 0o777)

            vManager.load()
            signatures = list(sManager.sign(dataHash))
            assert vManager.verify(dataHash, signatures), '\n'.join((
                'Private Key: %r' % (key.exportKey('PEM'),),
                'Data Hash: %r' % (dataHash,),
                'Signatures: %r' % (signatures,),
            ))


@pytest.mark.parametrize('keyClass', keyClasses)
def test_user_file_authorization(keyClass):
    """Test load of custom user keys"""
    with TempDir() as tmpDir:
        sManager = FileKeysSignManager()

        vManager = FileKeysVerifyManager(
            userKeyDirs=[os.path.join(tmpDir, '{userName}')],
            keyFiles=['key.public$']
        )

        with TempKey(signManager=sManager, verifyManager=None, keyClass=keyClass) as key:
            dataHash = key.hash().digest()  # hash for empty string

            userName = getUserName()

            fileName = os.path.join(tmpDir, userName, 'key.public')

            os.mkdir(os.path.dirname(fileName))

            with open(fileName, 'wb') as f:
                f.write(key.publicKey().exportKey())

            os.chmod(fileName, 0o700)

            vManager.load()
            signatures = list(sManager.sign(dataHash))
            assert vManager.verify(dataHash, signatures, userName), '\n'.join((
                'Private Key: %r' % (key.exportKey('PEM'),),
                'Data Hash: %r' % (dataHash,),
                'Signatures: %r' % (signatures,),
            ))


@pytest.mark.parametrize('keyClass', keyClasses)
def test_file_permissions(keyClass):
    with TempDir() as tmpDir:
        sManager = FileKeysSignManager(
            userKeyDirs=[os.path.join(tmpDir, '{userName}')],
            keyFiles=['key.private$']
        )

        vManager = FileKeysVerifyManager(
            userKeyDirs=[os.path.join(tmpDir, '{userName}')],
            keyFiles=['key.public$']
        )

        with TempKey(signManager=None, verifyManager=None, keyClass=keyClass) as key:
            userName = getUserName()
            os.mkdir(os.path.join(tmpDir, userName))

            publicFileName = os.path.join(tmpDir, userName, 'key.public')
            privateFileName = os.path.join(tmpDir, userName, 'key.private')

            with open(publicFileName, 'wb') as f:
                f.write(key.publicKey().exportKey())

            # Testgood permissions
            os.chmod(publicFileName, 0o700)

            vManager.load()
            assert key.fingerprint() in vManager.fingerprints()

            # Test bad permissions
            os.chmod(publicFileName, 0o777)

            vManager.load()
            assert key.fingerprint() not in vManager.fingerprints()

            with open(privateFileName, 'wb') as f:
                f.write(key.exportKey())

            # Test good permissions
            os.chmod(privateFileName, 0o700)

            sManager.load()
            assert key.fingerprint() in sManager.fingerprints()

            # Test bad permissions
            os.chmod(privateFileName, 0o744)

            sManager.load()
            assert key.fingerprint() not in sManager.fingerprints()


@pytest.mark.parametrize('keyClass', keyClasses)
def test_indented_file_authorization(keyClass):
    """Test load of user keys saved with indentation"""
    fileName = 'authorized_keys'
    with TempDir() as tmpDir:
        sManager = FileKeysSignManager()
        vManager = FileKeysVerifyManager(
            userKeyDirs=[os.path.join(tmpDir, '{userName}')],
            keyFiles=[fileName]
        )

        with TempKey(signManager=sManager, verifyManager=None, keyClass=keyClass) as key:
            dataHash = key.hash().digest()
            userName = getUserName()
            dirName = os.path.join(tmpDir, userName)
            filePath = os.path.join(dirName, fileName)
            os.mkdir(dirName)

            for indentation, options in (
                (
                    b'',
                    {},
                ),
                (
                    b' ',
                    {},
                ),
                (
                    b'  ',
                    {},
                ),
                (
                    b'\t',
                    {},
                ),
                (
                    b'from="*.sales.example.net,!pc.sales.example.net" ',
                    {
                        'from': "*.sales.example.net,!pc.sales.example.net",
                    },
                ),
                (
                    b'command="dump /home",no-pty,no-port-forwarding ',
                    {
                        'command': "dump /home",
                        'no-pty': None,
                        'no-port-forwarding': None,
                    },
                ),
                (
                    b'tunnel="0",command="sh /etc/netstart tun0" ',
                    {
                        'tunnel': '0',
                        'command': 'sh /etc/netstart tun0',
                    }
                ),
                (
                    b'cert-authority,principals="torkve" ',
                    {
                        'cert-authority': None,
                        'principals': 'torkve',
                    }
                ),
            ):
                with open(filePath, 'wb') as f:
                    f.write(indentation + key.publicKey().exportKey(keyformat='OpenSSH') + b'\n')
                # Testgood permissions
                os.chmod(filePath, 0o700)

                vManager.load()
                assert len(list(vManager)) == 1
                assert next(iter(vManager)).options == options
                assert vManager.verify(dataHash, list(sManager.sign(dataHash)), userName), 'Indentation `{}` in {} failed'.format(indentation, fileName)


def test_chain_sign():
    s1, v1 = SignManager(), VerifyManager()
    s2, v2 = SignManager(), VerifyManager()
    sc = ChainSignManager([s1, s2])
    vc = ChainVerifyManager([v1, v2])
    for key1Class in keyClasses:
        for key2Class in keyClasses:
            with TempKey(s1, v1, keyClass=key1Class):
                with TempKey(s2, v2, keyClass=key2Class):
                    dataHash = RSAKey.hash().digest()

                    assert v1.verify(dataHash, s1.sign(dataHash))
                    assert v2.verify(dataHash, s2.sign(dataHash))
                    assert not v1.verify(dataHash, s2.sign(dataHash))
                    assert not v2.verify(dataHash, s1.sign(dataHash))

                    assert vc.verify(dataHash, list(sc.sign(dataHash)))

                    assert v1.verify(dataHash, list(sc.sign(dataHash)))
                    assert v2.verify(dataHash, list(sc.sign(dataHash)))

                    assert vc.verify(dataHash, list(s1.sign(dataHash)))
                    assert vc.verify(dataHash, list(s2.sign(dataHash)))


@pytest.mark.skipif(ECDSAKey is None, reason="ecdsa is not available")
def test_bug_ecdsa_key_load():
    key = (
        b'\x00\x00\x00\x13ecdsa-sha2-nistp256\x00\x00\x00\x08nistp256'
        b'\x00\x00\x00A\x04\xa1\xbe?+\x15\xe0R\xd7\x12^\x0e\xf7\xc6\xfb'
        b'\x98\x0e\xec\xd8\xbd\xda\xfa\xfc_\xac\xdf\xf4\xab\xe5\xb1\xfd#'
        b'\xe8\xfcI\xa3Q\xf81\x17\x8d\xd9\xa7\x06\xe8\x0f\xd3Q@\x17)_{'
        b'\xa5\xeb\x84Y\xde\xa7;sT\xe6n\n'
    )
    assert isinstance(ECDSAKey.fromNetworkRepresentation(key), ECDSAKey)


@pytest.mark.skipif(paramiko is None, reason="paramiko is not available")
@pytest.mark.parametrize(
    'KeyClass,PKeyClass',
    [
        (RSAKey, (paramiko.RSAKey if paramiko else None)),
        (DSSKey, (paramiko.DSSKey if paramiko else None)),
    ] + ([] if ECDSAKey is None or paramiko is None else [(ECDSAKey, paramiko.ECDSAKey)]),
    indirect=not getattr(sys, 'is_standalone_binary', False),
)
def test_paramiko_compat(KeyClass, PKeyClass):
    """Test paramiko compatibility layer"""
    key = KeyClass.generate()
    key_data = key.exportKey('PEM') + b'\n'
    pkey = PKeyClass(file_obj=six.moves.StringIO(key_data.decode()))
    assert pkey is not None
    assert key.get_fingerprint() == pkey.get_fingerprint()
    assert key.get_base64() == pkey.get_base64()
    # assert key.get_bits() == pkey.get_bits()
    assert key.get_name() == pkey.get_name()
    assert key.can_sign() == pkey.can_sign()
    assert key.asbytes() == pkey.asbytes()

    data = sha1(b'x' * 8192).digest()
    sign = pkey.sign_ssh_data(data)
    sign.rewind()
    assert pkey.verify_ssh_sig(data, sign)
    # self sign and verify
    sign = key.sign_ssh_data(data)
    sign.rewind()
    assert key.verify_ssh_sig(data, sign), '\n'.join((
        'Private key: %r' % (key_data,),
        'Data: %r' % (data,),
        'Signature: %r' % (sign,),
    ))
    # paramiko sign, self verify
    sign = pkey.sign_ssh_data(data)
    sign.rewind()
    assert key.publicKey().verify_ssh_sig(data, sign), '\n'.join((
        'Private key: %r' % (key_data,),
        'Data: %r' % (data,),
        'Signature: %r' % (sign,),
    ))

    # self sign, paramiko verify
    sign = key.sign_ssh_data(data)
    sign.rewind()
    assert pkey.verify_ssh_sig(data, sign), '\n'.join((
        'Private key: %r' % (key_data,),
        'Data: %r' % (data,),
        'Signature: %r' % (sign,),
    ))


def test_skydev_2235():
    key_data = textwrap.dedent(
        """\
        -----BEGIN RSA PRIVATE KEY-----
        MIIEpAIBAAKCAQEAz7y6Fk+kTf9Wb15gPEmMyFlJz1XCD9Qx0qhCs23kJTvjWFwB
        KotBo5SHPDyJK7jYIskCxe4kzEWccnY8ap53Bg+WnuNXg3ZijWfkCHfjB+q7lUky
        /Hi5Rum2+PUoff60jKgI50gYqRyhLsUgEnrgni81n4Zb1WE9C1T3dSGqSGn1aBLx
        S4hG50Slgn+waQvv1cdkJTzu/k1fv+VBycOoLLZH4kVLc46H1b+OveInM3dGy44i
        7f4/WJXK5BBGhyfL8H3CHC5DvZa69dow3zzctCi7lOrvAIabrwnjP5GW2qFPKCtC
        du5N//D3OgX6oFLXns0fpbqHuaVb1SvaWdlVnQIDAQABAoIBAQCjOwBdl6bbQs+A
        IwArXv+oTcCZkiEynuGnTV9b7fVe7Hfvbhvceh/RkVvSER8Xo97xNlFfhYiUmf4c
        UpuWi32q3YTUxH53ZgZ6Zi/gF5PlHKTY4POXPqUW+34ykTCIECN+gmq8kQIPn9dn
        EK3q13K32RChkMmIGSgGR3VKLHHJA02o7hhUz3b6XEmqqVsGmjC2pXmatntKdfR5
        0dyStI9ZgiDBV6BUZnP+ROel16iuTpK3jab440I2qTLN79U5Kj4sl/L4BgNFB9rP
        5Rc47tt93P1syHzdO5Ok1Wvo4kRtG/ZBCB+x4sPU/GufYw7nSr5JLdly1rJmy8ZW
        2pz86v9hAoGBAPnF1nAiEffGsjgMDPhPr+6OTq6YwbLkgKGr0i5Srbl7by1wo4+S
        3xM0KJH9fmNVucuDTzI5q6H82SHR72O1JAHgVneu90tG92vGhX7Ksb2Mo0JxkC+a
        mcDD9SfuJNh1vUfxu877es44Xu/8O0z9gsLhb5E4JxoN/+UIe1Q8JuzvAoGBANTq
        mWazTmafTZDIo0o27CKaM8lQROEFygsdOtqmmvP90+r7HXNG+uXA7JU3Yp5+DpuB
        WpZ5LGuEU0MvJcMUMeLYfBUss+lqRy36Zbi7TxYuwdPspCevxqvoc2uqweejhtSF
        Trd6pL7vicDKJZnl+e5hwSCowMeuwen17Sd3Lf4zAoGAPDtVJt1Qb+vedl2Rkx/E
        UUbH+pqjuzwXOOVyXfFaK2x0twR+rNTCLqNZVTtICzRVBZq4smft17ZdtiYmEGlS
        MZGjyGDNCb4nVDofXyAKFKehhxhTqOcjRLT8nZB0Gi7qYjsNCzDhwHEfk8fDlSFd
        MxmiCeb8pu/RUa3l775BtMkCgYBCtEzi9AH3/KicV39w3MwV+yGGTHOH2/x0hu/N
        WKzrC/a8g3eyieZI3KxCA1wOM4f4CAqNYeEcvhQn1DmMaGNIonIdYCftVLKAIRaF
        TYl1daib8GNtW5BugyDF9nVRUNYI9sq2iOHcP8d0cuBc2sP2fwG2bagX5xQxbC6k
        Up5CKQKBgQCCeL1uC8VsNjHB4Z3WTOfsIjjMYUUR6W2nmQB8E9OJArlsfT1Wn/D9
        TgkDNoArUkiICAM0YVDj5/VIiUjsdhm9/MquWqRP0n49jjBfzIbqS4RYrtajlXxb
        iCZzrVVsJ4MNTyQqOLnwp1st1uWcpz0BjJHZGI3QQpeXPXng82p21g==
        -----END RSA PRIVATE KEY-----
        """)

    data = b'\x91\x05\xc5\xe7i\x95\x94:G\xc0\xb5J\x9d\x0c\xb2u\x19}\xf1\xaf'

    sign = (
        b"\x00\x00\x00\xff\x7f9\x1e\x849\xe7\x8d\x8a\x86~\x93\xfa\x0e#\x91"
        b".\x91\xe9\xe1\x87K\xd7[U\xe3\x89\x81\x9eA\x82x\xb0cWt\xad\x90;\x0c"
        b"RF\x04C\r\x92\xad\x9a\xc7\xd1O\x81\x13b\xe5\xa1e\xc2\xcb\xa2\xd0"
        b"\xc5d\xb0\n\xbc\xc7(\xbf\\j/\xb2\xe6\xdd\x81\xb9g\xaey\x96\xf5Q"
        b"\xecz\x07\x97\xad_h\x1e\xce\xd4\xd5\x0f\x7fMG(\x10\x7f\x92?|\xa6{"
        b"\xb3\xa6\xe2$\xc2\x82}\xb4\xd0\x83\xb2{\x94\xe1\x82\xf8\xbe\xb2{"
        b"\xf9roj\xfdq\xdcE^6RhS}\x18\xcf\xb79U\xc9$\xd2d$\x84\xeb\xd4L\xd6"
        b"\x13\xae\x00F\x91\x9a\xf1\xc8\x8bV\x94\xda\x10\xa8\n\xa7+\xd5\xff"
        b"7x\xe5\x819s\xebVlh\xb6\xdc\x05\xe9\xde\xa3[:\xe0\xf9sv_\x9f\x9d"
        b"\xf9F/\x10,(W\xe6\xd6F\x7f)\xf1y-\x81\xbf=\xfb\x80\xfb.)\x8d\xc6v"
        b"\xb69\x94\x01'\xd8\x0e\xbb\x7f(D\x1c\xcep\xde\xa6\xcb\xfb~6\x19"
        b"\x06\x11\x0b\xc4\x10\xc6'\x9b\x8a\x12\\"
    )

    key = next(RSAKey.loads(key_data))

    assert key.verify(key.correctedHash(data), key.decodeSign(sign))

    if paramiko is not None:
        msg = paramiko.Message()
        msg.add_string(key.get_name())
        msg.packet.write(sign)
        msg.rewind()
        assert key.verify_ssh_sig(data, msg)


def test_rtcsupport_9528():
    key_data = (
        'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABgQDBP/ioBJB5JJWaFAXhb0KiqMr26sfy+'
        'jhViED/r4nvRJxEnh+hNaoN19RscAHytT8+TOJyImcqv11gbD+2GNLT8O54G401Wk6WWz'
        '2CZKLwhvuqGTRIcKbeVZr0YtffB4iKZA2r1OfyMBe+gqosOQmhUcSm4y0kwbD0Dfz/lAJ'
        'zc3OO98dejVCrynrxIPtFodyIG/U//8ZlrKuMfAoo2aJvNUxsw26osaKufvBH2pE0IbZ2'
        'f4N1MCjeawAqa58aQSlaI0BSRk2qE5wWQE5jHOH04vXNOqLGpsd6s6hgFQ+UII+c/iECv'
        'gGQXZYqPuxqMN0Eu312M6nuRjgpFrAz4PJWQgLX3n9BIdMkCPAjuhRrDiSkiQzMdvBfZE'
        '42i4sMJ4Z1hFkTONb15qiOscVNyKk/JpHyxtl9h5Z0eD1lFtcG2UwYbTqA7946yU0keNY'
        'L0RScvNXEe/IIktBZC3d5bi+QLBxL22Nl3ds9pPD2tleiMwlfM/kpa6hYG32oynuDcPU='
        ' izetag@yandex-team.ru'
    )

    data = b'\xd3G\xea\x89\x8a\x91ls\x15\x94\xdd\xbf#\xa9\xa4\x84'

    sign = (
        b'\x00\x00\x01\x80"\x8ej\xd9\x11\xaaX5\xf4.\x0bM\xd3\xbb\xaeB\xa7V\xca'
        b'\x9c\xf8\xf5\x87\xa5\x8d\nrNz\x18\xfaE\x0bT\xce\x04\x95G\x11\xa5\x00'
        b'\xd6n\xe9\x99\xe1`X;-\xa7T`X*\xe6\x034>F\xa1\x123r\xf4:\x85s\x90ekx'
        b'\xef\xfc\xfa\xd0\xfe\x88\xa1K}\xe3\xe1\xcb\x81s\x84\xaa*S\xb0sry\xecX'
        b'\x11\xfc\xff_\xb8\xf9\xd4u\xf4{\x83\x85\xd6\xc22K\xddF\xe1\t=<\x8eaOm'
        b'\x80\xde\x1a\x91\xa8\xf0\xdfu\xf2/\x9fk\xaf\xc3\t\xbb\x17\x98#\x8bK'
        b'\x0f\x91\xa8\x8e\xc4<\xb1\x84\xd9?]\xdb\xf5\xdaGZ+\xd7\x02\x19\xbc'
        b'\x8c\x01\x0f\x11\x00\xdd\xca\x11\xf0$\xbb>b\xe7\xa5=Z\xe1Mh\x97a\x8a'
        b'\x00\x88hd\xdb\xca\xfa\x95\xe3\x0c\xb8\n\xad4\xa91\x1e\xf3=\x81\x87'
        b'\xfey\x13]\x0cx$\xa44A\xbb\xd6\xc2\x00\x9bA6\xfd \t[8\x01\xe3\xf7Mu'
        b'\xf4V\x9b\x0e\xfa\x0fCJF\xca\xa9\xe1\xc0\xb3\xde\xd0)~\xed\xf3\xcc'
        b'\xa5\x13\xbc\xc6\xa7\xec\xaf\x8f\xad9\xd13\x84\xa3\\\xa2\xb6\xe4\xbde'
        b'f\xc3"\x15\xd2F\xe5T4q\x19\xb7\xd7\xe2\xd3\xa8xN\xc8+\xca\x1c=\x85'
        b'\xe7\xad\xb8\x01"u\xc7\x01$\xdd\x8djz\xa6\x90\xe2\xd6\xb8!\x937T7\x88'
        b'|\xd6\xbf\x87\xc9_\x88\x9e9+A\x8a\xdf-\x0cMc+!\x9f\x89\xd6\xac>:\n%l'
        b'\x89\xde\xab\xc6\x87]g\x85!c\t\xeb\x9f"\xd7Oz\xb6\xfcz\xeb\xe2\xf9'
        b'\xd1\xa1\xc8\xd0*\xc8\x13G\xda'
    )

    key = next(RSAKey.loads(key_data))

    assert key.verify(key.correctedHash(data), key.decodeSign(sign))

    if paramiko is not None:
        msg = paramiko.Message()
        msg.add_string(key.get_name())
        msg.packet.write(sign)
        msg.rewind()
        assert key.verify_ssh_sig(data, msg)


def test_rtcsupport_12274():
    key_data = (
        b'ssh-rsa AAAAB3NzaC1yc2EAAAABJQAAAQBkDtPjwGBfbfBo3hzdCjqqDVhnl9EkxjE'
        b'8p4YnTYx4+kHhxcyIZcsxV5cvtWU3sMvnH8sx9TVzHrFMmN0OBLGSkZevT8M6BfmytK'
        b'E/rLN7HhugMJCFU1UI+eCg0yenqqyKGs30Uqe197+q8tS7Yp1nKpnCsLwhl83tPDyAY'
        b'UbtfkvzJFvg5iyLHCOfs41cLY+fufnE38o3hZtx54mvR2FsC5f6jvjCF8JRRRYw/6PR'
        b'jd2RxtrbGnUyeRtyfxo9UkjiBxtl6Zw1pqDkT6u2zFVetQRzDUKCloY5ge8CkTD4oVd'
        b'jUtgn4oiCXOrne97AMN/5USR7fVv//SNTy4bw/cYp rsa-key-20190717'
    )

    data = b'\xf73\xbe\xd1\xa1\xef<\xe47\xc9w\r\xe2O\xc2\x03'

    sign = (
        b'\x00\x00\x01\x00\x15\x07\xbb>\xf6\xdd\xb5\xdb6j&2 &\xf9T\x83\xf2=7H'
        b'UG\xbe\x87[2[/\xaf\xc4./\x17s\xc9\xe8\xbe!\x02\x011\r\xc84\xc7\xf5'
        b'\xd2CY\xa2i\xf2\xca\xc1P*\xf5\x84C\xd4F\xf3\xc6\xf19\xc1\x80\xb6'
        b'\xdf\xa7\xfb\xc2\x0e\xe1\xcd\xca\xee\x1bm\xe1-/\x95`\xabm\xf5\x8f'
        b'\x9b\xc9\xc5\xd7\x16- \xeaT\xf5\xbe\x9eeVD\xf5\xdb\xbc\x84\x91\xf6m'
        b'"MJ\x06\x0b\x1at~\xb4Y\xa7$\x11\xda&\x9e\x85\xb7\xf1|\xaa\xc3\xae'
        b'\xf6\x00\xd89\x1b\n$\x10\xa4\xb3\xdcq\x93`\xf63\xa3\xdc\xb3\'}\xfa'
        b'\x97\xb2\x8e^\x88\xd2\x1f\xf80m:2\xa9\x15<GT\xa9\xf1\xa1\xd8\xc4YT'
        b'\xc3\xe7\x08X\x8c\xfdp\xfa(0\xeb\xdbPi\xa7\x8c>\x8fC\xa9O\xb1\xc2E'
        b'\xbf\xcd\x11\x9fs\tf1\xe0\xe8TH\x7f\x90~K\xc1x\xed\xbb\xf3xh\xff'
        b'\x89X\x91W\xb3\xf4\'\xab\xc4A\xef\xf0\xc6\x98\xc6\xd0\xff5\x1av\x99'
        b'\xd2\xd7v\x00\x96\xa7\x86'
    )

    key = next(RSAKey.loads(key_data))

    assert key.verify(key.correctedHash(data), key.decodeSign(sign))

    if paramiko is not None:
        msg = paramiko.Message()
        msg.add_string(key.get_name())
        msg.packet.write(sign)
        msg.rewind()
        assert key.verify_ssh_sig(data, msg)
