"""
Here we defines authentication verifier
"""

import stat

import six

from kernel.util.functional import singleton
from kernel.util.sys.user import getUserName

from .base import FileKeysAuthManager, BaseAuthManager, ChainAuthManager


if six.PY2:
    b = lambda s: s
else:
    b = lambda s: s.encode('latin-1') if isinstance(s, str) else s


class VerifyManager(BaseAuthManager):
    """Verification manager"""

    def filterKey(self, key):
        """
        :type key: .key.CryptoKey
        """
        return not key.hasPrivate()

    def _verify_certificate(self, cert, hash, sign, user):
        """
        :type cert: .key.Certificate
        :param hash: hash received with hashFunct().digest()
        :param sign: encoded cryptographic signature of the hash
        :param user: username being validated
        :rtype: .key.CryptoKey or None
        """
        if not cert.is_user_certificate():
            self.log.debug("Certificate `%s` is not a user certificate", cert)
            return None

        if self._caStorage is None or not self._caStorage.cert_valid(cert):
            self.log.debug("Certificate `%s` signed by unknown CA: %s", cert, cert.signing_key)
            return None

        serveradmins = self._caStorage.serveradmins
        principals = set(cert.principals)
        in_allowed_roots = principals & serveradmins
        if b(user) not in principals and (b(user) != b'root' or not in_allowed_roots):
            self.log.debug("Certificate `%s` principals %s nor server admins %s does not contain user: %s", cert, cert.principals, serveradmins, user)
            return None

        sign = cert.decodeSign(sign)
        if cert.verify(cert.correctedHash(hash), sign):
            return cert

    def verify(self, hash, signs, user=None):
        """
        :param hash: hash received with hashFunc().digest()
        :param signs: iterable of (keyid, sign) or (keyid, sign, certificate) pairs
        :param user: if specified, there'll be used only specified user keys, otherwise default user will be used
        :rtype: .key.CryptoKey or None
        """
        if user is None:
            user = getUserName()

        signs = list(signs)

        # check certificates first
        for record in signs:
            if len(record) < 3:
                continue

            fingerprint = record[0]
            sign = record[1]
            certificate = record[2]

            cert = self._verify_certificate(certificate, hash, sign, user)
            if cert is not None:
                return cert

        userKeys = self._userKeys.get(user)

        if userKeys is None or not len(userKeys):
            self.load(user)
            userKeys = self._userKeys.get(user)
            if userKeys is None or not len(userKeys):
                return

        # now check keys
        for record in signs:
            fingerprint = record[0]
            sign = record[1]

            # skip already processed certs
            if len(record) > 2:
                continue

            if fingerprint not in userKeys:
                continue

            key = self._keys.get(fingerprint)
            """:type: services.cqueue.src.auth.key.CryptoKey"""
            if not key:
                continue

            # User keys shouldn't be used to authorize skynet session
            if user not in key.userNames:
                continue

            sign = key.decodeSign(sign)

            if key.verify(key.correctedHash(hash), sign):
                return key

    def _checkPermissions(self, filePath, uid, st):
        if st.st_mode & 0o22 != 0:
            self.log.warning("Key `{}` has invalid permissions: {:04o} (mustn't be writable)".format(filePath, stat.S_IMODE(st.st_mode)))
            return False

        if st.st_uid not in (0, uid):
            self.log.warning('Key `{}` owner is not valid: {} (0 or {} expected)'.format(filePath, st.st_uid, uid))
            return False

        return True


class FileKeysVerifyManager(VerifyManager, FileKeysAuthManager):
    pass


class ChainVerifyManager(ChainAuthManager):
    def verify(self, hash, signs, user=None):
        return any(manager.verify(hash, signs, user) for manager in self.managers)


@singleton
def gVerifyManager():
    """Global default verify manager"""
    return FileKeysVerifyManager()
