from __future__ import absolute_import, print_function

import re
import time
import gzip
import copy
import socket
import logging
import email.utils
import os
import sys

import six
import yaml
import py
import msgpack

from kernel.util import misc
from kernel.util.functional import singleton


REGISTRY_STORAGE_VERSION = 1  # Current registry data storage version.
CACHE_RECHECK_TIMEOUT = 10    # Cache  registry file for max 10 seconds.
CLOUD_RECHECK_TIMEOUT = 120   # Recheck cloud registry.
CLOUD_API_URL = 'http://api.genisys.yandex-team.ru/v1/hosts/{0}/skynet?fmt=msgpack'
CLOUD_API_CALL_TIMEOUT = 120  # Timeout in seconds for cloud API to respond.
CLOUD_FORCE_UPDATE = 7 * 24 * 60 * 60  # Forcedly fetch registry out from the cloud every week.

ACTIVE_FN = 'active.yaml'

RE_PLACEHOLDER = re.compile(r'\${([a-z0-9_.-]+)}', re.IGNORECASE)


class Config(dict):
    """
    The common parent class for any configuration slotted class' instances, which will
    be produced by @c query() function, commonly designed to distinguish them from plain dictionaries.
    """
    __slots__ = []


class RegistryStorageOutdated(Exception):
    """
    Special exception class to signal the registry cannot be loaded because it is out-of-date.
    """


@singleton
def registrydir():
    from api.srvmngr import getRoot

    return py.path.local(getRoot()).join('etc', 'registry').realpath()


@singleton
def logger():
    return logging.getLogger(__name__)


def _load(path, meta=True):
    """
    Internal caching function, which will fetch the registry raw content.
    :return: Cache for the registry path given.
    """

    class Cache(object):
        """
        The class represents local data cache storage with appropriate metadata.
        """
        __slots__ = ['atime', 'mtime', 'meta', 'data']

        def __init__(self):
            self.atime, self.mtime, self.meta, self.data = 0, 0, None, None

    ths = _load
    if not getattr(ths, '_cache', None):
        ths._cache = {}
    cache = ths._cache.get(path, None)
    if not cache:
        cache = ths._cache[path] = Cache()

    now = time.time()
    if cache.atime + CACHE_RECHECK_TIMEOUT > now:
        return cache

    mtime = path.stat().mtime
    if mtime <= cache.mtime:
        return cache

    logger().debug('Loading registry data from %r', path.strpath)
    data = yaml.load(path.open('rb'), Loader=getattr(yaml, 'CLoader', yaml.Loader))
    if meta:
        cache.meta = data['meta']
    else:
        cache.meta = {}

    cache.data = data['data']
    cache.mtime = mtime

    ver = cache.meta.get('version', None)
    if meta and ver != REGISTRY_STORAGE_VERSION:
        ex = RegistryStorageOutdated(
            'Registry storage %r version mismatch: current (%r) did not match storage (%r). Removing the storage.' %
            (path.strpath, REGISTRY_STORAGE_VERSION, ver)
        )
        logger().warn(str(ex))
        path.remove()
        raise RegistryStorageOutdated('registry storage outdated')
    else:
        cache.atime = now
    return cache


def _dict2slotted(dct, fix_keys=True):
    """
    Internal function for dictionary to slotted class instance transformation.
    :param dct:     Dictionary to be processed.
    :param fix_keys: Flags the dictionary keys should be transliterated to be safe Python identifier.
    :return:        Slotted class instance.
    """
    def str_key(k):
        return k.encode('utf-8') if six.PY2 and isinstance(k, six.text_type) else str(k)

    if dct is None:
        raise ValueError("Cannot convert NoneType to slotted class")

    re_key_esc = re.compile(r'(^[^a-z_]|[^a-z_0-9]|^\s*$)', re.IGNORECASE)
    keys = [str_key(k) if not fix_keys else re_key_esc.sub(r'_', str_key(k)) for k in dct.keys()]

    try:
        class _Config(Config):
            __slots__ = keys

            def __init__(self):
                super(_Config, self).__init__(dct)
    except TypeError:
        raise TypeError('Unable to set %r as class slots: not an identifier detected.' % keys)

    c = _Config()
    for key, value in dct.items():
        setattr(
            c,
            str_key(key) if not fix_keys else re_key_esc.sub(r'_', str_key(key)),
            _dict2slotted(value, fix_keys) if isinstance(value, dict) else value
        )
    return c


def detect_hostname():
    hostname = socket.gethostname().lower()

    if sys.platform == 'cygwin' and '.' not in hostname:
        hostname = socket.getfqdn(hostname).split()[0]

    ips = set()

    not_found_errs = (
        getattr(socket, 'EAI_NONAME', None),
        getattr(socket, 'EAI_NODATA', None),
    )

    try:
        addrinfo = socket.getaddrinfo(hostname, 0, 0, socket.SOCK_STREAM, 0)
    except socket.gaierror as ex:
        if ex.errno in not_found_errs:
            # Unable to resolve ourselves
            return hostname
        raise
    else:
        for ipinfo in addrinfo:
            ips.add(ipinfo[4][0])

        fqdns = set()

        for ip in ips:
            try:
                fqdn = socket.gethostbyaddr(ip)[0]
                if sys.platform == 'cygwin':
                    # on cygwin gethostbyaddr can return all DNS prefixes as an address, so we cut it here
                    fqdn = fqdn.split()[0]
                fqdns.add(fqdn)
            except socket.gaierror as ex:
                if ex.errno in not_found_errs:
                    continue
                raise
            except socket.herror as ex:
                # python has no error constants for gethostbyaddr
                if ex.strerror == 'Unknown host' or ex.errno == 1:
                    continue
                raise

        fqdns = list(fqdns)

        if hostname in fqdns:
            # Found hostname in fqdns
            return hostname
        elif len(fqdns) == 1 and fqdns[0].startswith(hostname):
            # Got only 1 fqdn
            return fqdns[0]
        else:
            # Got many fqdns, dont know how to choose one
            return hostname


def update(host=None):
    """
    Perform configuration registry update.
    Actually, it will check for registry last update time, and, if it will old enough,
    query @c genisys.yandex-team.ru service for last services configuration data.
    The data fetched for will be specific to the host name given.
    :param host:        Host name to be passed to the cloud storage.
                        Will be auto-detected in case of @c None.
    :return:            CMS configuration ID in case of configuration registry has been updated and @c None otherwise.

    :raise urllib2.HTTPError:   In case of any problems with cloud API call.
    :raise json.error:          In case of any problems with parsing API call result.
    :raise OSError:             In case of any problems with dumping the registry to the disk.
    """

    # Converts date string as per RFC 2822 into timestamp ..
    str2ts = lambda x: email.utils.mktime_tz(email.utils.parsedate_tz(x))
    # .. and vise versa ..
    ts2str = lambda x: email.utils.formatdate(x)

    actpath = registrydir().join(ACTIVE_FN)
    now = time.time()
    log = logger()
    cache = None
    try:
        atime = actpath.stat().mtime
    except py.error.Error:
        log.info('No active configuration found at %r', actpath.strpath)
        atime = None

    newpath = registrydir().join('new.yaml')
    oldpath = actpath.realpath()
    if not host:
        host = detect_hostname()

    mtime = None
    try:
        opener = six.moves.urllib.request.build_opener()
        opener.addheaders = [
            ('Accept-Encoding', 'gzip'),
            ('User-Agent', 'SkynetLibrary/1.0')
        ]
        if atime:
            try:
                cache = _load(actpath)
                mtime = cache.meta['mtime']
                ctime = str2ts(cache.meta['ctime'])
            except Exception as ex:
                log.info("Forcedly update registry because currently we have garbage we cannot load (%s)", ex)
            else:
                if atime and atime + CLOUD_RECHECK_TIMEOUT > now:
                    log.debug('Active configuration %r has been checked recently. Skipping update.', actpath.strpath)
                    return None
                elif ctime + CLOUD_FORCE_UPDATE < now:
                    log.info("Forcedly update registry because it is %s old.", misc.td2str(now - ctime))
                elif host != cache.meta['host']:
                    log.info(
                        "Forcedly update registry because hostname changed from %r to %r.",
                        cache.meta['host'], host
                    )
                else:
                    opener.addheaders.append(('If-Modified-Since', mtime))
        url = CLOUD_API_URL.format(host)
        log.debug('Requesting cloud API via %r', url)
        req = opener.open(url, timeout=CLOUD_API_CALL_TIMEOUT)
    except six.moves.urllib.error.HTTPError as ex:
        if ex.code == 304:  # "Not Modified"
            log.debug('Configuration has not been modified since %r.', mtime)
            # The configuration has not been changed since the last update. "touch" the registry file.
            actpath.setmtime()
            return None
        raise

    data = req.read()
    size = misc.size2str(len(data))
    if req.info().get('Content-Encoding') == 'gzip':
        data = gzip.GzipFile(fileobj=six.moves.cStringIO(data)).read()
        size += ' (%s)' % misc.size2str(len(data))
    data = msgpack.loads(data)
    mtime = req.headers.get('Last-Modified')
    path = registrydir().join('%d.yaml' % str2ts(mtime))
    log.info('Received %s. Dump new configuration data into %r', size, path.strpath)
    registrydir().ensure(dir=1)
    source = req.headers.get('x-genisys-hostname', None)
    conf_id = req.headers.get('X-Ya-Cms-Conf-Id', str2ts(mtime))

    with path.open('wb') as fp:
        yaml.safe_dump(
            {
                'meta': {
                    'host': host,
                    'mtime': mtime,
                    'conf_id': conf_id,
                    'ctime': ts2str(now),
                    'version': REGISTRY_STORAGE_VERSION,
                    'source': req.headers.get('X-Ya-Cms-Host', 'unknown') if not source else source,
                },
                'data': data
            },
            fp,
            default_flow_style=False,
        )

        # fsync config file to disk
        os.fsync(fp.fileno())

    log.debug('Set new configuration as active by symlinking to %r', actpath.strpath)
    if newpath.check(file=1):
        log.warning('Removing conflicting file at %r', newpath.strpath)
        newpath.remove()
    newpath.mksymlinkto(path.relto(newpath.dirpath()))
    newpath.move(actpath)
    if oldpath != path and oldpath != actpath and oldpath.check(exists=1):
        log.debug('Removing previous configuration at %r', oldpath.strpath)
        oldpath.remove()

    # Reset cache last access time to force future registry reload.
    if cache:
        cache.atime = 0
    return conf_id


def _fetch_registry_content(allow_update):
    """
    Internal function, an argument-less part of @c query() function, which was
    "Too large to analyse" as PyCharm says. Loads the registry data out from the disk storage..
    :param allow_update: If not set, outdated config will never be updated (currently it's updated on local
                         installations).
    :return:             Registry data dictionary.
    """
    def _load_data():
        return _load(registrydir().join(ACTIVE_FN)).data

    try:
        data = _load_data()
    except (RegistryStorageOutdated, Exception) as ex:
        outdated = isinstance(ex, RegistryStorageOutdated)

        log = logger()

        if allow_update:
            if outdated:
                log.info('Registry storage outdated and we allowed to update it, doing that now')
            else:
                log.warning('Got error while loading registry: %s, attempt to forcedly update', str(ex))

            update()
            data = _load_data()
        else:
            if outdated:
                log.warning('Registry storage outdated and we do not allowed to update it')
            else:
                log.warning('Got error while loading registry: %s', str(ex))
            raise

    confd = registrydir().join('conf.d')
    if confd.check(exists=1, dir=1):
        for fn in sorted(confd.listdir()):
            if fn.ext != '.yaml':
                continue

            try:
                data_override = _load(fn, meta=False).data
                data = _merge_recursive(data, data_override)
            except Exception as ex:
                sys.stderr.write('Unable to merge "%s": %s\n' % (fn, str(ex)))
                pass

    return data


def _dict_query(dct, path):
    for k in path.split('.') if path else []:
        dct = dct[k]
    return dct


def _merge_recursive(base, other):
    if not isinstance(other, dict):
        return other
    if not isinstance(base, dict):
        return copy.deepcopy(other)
    result = copy.deepcopy(base)
    for k, v in six.iteritems(other):
        if isinstance(result.get(k), dict):
            result[k] = _merge_recursive(result[k], v)
        else:
            result[k] = copy.deepcopy(v)
    return result


def _evaluate(data):
    # Recursively evaluate placeholders inplace
    queue = [data]
    while queue:
        next_queue = []
        for item in queue:
            if isinstance(item, dict):
                items = item.items()
            elif isinstance(item, list):
                items = enumerate(item)
            else:
                raise ValueError("cannot evaluate item type %r" % (type(item).__name__,))

            for k, v in items:
                if isinstance(v, (dict, list)):
                    next_queue.append(v)
                elif isinstance(v, six.string_types):
                    changed = False
                    while True:
                        match = RE_PLACEHOLDER.search(v)
                        if not match:
                            break
                        v = v[:match.start()] + _dict_query(data, match.group(1)) + v[match.end():]
                        changed = True
                    if changed:
                        item[k] = v

            queue = next_queue


def _convert_format(data, key):
    result = {}
    subsections = data['subsections']
    if subsections:
        for k, v in six.iteritems(data['subsections']):
            result.update(_convert_format(v, '{}.{}'.format(key, k) if key else k))
    elif data['config']:
        result.update({key: data['config']})

    return result


def query(
    section=None,
    subsection='config',
    base=None,
    overrides=None,
    as_dict=False,
    fix_keys=True,
    evaluate=True,
    allow_update=False,
    old=False
):
    """
    Perform __local__ registry query for given section and subsection.

    :param section:      Section name to be checked. ``skynet.services.srvmgr`` for example.
                         In case of @c None passed, the whole registry content will be returned,
                         @c subsection parameter will be ignored.
    :param subsection:   Sub-section of the queried section. ``config`` by default.
                         In case of @c None the whole section will be returned.
                         Can fetch nested dictionary be separating their keys with commas.
    :param base:         Dictionary of values to be considered as base result, its values will be
                         overrided by the ones from the fetched config.
    :param overrides:    Plain dictionary of values to be placed into result additionally
                         __relatively__ to the sub-section specified above, overriding
                         any existing data. Key can be comma-separated to represent inner dictionaries.
    :param as_dict:      Return configuration as plain dictionary instead of slotted class instance.
    :param fix_keys:     In case of slotted class instance requested, perform automatic transliteration
                         of unsafe key characters.
    :param evaluate:     Perform evaluation of string values' placeholders in form of ``${path}``.
                         Placeholders evaluation are __relative__ to the sub-section specified above.
    :param allow_update: Allow to update registry automatically (e.g. if outdated or not exists at all).
                         Note that you must have write access rights (usually it means you're root) to
                         update the config.
    :param old:          Use old registry.
                         Actually this is mostly for testing, cuz new format fully equals to the old one.
    :return:             Dictionary with the configuration data for the queried section and sub-section.

    :raise OSError:      In case of no local registry file exists or not readable.
    :raise KeyError:     In case of no such section or sub-section exists.
    """
    fmt = (
        lambda x: _dict2slotted(x, fix_keys=fix_keys),
        lambda x: x,
    )[int(as_dict)]

    try:
        if old:
            raise ImportError()

        from api.skycore import query_section
        from api.skycore.errors import ConfigNotReadyError

        # convert section name to the path
        path = []
        if section:
            path = section.split('.')

        # just get config from the Registry without pre-processing it, it will be done later
        try:
            data = query_section(
                path=path,
                base=None,
                overrides=None,
                as_dict=True,
                fix_keys=False
            )
        except ConfigNotReadyError:
            # there is no config file yet, try old one
            raise ImportError()

        if path:
            data = data['config']
        else:
            result = _convert_format(data, None)
            result = _merge_recursive(base, result)
            return fmt(result)

    except ImportError:
        # there is no skycore, it's strange, but let's try old way
        data = _fetch_registry_content(allow_update=allow_update)

        if not section:
            result = {k: v['data'] for k, v in six.iteritems(data)}
            result = _merge_recursive(base, result)
            return fmt(result)

        data = data[section]['data']

    data = _dict_query(data, subsection)

    # merge with base
    data = _merge_recursive(base, data)

    # Perform overrides merge
    if overrides:
        for k, v in six.iteritems(overrides):
            path = k.split('.')
            curr, prev = data, None
            for p in path[:-1]:
                prev, curr = curr, curr.get(p, None)
                if not isinstance(curr, dict):
                    curr = prev[p] = {}
            curr[path[-1]] = v

    # evaluate if required
    if evaluate and data:
        _evaluate(data)

    return fmt(data)
