import os
import sys
import stat
import socket
import subprocess
import threading
import time
import cPickle as pickle
import signal
import struct


class WriteError(Exception):
    pass


class TimeoutException(Exception):
    pass


# stolen from skynet code
def formatException(err):
    trace = [
        'Traceback (most recent call last):\n',
        '%s: %s\n' % (type(err).__name__, err)
        ]
    chainedTrace = getattr(err, '_traceback', None)
    if chainedTrace:
        trace[-1:-1] = chainedTrace[0 if len(trace) > 2 else 1:]
    return ''.join(trace)


class Message(object):
    __slots__ = ['__data']

    def __init__(self, data):
        self.__data = data

    @property
    def data(self):
        return self.__data

    @staticmethod
    def pop(stream=sys.stdin):
        return Message(pickle.load(stream))

    def push(self, stream=sys.stdout):
        pickle.dump(self.data, stream, pickle.HIGHEST_PROTOCOL)
        stream.flush()

    def getPushData(self):
        return pickle.dumps(self.data, pickle.HIGHEST_PROTOCOL)


# The class is for writing a queue of data into the stream.
# The push method is to append data into the queue.
# The writing is performed in a separate thread.
class ThreadWrite(object):
    __slots__ = ['__lock', '__list', '__file', '__state', '__thread']

    def __init__(self, writeFile=sys.stdout):
        self.__lock = threading.Condition()
        self.__list = []
        self.__file = writeFile
        self.__state = True
        self.__thread = threading.Thread(target=self.__threadedWrite)
        self.__thread.daemon = True
        self.__thread.start()

    def __threadedWrite(self):
        try:
            while True:
                with self.__lock:
                    while not self.__list:
                        self.__lock.wait()
                    job = self.__list
                    self.__list = []

                for item in job:
                    if item is None:
                        self.__state = False
                        self.__file.close()
                        return
                    else:
                        self.__file.write(item)
                    if __debug__:
                        if len(item) < 20:
                            log("written: " + item)
                self.__file.flush()
        except Exception as err:
            self.writeError(err)
            self.__state = False
        except:
            self.writeError()
            self.__state = False
        finally:
            try:
                self.__file.close()
            except:
                pass

    def writeError(self, err=None):
        pass

    # append raw data into the write queue
    # raw data is to be written as is
    # if data is None then stream will be closed
    def push(self, data):
        if not self.__state:
            raise WriteError("Write stream is closed")
        with self.__lock:
            self.__list.append(data)
            if len(self.__list) == 1:
                self.__lock.notify()

    def pushMessage(self, data):
        msg = Message(data)
        self.push(msg.getPushData())


class EndOfStream(Exception):
    pass


# Class is for messages that hosts exchange of.
# If the message is received from uplink of the execution tree
# (link to a command starter) then the host field contains a hostname
# of a host where the message should be delivered to.
# If the message is received from downlink then the host field constains
# a hostname of a message generator.
class HostMessage(object):
    __slots__ = ['__data', '__host', '__msgtype']

    def __init__(self, host, data, msgtype='stdout'):
        self.__host = host
        self.__msgtype = msgtype
        self.__data = data

    @property
    def host(self):
        return self.__host

    @property
    def msgtype(self):
        return self.__msgtype

    @property
    def data(self):
        return self.__data

    @staticmethod
    def pop(stream=sys.stdin):
        popObj = pickle.load(stream)
        if isinstance(popObj, list):
            (host, msgtype, data) = popObj
            return HostMessage(host, data, msgtype)
        elif isinstance(popObj, tuple):
            if __debug__:
                log("pop pack")
            header, popdata = popObj
            if header == "pack":
                packresult = []
                for item in popdata:
                    (host, msgtype, data) = pickle.loads(item)
                    msg = HostMessage(host, data, msgtype)
                    packresult.append(msg)
                return packresult

    @staticmethod
    def pushPack(pack, stream=sys.stdout):
        if __debug__:
            log("push pack")
        dataPack = [pmsg.getPushData() for pmsg in pack]
        pickle.dump(("pack", dataPack), stream, pickle.HIGHEST_PROTOCOL)
        stream.flush()
        if __debug__:
            log("%d items has been flushed to a stream" % len(dataPack))

    def push(self, stream=sys.stdout):
        pickle.dump([self.host, self.msgtype, self.data], stream,
                    pickle.HIGHEST_PROTOCOL)
        stream.flush()
        if __debug__:
            if type(self.data) in (str, unicode):
                log("%d has been flushed to a stream" % len(self.data))
            else:
                log("data has been flushed to a stream")

    def getPushData(self):
        return pickle.dumps([self.host, self.msgtype, self.data],
                            pickle.HIGHEST_PROTOCOL)


# Class for creating a ssh connection with a remote execution node.
class SSHConnect(ThreadWrite):
    __slots__ = ['__state', '__stateError', '__host', '__sp',
                 '__receiver', '__popThread']

    # __state is initially True. It is false after ssh process has finished.
    @property
    def isActive(self):
        return self.__state == 1

    @property
    def isClosed(self):
        return self.__state >= 2

    # ctx is a NodeBootstrapCtx object.
    # ctx.cmd defines a command to run on the remote host.
    # receiver is an ExecutionTreeNode object that receives an output
    # of the ssh process.
    def __init__(self, ctx, receiver):
        if __debug__:
            log("ssh session init for: " + ctx.host)
        self.__state = 0
        self.__stateError = None
        # target host for a ssh connection
        self.__host = ctx.host

        self.__sp = subprocess.Popen(
            ["ssh", "-a",
             "-o", "BatchMode=yes",
             "-o", "TCPKeepAlive=yes",
             "-o", "StrictHostKeyChecking=no",
             "-o", "CheckHostIP=no",
             "-o", "Compression=no",
             "-o", "LogLevel=ERROR"] +
            ctx.sshOptions +
            [ctx.host] + ctx.cmd,
            stdin=subprocess.PIPE, stdout=subprocess.PIPE,
            stderr=subprocess.PIPE, close_fds=True, bufsize=4093)
        if __debug__:
            log("returns from Popen in SSHConnection.__init__")
        ThreadWrite.__init__(self, self.__sp.stdin)
        try:
            # Push bootstrap initialization string to the started process.
            # This is the content of nomoresky.py
            ctx.push(self)
        except WriteError:
            pass

        self.__receiver = receiver
        # Creating a thread that reads stdout of the ssh process.
        self.__popThread = threading.Thread(target=self.__threadedPipeRead)
        self.__popThread.daemon = True
        self.__popThread.start()

    def writeError(self, err=None):
        if err is not None:
            self.__stateError = err
        self.__state = 2

    def __threadedPipeRead(self):
        try:
            # Check if the remote command has been started
            firstbyte = self.__sp.stdout.read(1)
            if len(firstbyte) == 0 or firstbyte != '?':
                # to avoid race condition the next line should be first
                self.__state = 2
                self.__receiver.linkError(self.__host)
                self.__receiver._push(
                    HostMessage(self.__host, 'not started', 'cmd'))
                raise EndOfStream()

            self.__state = 1
            if __debug__:
                log("host %s has been started " % self.__host)
            self.__receiver._started(self.__host)

            while True:
                msg = HostMessage.pop(self.__sp.stdout)
                if __debug__:
                    if isinstance(msg, HostMessage):
                        if type(msg.data) in (str, unicode):
                            log("%d bytes of data received from %s, stream %s" %
                                (len(msg.data), msg.host, msg.msgtype))
                        else:
                            log("data received from %s, stream %s" %
                                (msg.host, msg.msgtype))
                    else:
                        log("pack of data received")
                self.__receiver.pushThroughCache(msg)
        except (EndOfStream, EOFError):
            if __debug__:
                log("end of ssh stream for host %s" % self.__host)
            self.__state = 2
        except Exception as err:
            self.__sp.kill()
            self.__stateError = err
            self.__state = 2
            self.__receiver._push(
                HostMessage(self.__host, formatException(err), 'exception'))
        except:
            self.__sp.kill()
            self.__state = 2

        try:
            # FIXME: some deadlocks can occur if output of the ssh process
            # was big enough
            err = self.__sp.stderr.read(65536).strip()
            self.__sp.wait()
            if self.__sp.returncode == 255:
                if len(err) == 0:
                    err = "ssh silently exitted with error code"
                self.__receiver._push(
                    HostMessage(self.__host, err, 'ssh error'))
            elif __debug__ and len(err) != 0:
                self.__receiver._push(HostMessage(self.__host, err, 'error'))
        except:
            pass

        try:
            self.__sp.stdout.close()
            self.__sp.stderr.close()
        except:
            pass

        try:
            self.__receiver._done(self.__host)
        except:
            pass


class PayloadCtx(object):
    def __init__(self, flags):
        self.__flags = flags

    @property
    def flags(self):
        return self.__flags

    def popPayload(self, stream):
        if self.__flags is not None:
            if 'RESULT_IN_ONE_MSG' in self.__flags:
                return stream.read()
            elif 'RESULT_IN_PICKLE' in self.__flags:
                return pickle.load(stream)
            elif 'RESULT_IN_LINES' in self.__flags:
                return stream.readline()

        return os.read(stream.fileno(), 32768)


# Some values for launching python scripts
class PythonPayloadCtx(PayloadCtx):
    def __init__(self, payload, flags):
        super(PythonPayloadCtx, self).__init__(flags)
        self.__payload = payload

    # This is a command for subprocess.Popen
    @property
    def cmd(self):
        return ["python", "-O", "-c",
                "import sys\n" +
                "s = sys.stdin.read(%d)\n" % len(self.__payload) +
                "if len(s) != %d: exit(2)\n" % len(self.__payload) +
                "exec s in {'__name__': '__task__'}\n"]

    # push into stdin the initialization content for a starting python
    def push(self, writer):
        writer.push(self.__payload)


# Some values for launching arbitrary command.
# cmd is defined by user in argv of a started sdel script
class CmdPayloadCtx(PayloadCtx):
    def __init__(self, cmd, flags):
        super(CmdPayloadCtx, self).__init__(flags)
        self.__cmd = cmd

    @property
    def cmd(self):
        return self.__cmd

    # nothing to write into stdin for starting a program
    def push(self, writer):
        return


def preExecFunc():
    for sigNum in [signal.SIGPIPE, signal.SIGTERM, signal.SIGINT,
                   signal.SIGUSR1, signal.SIGUSR2, signal.SIGIO,
                   signal.SIGHUP, signal.SIGQUIT, signal.SIGABRT,
                   signal.SIGALRM, signal.SIGCHLD]:
        signal.signal(sigNum, signal.SIG_DFL)
    os.setsid()


# Class for launching payload process and communicating with it
class RunPayload(ThreadWrite):
    # ctx is a CmdPayloadCtx or PythonPayloadCtx (or similar) object.
    # receiver is an ExecutionTreeNode object that receives output.
    def __init__(self, host, ctx, receiver=None):
        if ctx is None:
            raise ValueError("ctx is None")

        self.__host = host
        self.__ctx = ctx

        self.__sp = subprocess.Popen(
            ctx.cmd, stdout=subprocess.PIPE, stdin=subprocess.PIPE,
            stderr=subprocess.PIPE, close_fds=True, bufsize=4096,
            preexec_fn=preExecFunc)
        if __debug__:
            log("payload launched")
        ThreadWrite.__init__(self, self.__sp.stdin)
        self.__ctx.push(self)

        self.__receiver = receiver
        if self.__receiver is not None:
            self.__challenge = 2
            self.__lock = threading.Lock()

            self.__thread = threading.Thread(
                target=self.__threadedPipeRead,
                args=(self.__sp.stdout, "stdout"))
            self.__thread.daemon = True
            self.__thread.start()

            self.__thread = threading.Thread(
                target=self.__threadedPipeRead,
                args=(self.__sp.stderr, "stderr"))
            self.__thread.daemon = True
            self.__thread.start()

    def __done(self):
        with self.__lock:
            self.__challenge -= 1
            if self.__challenge != 0:
                return

            self.__sp.wait()
            try:
                os.killpg(self.__sp.pid, signal.SIGKILL)
            except:
                pass
            msg = HostMessage(
                self.__host, self.__sp.returncode, 'exitcode')
            self.__receiver._push(msg)
            self.__receiver._done(self.__host)

    def __threadedPipeRead(self, pipe, msgtype):
        if __debug__:
            log("reading of " + msgtype + " has been started")
        try:
            # Wait until the full execution tree is created
            self.__receiver._waitForAllowExit()
            while True:
                output = self.__ctx.popPayload(pipe)
                if len(output) == 0:
                    if __debug__:
                        log(msgtype + " has been closed")
                    break
                if __debug__:
                    log("got %d bytes from %s" % (len(output), msgtype))
                msg = HostMessage(self.__host, output, msgtype)
                self.__receiver._push(msg)

        except Exception as err:
            try:
                msg = HostMessage(
                    self.__host, formatException(err), 'exception')
                self.__receiver._push(msg)
            except:
                pass
        self.__done()

    def kill(self):
        try:
            os.killpg(self.__sp.pid, signal.SIGCONT)
            os.killpg(self.__sp.pid, signal.SIGTERM)
            time.sleep(2)
            os.killpg(self.__sp.pid, signal.SIGKILL)
        except:
            pass


def getModuleSourceFilename():
    if __file__.endswith(".pyc") or __file__.endswith(".pyo"):
        return __file__[:-1]
    else:
        return __file__


# Context is for the process of creating of new nodes.
class NodeBootstrapCtx(object):
    # bootstrap is the content of nomoresky.py
    # host is where the child node is to be created
    # ctx is a dict to be push to the child node
    def __init__(self, host, bootstrap, ctx):
        self.__bootstrap = bootstrap
        self.__host = host
        self.__ctx = ctx
        self.__ctx["host"] = self.__host

    @property
    def sshOptions(self):
        options = []

        ctxOpts = []
        try:
            ctxOpts = self.__ctx['flags']['SSH_OPTIONS']
            ctxOpts = [x for x in ctxOpts if type(x) in (str, unicode)]
            for x in ctxOpts:
                options += ["-o", x]
        except:
            pass

        for defaultOption, defaultValue in (
            ("ConnectTimeout", "ConnectTimeout=1"),
            ("ConnectionAttempts", "ConnectionAttempts=1"),
            ("ServerAliveInterval", "ServerAliveInterval=1"),
            ("ServerAliveCountMax", "ServerAliveCountMax=2")
        ):
            if sum(1 for x in ctxOpts if x.startswith(defaultOption)) == 0:
                options.extend(["-o", defaultValue])

        try:
            options.extend(["-l", self.__ctx['flags']['SSH_USER']])
        except:
            pass

        return options

    # cmd is to be appended to ssh parameters
    @property
    def cmd(self):
        # FIXME: should check len(bootstrap) after reading
        return ["python"] +\
            (["-c"]
             if 'flags' in self.__ctx and 'DEBUG' in self.__ctx['flags']
             else ["-O", "-c"]) +\
            ["\"import sys\n" +
             "s = sys.stdin.read(%d);\n" % len(self.__bootstrap) +
             "if len(s) != %d: exit(2)\n" % len(self.__bootstrap) +
             "exec s in {'__name__': '__child__', 'bootstrap': s}\n\""]

    @property
    def host(self):
        return self.__host

    # pushing the context on the parent node.
    def push(self, writer):
        writer.push(self.__bootstrap)
        rawctx = pickle.dumps(self.__ctx, pickle.HIGHEST_PROTOCOL)
        writer.pushMessage(rawctx)

    # popping the context on the child node.
    @staticmethod
    def pop(bootstrap, stream=sys.stdin):
        msg = Message.pop()
        ctx = pickle.loads(msg.data)
        ctx["bootstrap"] = bootstrap
        return ExecutionTreeNode(ctx=ctx)


# Implements a node of the execution tree
class ExecutionTreeNode(object):
    CONNECTIONS_PER_NODE = 20

    def __init__(self, hosts=None, cmd=None, flags=None, ctx=None):
        if ctx is None:
            parent = True
            self.__parent = True
            self.__host = socket.getfqdn()
            self.__bootstrap = open(getModuleSourceFilename(), "r").read()
            self._flags = flags
            self.__cmd = cmd
            if self.__cmd is None:
                import __main__
                self.__payload = open(__main__.__file__, "r").read()
            else:
                self.__payload = None
        else:
            parent = False
            self.__parent = False
            self.__bootstrap = ctx.get('bootstrap', None)
            self.__payload = ctx.get('payload', None)
            self.__cmd = ctx.get('cmd', None)
            self.__host = ctx.get('host', socket.getfqdn())
            self._flags = ctx.get('flags', None)

        self.__conns = {}
        self.__forwards = {}
        self.__forwardedTo = {}
        self.__hostListToForward = []

        self._doneFlag = False
        self._doneLock = threading.Condition()
        self.__doneHosts = set()

        self._allowedToExit = False

        if not parent:
            self.__pushLock = threading.RLock()
            self.__pushQueue = []
            self.__pushQueueLock = threading.Condition()

        if not parent:
            if self.__payload is not None:
                pctx = PythonPayloadCtx(self.__payload, self._flags)
            elif self.__cmd is not None:
                pctx = CmdPayloadCtx(self.__cmd, self._flags)
            else:
                pctx = None

            try:
                self.__localDone = False
                self.__localTask = RunPayload(self.__host, pctx, receiver=self)
            except Exception as err:
                if __debug__:
                    log("run payload err: %s" % err)
                self.__localDone = True
                self.__localTask = None
        else:
            self.__localDone = True
            self.__localTask = None

        if not parent:
            self.__portLock = threading.RLock()
            self.__ports = []
            UnixStreamServer(self)
            self.__agentCache = {}

            self.__popperLock = threading.Condition()
            popper = threading.Thread(target=self.__threadedUplinkPopper)
            popper.daemon = True
            popper.start()

        self.addHosts(hosts)

    def addHosts(self, hosts):
        if hosts is not None:
            for host in hosts:
                self.addNewHost(host)
            if __debug__:
                log("all hosts has been dispatched")

    def linkError(self, host):
        if not self.__forwards.get(host, []):
            return
        for fhost in self.__forwards[host]:
            del self.__forwardedTo[fhost]
            self._push(HostMessage(self.__host, fhost, 'link failed'))
            self.__forwards[host] = []

    def __checkInAgentCache(self, msg):
        if msg.msgtype != 'agent':
            return False
        if len(msg.data) >= 30:
            return False
        if __debug__:
            log("try to intercept agent request")
        (port, data) = pickle.loads(msg.data)
        with self.__portLock:
            if data not in self.__agentCache:
                return False
            if isinstance(self.__agentCache[data], list):
                return False
            if __debug__:
                log("agent request is found in the cache")
            reply = pickle.dumps(
                (port, self.__agentCache[data]),
                pickle.HIGHEST_PROTOCOL)
            downMsg = HostMessage(msg.host, reply, 'agent')
            with self._doneLock:
                self._pushDown(downMsg)
                return True

    def pushThroughCache(self, msg):
        if isinstance(msg, HostMessage):
            if not self.__checkInAgentCache(msg):
                self._push(msg)
        else:
            msg = [item for item in msg
                   if not self.__checkInAgentCache(item)]
            if len(msg) > 0:
                self._push(msg)

    def _push(self, msg):
        with self.__pushQueueLock:
            if isinstance(msg, list):
                self.__pushQueue.extend(msg)
            else:
                self.__pushQueue.append(msg)
        with self.__pushLock:
            with self.__pushQueueLock:
                if len(self.__pushQueue) == 0:
                    return
                while True:
                    currLen = len(self.__pushQueue)
                    self.__pushQueueLock.wait(0.004)
                    if currLen == len(self.__pushQueue):
                        break
                pack = self.__pushQueue
                self.__pushQueue = []
            if len(pack) == 1:
                pack[0].push()
            else:
                HostMessage.pushPack(pack)

    def pushPort(self, data, replyTo):
        with self.__portLock:
            if len(data) < 10:
                if data in self.__agentCache:
                    if not isinstance(self.__agentCache[data], list):
                        if __debug__:
                            log("reply to agent request from the cache")
                        replyTo(self.__agentCache[data])
                    else:
                        if __debug__:
                            log("agent request is found in cache")
                        self.__agentCache[data].append(replyTo)
                    return
                else:
                    cacheKey = data
                    self.__agentCache[data] = []
            else:
                cacheKey = None

            choosed = None
            for port in range(len(self.__ports)):
                if self.__ports[port] is None:
                    choosed = port
                    self.__ports[port] = (replyTo, cacheKey)
                    break
            if choosed is None:
                choosed = len(self.__ports)
                self.__ports.append((replyTo, cacheKey))
        ldata = pickle.dumps((choosed, data), pickle.HIGHEST_PROTOCOL)
        msg = HostMessage(self.__host, ldata, 'agent')
        self._push(msg)

    def _started(self, host):
        with self._doneLock:
            self._push(HostMessage(host, 'started', 'cmd'))
            self._doneLock.notifyAll()

    def _pushAllowExit(self, host):
        if self.__conns[host].isClosed:
            return
        forwardMsg = HostMessage(host, 'allow exit', 'cmd')
        try:
            if __debug__:
                log("push allow exit to host: " + host)
            self.__conns[host].push(forwardMsg.getPushData())
        except WriteError:
            pass

    def _pushAllowExitToAllConns(self):
        for host in self.__conns:
            self._pushAllowExit(host)

    def allowExit(self):
        self._pushAllowExitToAllConns()

        with self._doneLock:
            self._doneLock.notifyAll()
            self._allowedToExit = True
            if __debug__:
                log("allow exit")
            if len(self.__doneHosts) == len(self.__conns):
                if self.__localDone:
                    self._doneFlag = True
                    if __debug__:
                        log("done = True")
                    self._doneLock.notifyAll()
                    self._hasBeenDone()

    def _waitForAllowExit(self):
        with self._doneLock:
            while not self._allowedToExit:
                self._doneLock.wait()

    def _hasBeenDone(self):
        pass

    def _done(self, host):
        with self._doneLock:
            if __debug__:
                log("done with " + host)
            if host in self.__conns:
                self.__doneHosts.add(host)
            elif host == self.__host:
                self.__localDone = True
            if len(self.__doneHosts) == len(self.__conns) and self.__localDone:
                if self._doneFlag:
                    return
                self._doneFlag = True
                if __debug__:
                    log("done = True")
                self._doneLock.notifyAll()
                self._hasBeenDone()

    def __insistDone(self):
        with self._doneLock:
            self._doneFlag = True
            self._doneLock.notifyAll()

    def waitForAllDone(self):
        if __debug__:
            log("wait for all done")
        with self._doneLock:
            if self._doneFlag:
                return
            while not self._doneFlag:
                try:
                    self._doneLock.wait()
                except Exception:
                    self._doneFlag = True
                    if __debug__:
                        log("done = True, by exception")
                    self._doneLock.nofityAll()
                    self._hasBeenDone()
                    raise

    def _forwardToHost(self, newhost):
        if __debug__:
            log("forward host to existing connection: " + str(newhost))
        activeConns = [
            (len(v), k) for k, v in self.__forwards.iteritems()
            if not self.__conns[k].isClosed]
        if len(activeConns) == 0:
            raise WriteError()
        if type(newhost) not in (set, frozenset, tuple, list):
            self.__hostListToForward.append(newhost)
            totalForwarded = sum(x[0] for x in activeConns)
            needForwarded = totalForwarded + len(self.__hostListToForward)
            forwardedPerNode =\
                (needForwarded - 1) / self.CONNECTIONS_PER_NODE + 1
            minCount, minhost = min(activeConns)
            newhost = self.__hostListToForward[:forwardedPerNode - minCount]
            del self.__hostListToForward[:forwardedPerNode - minCount]
        else:
            minhost = min(activeConns)[1]
        if __debug__:
            log(str(newhost) + " is being forwarded to " + minhost)
        forwardMsg = HostMessage(minhost, newhost, 'new host')
        self.__conns[minhost].push(forwardMsg.getPushData())
        if type(newhost) in (set, frozenset, tuple, list):
            for item in newhost:
                self.__forwards[minhost].append(item)
                self.__forwardedTo[item] = minhost
        else:
            self.__forwards[minhost].append(newhost)
            self.__forwardedTo[newhost] = minhost

    def _addNewHostUnderDoneLock(self, newhost):
        if (
            newhost == self.__host or
            newhost in self.__conns
        ):
            if __debug__:
                log("ignore new host: " + newhost)
            return

        if __debug__:
            log("is about to add new host: " + newhost)
        while True:
            if sum(1 for v in self.__conns.values() if not v.isClosed) < \
                    self.CONNECTIONS_PER_NODE:
                if __debug__:
                    log("add new connection for " + newhost)
                self.__forwards[newhost] = []

                ctx = {}
                if self.__payload is not None:
                    ctx['payload'] = self.__payload
                if self.__cmd is not None:
                    ctx['cmd'] = self.__cmd
                if self._flags is not None:
                    ctx['flags'] = self._flags

                self.__conns[newhost] = SSHConnect(
                    NodeBootstrapCtx(newhost, self.__bootstrap, ctx), self)
                if self._allowedToExit:
                    self._pushAllowExit(newhost)
                break
            else:
                try:
                    self._forwardToHost(newhost)
                except WriteError:
                    continue
                break

    def addNewHost(self, newhost):
        with self._doneLock:
            if not self._doneFlag:
                if type(newhost) in (list, tuple, set, frozenset):
                    self.__hostListToForward += list(newhost)
                    while len(self.__hostListToForward) > 0:
                        newhost = self.__hostListToForward.pop(-1)
                        self._addNewHostUnderDoneLock(newhost)
                else:
                    self._addNewHostUnderDoneLock(newhost)
            else:
                if type(newhost) in (list, tuple, set, frozenset):
                    for item in newhost:
                        self.linkError(newhost)
                else:
                    self.linkError(newhost)

    def __treatUplinkMessage(self, msg):
        if msg.msgtype == 'new host':
            self.addNewHost(msg.data)
        elif msg.msgtype == 'stdin':
            if self.__localTask is not None:
                if __debug__:
                    log("push message into stdin of local task")
                self.__localTask.push(msg.data)
        elif msg.msgtype == 'cmd':
            if msg.data == 'allow exit':
                self.allowExit()
            elif msg.data == 'close stdin':
                if self.__localTask is not None:
                    # None means close stream
                    self.__localTask.push(None)
        elif msg.msgtype == 'agent':
            (port, data) = pickle.loads(msg.data)
            (replyTo, cacheKey) = self.__ports[port]
            replyTo(data)
            replyToList = []
            with self.__portLock:
                self.__ports[port] = None
                if cacheKey is not None:
                    if __debug__:
                        log("put reply into the cache")
                    replyToList = self.__agentCache[cacheKey]
                    self.__agentCache[cacheKey] = data
            for replyTo in replyToList:
                replyTo(data)

    def __threadedUplinkPopper(self):
        try:
            while True:
                try:
                    msg = HostMessage.pop()
                except EOFError:
                    # EOFError is sent by pickle
                    break
                if __debug__:
                    log("readed from uplink: " +
                        str((msg.host, msg.msgtype, msg.data)))
                if msg.host == self.__host:
                    self.__treatUplinkMessage(msg)
                    continue
                if msg.host == "*":
                    self.__treatUplinkMessage(msg)
                    self._pushDown(msg)
                else:
                    self._pushDown(msg)
        except Exception as err:
            if __debug__:
                log("type of exception: %s" % str(dir(err)))
            if __debug__:
                log("exception while popping from uplink: %s" % err)
        except:
            if __debug__:
                log("exception while popping from uplink")
        try:
            if __debug__:
                log("connection with uplink is lost, let's suicide")
            self.__localTask.kill()
            if __debug__:
                log("killed")
        finally:
            self.__insistDone()

    def _pushDown(self, msg):
        if msg.host == "*":
            if __debug__:
                log("push down broadcast")
            for host, conn in self.__conns.iteritems():
                if __debug__:
                    log("push down to " + host)
                try:
                    conn.push(msg.getPushData())
                except WriteError:
                    pass
            return
        if msg.host in self.__conns:
            forwardToHost = msg.host
        elif msg.host in self.__forwardedTo:
            forwardToHost = self.__forwardedTo[msg.host]
        else:
            if __debug__:
                log("unknown message %s, %s" % (msg.msgtype, msg.data))
            self._push(HostMessage(msg.host, 'No such host', 'error'))
            return
        try:
            self.__conns[forwardToHost].push(msg.getPushData())
        except WriteError:
            self._push(HostMessage(
                    msg.host, 'Link to host has been closed', 'error'))

    @property
    def isActive(self):
        return not self._doneFlag


class ServiceConnection(threading.Thread):
    def __init__(self, sock, receiver):
        super(ServiceConnection, self).__init__()
        self.__sock = sock.makefile()
        self.__receiver = receiver
        self.daemon = True
        self.start()

    def run(self):
        try:
            if __debug__:
                log("new connection service loop: %d" %
                    self.__sock.fileno())
            while True:
                data = self.__sock.read(4)
                if len(data) < 4:
                    raise Exception
                (dataLen, ) = struct.unpack("!I", data[:4])
                data += self.__sock.read(dataLen)
                if len(data) < 4 + dataLen:
                    raise Exception
                if __debug__:
                    log("agent message received: %d"
                        % self.__sock.fileno())
                self.__receiver.pushPort(data, self.push)
        except:
            self.__sock.close()

    def push(self, msg):
        if self.__sock.closed:
            return
        try:
            if __debug__:
                log("write agent reply with %d bytes length" %
                    len(msg))
            self.__sock.write(msg)
            self.__sock.flush()
        except:
            self.__sock.close()


class UnixStreamServer(threading.Thread):
    def __init__(self, receiver):
        super(UnixStreamServer, self).__init__()
        for path in [".nomoresky-agent-forward-%d" % os.getpid(),
                     "/run/shm/.nomoresky-agent-forward-%d" % os.getpid(),
                     "/tmp/.nomoresky-agent-forward-%d" % os.getpid(),
                     "/var/tmp/.nomoresky-agent-forward-%d" % os.getpid(),
                     None]:
            if path is None:
                raise Exception("create agent forwarder")
            try:
                os.unlink(path)
            except OSError:
                pass
            try:
                self.__path = path
                oldmask = os.umask(~stat.S_IRWXU)
                self.__sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
                self.__sock.bind(path)
                self.__sock.listen(65535)
                os.putenv("SSH_AUTH_SOCK", path)
                os.umask(oldmask)
                break
            except:
                pass

        self.__receiver = receiver
        self.daemon = True
        self.start()

    def run(self):
        try:
            if __debug__:
                log("run agent forwarder loop")
            while True:
                newsock, address = self.__sock.accept()
                if __debug__:
                    log("accepted agent forward")
                ServiceConnection(newsock, self.__receiver)
        except:
            pass

    def __del__(self):
        try:
            os.unlink(self.__path)
        except:
            pass


def log(line):
    try:
        log = open("nomoresky.log", "a")
        print >> log, os.getpid(), time.time(), line
        log.close()
    except:
        pass


if __name__ == "__child__":
    # tell uplink about successful start
    sys.stdout.write('?')
    sys.stdout.flush()
    if __debug__:
        log("started, yeah!")
    try:
        r = NodeBootstrapCtx.pop(bootstrap)  # noqa
        r.waitForAllDone()
    except SystemExit:
        raise
    except Exception as err:
        myhostname = socket.getfqdn()
        HostMessage(myhostname, formatException(err), 'exception').push()
