import time
import gevent


class SimpleTaskPoll(object):
    def __init__(self, tasks=None):
        self.tasks = set()
        self.finishedTasks = []
        if tasks is None:
            tasks = []
        for task in tasks:
            self.add(task)

    def add(self, task):
        if task not in self.tasks:
            self.tasks.add(task)
            task.connect("destroyed", self.__onFinished, dict(obj=task))

    def remove(self, task):
        if task in self.tasks:
            task.disconnect("destroyed", self.__onFinished)
            self.tasks.discard(task)

    def __onFinished(self, obj):
        self.finishedTasks.append(obj)
        self.tasks.discard(obj)

    def finished(self):
        return len(self.tasks) == 0

    def takeFinished(self):
        res = self.finishedTasks
        self.finishedTasks = []
        return res

    def poll(self, timeout=None):
        startTime = time.time()
        while (timeout is None) or (time.time() - startTime < timeout):
            if (not len(self.tasks)) or (self.finishedTasks and timeout is None):
                return self.takeFinished()
            gevent.sleep()
        return self.takeFinished()


class CopyTaskPoll(object):
    def __init__(self, tasks=None):
        self.tasks = set()
        self.finished_tasks = []
        if tasks is None:
            tasks = []
        for task in tasks:
            self.add(task)

    def add(self, task):
        if task not in self.tasks:
            self.tasks.add(task)

    def remove(self, task):
        if task in self.tasks:
            self.tasks.discard(task)

    def __on_finished(self, obj):
        self.finished_tasks.append(obj)
        self.tasks.discard(obj)

    def __inner_poll(self, timeout):
        start = time.time()
        end = start
        while end - start <= timeout:
            finished = [obj for obj in self.tasks if obj.finished()]
            for obj in finished:
                self.__on_finished(obj)
            if finished:
                return
            end = time.time()
            sleep_time = max(0, min(timeout - end + start, 0.1))
            time.sleep(sleep_time)

    def finished(self):
        self.__inner_poll(0)
        return len(self.tasks) == 0

    def takeFinished(self):
        res = self.finished_tasks
        self.finished_tasks = []
        return res

    def poll(self, timeout=None):
        start = time.time()
        while (timeout is None) or (time.time() - start < timeout):
            self.__inner_poll(0.1 if timeout is None else max(0, min(timeout - time.time() + start, 0.1)))
            if (not len(self.tasks)) or (self.finished_tasks and timeout is None):
                return self.takeFinished()
        return self.takeFinished()
