"""Wrapper around the porto to reconnect and retry calls if porto has been restarted"""

from __future__ import absolute_import

import functools
import socket
import time
import types


class PortoAdapter(object):
    def __init__(self, slave, conn=None):
        self.__slave = slave

    @staticmethod
    def _is_porto(obj):
        if 'porto.api' not in getattr(obj, '__module__', ''):
            return False
        if not hasattr(obj, '__class__'):
            return False
        if '_RPC' == obj.__class__.__name__:
            return False
        return True

    def __wrap(self, item):
        if callable(item):
            return self.__wrapper(item)
        elif self._is_porto(item):
            return PortoAdapter(item)
        elif isinstance(item, (list, types.GeneratorType)):
            return [
                (PortoAdapter(x) if self._is_porto(x) else x)
                for x in item
            ]
        else:
            return item

    def __wrapper(self, cb):
        from porto.exceptions import SocketError, SocketTimeout

        @functools.wraps(cb)
        def _fn(*args, **kwargs):
            retries = 3
            while True:
                try:
                    result = cb(*args, **kwargs)
                    return self.__wrap(result)
                except (socket.error, SocketError, SocketTimeout):
                    if retries < 0:
                        raise
                    retries -= 1
                    rpc = getattr(self.__slave, 'rpc', getattr(self.__slave, 'conn', None))
                    rpc.disconnect()
                    time.sleep(1.0)
                    try:
                        rpc.connect()
                    except (socket.error, SocketError, SocketTimeout):
                        pass

        return _fn

    def __getattr__(self, item):
        res = getattr(self.__slave, item)
        return self.__wrap(res)
