import errno
import hashlib
import socket
import time

from kernel.util.sys.user import getUserName
from kernel.util.net.socketstream import SocketStream

import msgpack


def make_message(user,
                 slot,
                 configuration_id='',
                 extra_env=None,
                 unset_env=None,
                 command=None,
                 interactive_cmd=False,
                 api_mode=False,
                 streaming=True,
                 inactivity_timeout=3600,
                 files=None,
                 ):
    token = {
        'acc_user': getUserName(),
        'acc_host': socket.getfqdn(),
        'user': user,
        'slot': slot,
        'configuration_id': configuration_id,
        'ctime': time.time(),
    }

    msg = {
        'token': token,
        'user': token['user'],
        'slot': token['slot'],
        'configuration_id': token['configuration_id'],
        'signs': [],
        'extra_env': extra_env,
        'unset_env': unset_env,
        'command': command,
        'interactive_cmd': interactive_cmd,
        'api_mode': api_mode,
        'streaming': streaming,
        'telnet_timeout': inactivity_timeout,
        'files': files,
    }

    return msg


def hash_message(message):
    token = message['token']

    # just to be sure
    token['user'] = message['user']
    token['slot'] = message['slot']
    token['configuration_id'] = message.get('configuration_id', '')

    return hash_token(token)


def str_token(token):
    return ''.join(
        map(str,
            (token[item]
             for item in ('acc_user', 'acc_host', 'user', 'slot', 'configuration_id', 'ctime')))
    )


def hash_token(token):
    """
    Token required structure:
        {
            'acc_user': str_accounting_username,
            'acc_host': str_accounting_hostname,
            'user': str_user_name,
            'slot': str_slotname,
            'ctime': float_unix_timestamp,
        }
    """
    return hashlib.md5(str_token(token)).digest()


def sign_message(message, signer):
    md5 = hash_message(message)

    signs = list(signer.sign(md5))
    message['signs'] = signs


def establish_connection(host, port, init_message):
    addrs = socket.getaddrinfo(host, port, 0, socket.SOCK_STREAM)
    conn_errors = []
    sock = None
    for family, socktype, proto, canonname, sockaddr in addrs:
        if family not in (socket.AF_INET, socket.AF_INET6):
            continue

        try:
            sock = socket.socket(family)
            sock.connect(sockaddr)
        except EnvironmentError as e:
            conn_errors.append('%s: %s' % (sockaddr[0], e))
            sock = None
        else:
            break

    if sock is None:
        if conn_errors:
            raise EnvironmentError(errno.EHOSTUNREACH, "Failed to connect to `%s`:\n%s" % (
                host,
                '\n'.join(conn_errors)
            ))
        else:
            raise EnvironmentError(errno.EHOSTUNREACH, "Failed to resolve `%s`, cannot connect" % (host,))

    msg = msgpack.dumps(init_message)
    try:
        SocketStream(sock).writeBEStr(msg)
    except EnvironmentError as e:
        raise EnvironmentError(errno.EFAULT, "Failed to connect to `%s`: %s" % (host, e))

    return sock
