import time
import sys

from skycmd import Cmd
from .. import pbModule


class CmdCopier(Cmd):
    related_services = 'copier|skybone'

    def __init__(self):
        super(CmdCopier, self).__init__()

        self.parser.add_option(
            '--traceback', dest='traceback', action='store_true', default=False,
            help='Dont hide known error (e.g. for ResourceNotAvailable or CopierError) tracebacks'
        )
        self.parser.add_option(
            '--opts', dest='opts', default=None,
            help='Extra option to pass to copier api'
        )

        self._lastProgressFunc = None
        self._progressBar = None

    @property
    def opts(self):
        opts_raw = self.options.opts
        if not opts_raw:
            return {}

        import yaml
        opts = yaml.load(opts_raw)

        if not isinstance(opts, dict):
            print >> sys.stderr, 'options didnt converted to dictionary, ignored (got %r)' % (opts, )
            return {}

        return opts

    def connecting_skybone_progress_meter(self):
        pb = pbModule()
        if not self._progressBar:
            widgets = [
                '  [', pb.AnimatedMarker(), ']', ' Connecting skybone... ',
            ]
            self._progressBar = pb.ProgressBar(widgets=widgets, maxval=pb.UnknownLength).start()
        self._progressBar.update(self._progressBar.currval + 1)

    def create_handle_progress_meter(self, done=False):
        pb = pbModule()
        if not done:
            widgets = ['  [ ] Creating handle...']
        else:
            widgets = ['  [|] Create handle']
        self._progressBar = pb.ProgressBar(widgets=widgets, maxval=pb.UnknownLength).start()

    def create_subproc_connect_master_progress_meter(self, done):
        pb = pbModule()
        if not done:
            widgets = ['  [ ] Connecting skybone daemon...']
        else:
            widgets = ['  [|] Connect skybone daemon']

        self._progressBar = pb.ProgressBar(widgets=widgets, maxval=pb.UnknownLength).start()

    def get_resource_head_progress_meter(self):
        pb = pbModule()
        if not self._progressBar:
            widgets = [
                '  [', pb.AnimatedMarker(), ']', ' Get resource head (info)... ',
            ]
            self._progressBar = pb.ProgressBar(widgets=widgets, maxval=pb.UnknownLength).start()
        self._progressBar.update(self._progressBar.currval + 1)

    def lock_paths_progress_meter(self, total, locked):
        pb = pbModule()
        if not self._progressBar:
            lock_label = pb.FormatLabel('Lock files (%(value)d of %(max)d done)')

            widgets = [
                '  [', pb.AnimatedMarker(), ']', ' ', lock_label, ' [', pb.Timer(), '] ',
                pb.Bar(marker='='), pb.Percentage(), '  '
            ]
            self._progressBar = pb.ProgressBar(widgets=widgets, maxval=total).start()
        self._progressBar.update(locked)

    def spawn_downloader_progress_meter(self, done):
        pb = pbModule()
        if not self._progressBar:
            widgets = [
                '  [', pb.AnimatedMarker(), ']', ' Spawning new downloader process... ',
            ]
            self._progressBar = pb.ProgressBar(widgets=widgets, maxval=pb.UnknownLength).start()
        self._progressBar.update(self._progressBar.currval + 1)

    @staticmethod
    def analyze_error(err):
        import api.copier  # NOQA
        import py

        code = 1
        known_errors = (
            (api.copier.ApiConnectionError, 3),
            (api.copier.ApiError, 4),
            (api.copier.Timeout, 5),
            (api.copier.ResourceNotAvailable, 6),
            (api.copier.ResourceDownloadError, 7),
            (api.copier.ResourceNotAllowedByNetwork, 8),
            (api.copier.FilesystemError, 10),
            (api.copier.CopierError, 2),
            (py.error.EACCES, 2),
            (py.error.EPERM, 2),
        )

        for knownError, knownErrorCode in known_errors:
            # noinspection PyUnusedLocal
            if isinstance(err, knownError):
                code = knownErrorCode
                break

        return code

    def print_error(self, err, code):
        if self.options.traceback:
            from kernel.util.errors import formatException
            print >> sys.stderr, ' Error '.center(70, '=')
            print >> sys.stderr, formatException()
        else:
            print >> sys.stderr, 'Error: {0}: {1}'.format(err.__class__.__name__, err)

        if code == 3:
            print >> sys.stderr, 'Hint: check whether copier service is running on this machine'
            return code
        elif code != 1:
            return code
        else:
            print >> sys.stderr, (
                'Read about known possible errors here: '
                'http://wiki.yandex-team.ru/Skynet/KnownErrors'
            )
            return code

    def progress_meter(self, obj):
        fmt = getattr(self.options, 'progress_format', 'human')

        if fmt == 'json':
            try:
                import simplejson as json
            except ImportError:
                import json

            if isinstance(obj, dict):
                sys.stderr.write(json.dumps(obj) + '\n')
                sys.stderr.flush()
            else:
                dct = {}

                if hasattr(obj, '__slots__'):
                    for key in obj.__slots__:
                        dct[key] = getattr(obj, key)
                    if hasattr(obj, 'stage'):
                        dct['stage'] = obj.stage

                elif hasattr(obj, '__dict__'):
                    for key in obj.__dict__:
                        if not key.startswith('_'):
                            dct[key] = getattr(obj, key)

                else:
                    return
                sys.stderr.write(json.dumps(dct) + '\n')
                sys.stderr.flush()

            return

        func = None
        if not obj:
            func = None
        elif not isinstance(obj, dict):
            if obj.stage == 'connecting':
                func = self.connecting_skybone_progress_meter
                args = ()
            elif obj.stage == 'create_handle':
                func = self.create_handle_progress_meter
                args = (obj.done, )
            elif obj.stage == 'subproc_connect_master':
                func = self.create_subproc_connect_master_progress_meter
                args = (obj.done, )
            elif obj.stage == 'get_resource':
                func = self.download_progress_meter
                args = (obj.total_bytes, obj.done_bytes, obj.extra)
            elif obj.stage == 'get_resource_head':
                func = self.get_resource_head_progress_meter
                args = ()
            elif obj.stage == 'lock_paths':
                func = self.lock_paths_progress_meter
                args = (obj.total_count, obj.locked_count)
            elif obj.stage == 'spawn_downloader':
                func = self.spawn_downloader_progress_meter
                args = (obj.done, )
            elif obj.stage == 'hashing':
                func = self.hashing_progress_meter
                args = (obj.total_bytes, obj.hashed_bytes)

        last = self._lastProgressFunc
        if func != last and last and self._progressBar:
            self._progressBar.finish()
            self._progressBar = None
        if func:
            func(*args)
        self._lastProgressFunc = func

    def connect_progress(self, version):
        import threading

        if version != 1:
            return threading.Event()

        flag = threading.Event()

        class CnPrgrs(object):
            stage = 'connecting'

        self.progress_meter(CnPrgrs)

        def _connecting_progress():
            try:
                while not flag.isSet():
                    self.progress_meter(CnPrgrs)
                    time.sleep(0.1)
            except:
                pass

        thr = threading.Thread(target=_connecting_progress)
        thr.daemon = True
        thr.start()

        return flag

    def create_handle_progress(self, version, done=False):
        isdone = done

        class CreateHandleProgress(object):
            stage = 'create_handle'
            done = isdone

        self.progress_meter(CreateHandleProgress)

    def _add_progress_args(self):
        self.parser.add_option(
            '-p', '--progress', dest='progress', action='store_true', default=None,
            help='Show the progress bar.'
        )
        self.parser.add_option(
            '--progress-format', dest='progress_format', action='store', choices=('human', 'json'),
            default='human', help='Progress format. "human" and "json" are good values'
        )
        self.parser.add_option(
            '--progress-version', dest='progress_version', action='store', type=int, default=1,
            help='Progress version. As of now only version 1 is supported'
        )
        self.parser.add_option(
            '--progress-report-freq', dest='progress_report_freq', action='store', type=int, default=None,
            help='Max progress report frequency in milliseconds'
        )

    def _handle_progress(
        self, handle, timeout=None,
        freq=0, version=1
    ):
        if freq:
            prev_progress = (0, None)
        else:
            prev_progress = None

        progress = None

        for progress in handle.iter(timeout=timeout, state_version=version):
            now_ms = int(time.time() * 1000)

            if prev_progress is not None:
                if (
                    prev_progress[1] is None or
                    progress.stage != prev_progress[1].stage or
                    now_ms - prev_progress[0] > freq
                ):
                    prev_progress = (now_ms, progress)
                    self.progress_meter(progress)
            else:
                self.progress_meter(progress)

        # Put final progress
        if prev_progress is not None and progress is not None:
            self.progress_meter(progress)

        self.progress_meter(None)
