import errno
import sys

from . import CmdCopier


class CmdFiles(CmdCopier):
    usage = '\n'.join([
        '<resid>',
        'Example: sky files rbtorrent:12f92a268d3326e4ba4259c94213850f7875c36f',
    ])
    mode = 'files'
    description = 'list copier resource files'

    def __init__(self):
        super(CmdFiles, self).__init__()

        self.parser.add_option(
            '-t', '--timeout', dest='timeout', default=None,
            help='Maximum timeout for download operation'
        )
        self.parser.add_option(
            '-f', '--full', dest='full', action='store_true', default=False,
            help='Show also directories and symlinks.'
        )

        self._add_progress_args()

        self.parser.add_option(
            '-P', '--priority', dest='priority', action='store', default='Normal',
            choices=('Idle', 'Low', 'BelowNormal', 'Normal', 'AboveNormal', 'High', 'RealTime'),
            help='Choose priority for transfer (not all transports supported)'
        )
        self.parser.add_option(
            '-N', '--network', dest='network', action='store', default=None,
            choices=('Auto', 'Backbone', 'Fastbone'),
            help='Choose preferred network to use (Auto, Fatstbone or Backbone)'
        )
        self.parser.add_option(
            '--json', dest='json', action='store_true', default=False,
            help='Force output to be jsonified.'
        )
        self.parser.add_option(
            '--ls', dest='lsoutput', action='store_true', default=False,
            help='Output result in "ls -l" human-readable format.'
        )
        self.parser.add_option(
            '--lsh', dest='lsoutputh', action='store_true', default=False,
            help='Output result in "ls -lh" human-readable format.'
        )
        self.parser.add_option(
            '--lsS', dest='lsoutputSort', action='store_true', default=False,
            help='Output like --ls, but also sort by size.'
        )
        self.parser.add_option(
            '--lshS', dest='lsoutputhSort', action='store_true', default=False,
            help='Output like --lsh, but also sort by size.'
        )

    def run(self):
        import api.copier

        if len(self.argv) != 1:
            print >> sys.stderr, 'Error: expected 1 resource id as parameter\n'
            self.help()
            return 1

        if self.options.timeout is not None:
            try:
                self.options.timeout = float(self.options.timeout)
                assert self.options.timeout >= 0.0
            except (ValueError, AssertionError):
                print >> sys.stderr, 'Invalid argument for timeout (-t): got {0!r}'.format(self.options.timeout)
                return 1

        # See commends in CmdGet for this
        timeout = self.options.timeout or None

        priority = getattr(api.copier.Priority, self.options.priority)
        if self.options.network:
            network = getattr(api.copier.Network, self.options.network)
        else:
            network = None

        resid = self.argv[0]
        if self.options.json:
            assert not (
                self.options.lsoutput or self.options.lsoutputh or
                self.options.lsoutputSort or self.options.lsoutputhSort
            ), '--ls* options cant be used together with --json'

        try:
            if self.options.progress:
                flag = self.connect_progress(version=self.options.progress_version)

            handle = api.copier.Copier().handle(resid)

            handle_list = handle.list(priority=priority, network=network, **self.opts)

            if self.options.progress:
                flag.set()
                self._handle_progress(
                    handle_list, timeout,
                    freq=self.options.progress_report_freq,
                    version=self.options.progress_version
                )

            filesList = handle_list.wait(timeout=timeout).files()

            if not self.options.full:
                filesList = [item for item in filesList if item['type'] == 'file']
        except Exception as err:
            if self._progressBar:
                sys.stderr.write('\n')
            code = self.analyze_error(err)
            return self.print_error(err, code)

        def _humanSize(size):
            if size >= 1024 * 1024 * 1024:
                sizeStr = '{0:.2f}G'.format(float(size) / 1024 / 1024 / 1024, )
            elif size >= 1024 * 1024:
                sizeStr = '{0:.2f}M'.format(float(size) / 1024 / 1024, )
            elif size >= 1024:
                sizeStr = '{0:.2f}K'.format(float(size) / 1024, )
            else:
                sizeStr = '{0:d}'.format(size, )

            return sizeStr

        if (
            self.options.lsoutput or self.options.lsoutputh or
            self.options.lsoutputSort or self.options.lsoutputhSort
        ):
            resourceSize = sum([x['size'] for x in filesList if x['type'] == 'file'])
            if self.options.lsoutputh or self.options.lsoutputhSort:
                resourceSizeStr = _humanSize(resourceSize)
            else:
                resourceSizeStr = str(resourceSize)

            print('Resource {0!r} (total items {1:d}, size {2}):'.format(resid, len(filesList), resourceSizeStr))

            if self.options.lsoutputSort or self.options.lsoutputhSort:
                filesList = sorted(filesList, key=lambda x: x.get('size', 0))
            else:
                filesList = sorted(filesList, key=lambda x: x['name'])

            for item in filesList:
                if item['type'] == 'file':
                    mode = '-rwx' if item['executable'] else '-rw-'
                elif item['type'] == 'dir':
                    mode = 'drwx'
                elif item['type'] == 'symlink':
                    mode = 'lrwx'
                else:
                    assert False, 'Invalid item type: {0!r}'.format(item, )

                user = 'skynet'
                group = 'skynet'
                size = int(item.get('size', 0))
                month = '---'
                day = '--'
                time_ = '--:--'
                path = item['name']

                if self.options.lsoutputh or self.options.lsoutputhSort:
                    sizeStr = '{0:>8s}'.format(_humanSize(size), )
                else:
                    sizeStr = '{0:11d}'.format(size, )

                print '{0:s} {1:<6s} {2:<6s} {3:s} {4:>3s} {5:>2s} {6:>5s} {7:s}'.format(
                    mode, user, group, sizeStr, month, day, time_, path
                )

        else:
            if self.options.json:
                try:
                    import simplejson as json
                except ImportError:
                    json = None

                if json is None:
                    import json
                buf = json.dumps(filesList)
            else:
                buf = filesList
            try:
                print(buf)
            except IOError as e:
                # Stdout closed (e.g. if piped somewhere),
                # not a problem to report
                if e.errno == errno.EPIPE:
                    pass
                else:
                    raise
