import sys

from kernel.util.misc import size2str

from . import CmdCopier, pbModule


class CmdGet(CmdCopier):
    """
    <resid>

    Examples:
        Download resource. Process may hang for a long period of time if the network is congested.
            sky get <resid>

        Download resource with timeout. Process will wait up to given amount of seconds.
            sky get -t <seconds> <resid>

        Download into given directory. The given directory must be write-accessible for
        other users. If the directory's subtree does not exists it will be created.
            sky get -d /var/tmp -w <resid>
            sky get -d /var/tmp/new_dir_will_be_created -w <resid>
            sky get -d /var/tmp/a/b/c <resid> # a/b/c will be created

        Download resource creating files from the current user. -w option must always be
        specified for settings correct permissions on files after downloading.
            sky get -u <resid>

        Download resource and attempt to deduplicate files. If there are same files by same
        owner on this machine somethere (even in resource itself) -- copier will make
        hardlinks or symlinks depending on option instead of copying. This works only in
        user (-u/--user) mode.
            sky get -u -D Hardlink <resid>

        Download resource with progress bar
            sky get -u -p <resid>
    """
    usage = __doc__
    mode = 'get'
    description = 'download resource to the localhost using copier'

    def __init__(self):
        super(CmdGet, self).__init__()

        import optparse

        self.parser.add_option(
            '-d', '--dir', dest='dir', action='store', default=None,
            help='Target directory for download. By default cwd is used.'
        )
        self.parser.add_option(
            '-t', '--timeout', dest='timeout', default=None,
            help='Maximum timeout for download operation.\n'
                 ' Will be ignored if -w options is not specified'
        )
        self.parser.add_option(
            '-u', '--user', dest='user', action='store_true', default=True,
            help='[set by default] Attempt to download file with permissions of current user.'
        )

        self._add_progress_args()

        self.parser.add_option(
            '-P', '--priority', dest='priority', action='store', default='Normal',
            choices=('Idle', 'Low', 'BelowNormal', 'Normal', 'AboveNormal', 'High', 'RealTime'),
            help='Choose priority for transfer (not all transports supported)'
        )
        self.parser.add_option(
            '-N', '--network', dest='network', action='store', default=None,
            choices=('Auto', 'Backbone', 'Fastbone'),
            help='Choose preferred network to use (Auto, Fastbone or Backbone)'
        )
        self.parser.add_option(
            '-D', '--deduplicate', dest='deduplicate', default='No',
            choices=('No', 'Hardlink', 'Symlink'),
            help='Try to deduplicate files if they are already present on this machine'
        )
        self.parser.add_option(
            '--max-dl-speed',
            help='Max dl speed (bytes or mbps e.g. "10M (10Mbytes) or 10mbps (10 Mbits)")'
        )
        self.parser.add_option(
            '--max-ul-speed',
            help='Max ul speed (bytes or mbps e.g. "10M (10Mbytes) or 10mbps (10 Mbits)")'
        )
        self.parser.add_option(
            '--tag',
            help='Mark download with specified tag. Used for fwmarks, stats, etc.'
        )

        # Left for compatibility
        self.parser.add_option(
            '-w', '--wait', dest='wait', action='store_true', default=True,
            help=optparse.SUPPRESS_HELP
        )

        self.parser.add_option(
            '--sign-key', action='store', default=None,
            help='Sign connections with specified private key'
        )

    def download_progress_meter(self, total_bytes, done_bytes, state):
        if total_bytes == 0:
            return

        pb = pbModule()

        for st in (
            'hardlinked', 'symlinked', 'copied', 'torrents_done',
            'copying', 'torrents_queued', 'torrents_waiting', 'torrents_checking',
            'torrents_downloading'
        ):
            if st not in state:
                state[st] = -1

        if not self._progressBar:
            class StateLabel(pb.FormatLabel):
                def set_state(self, state):
                    self.state = state
                    for key, value in self.state.items():
                        if isinstance(value, dict):
                            for subkey, subvalue in value.iteritems():
                                self.state[key + '_' + subkey] = subvalue

                def update(self, pbar):
                    return self.format % self.state

            size_label = pb.FormatLabel('[%(done)s]')
            size_label.mapping['done'] = ('currval', size2str)

            counts_label = StateLabel(
                '['
                'h:%(hardlinked)d '
                's:%(symlinked)d '
                'c:%(copied)d '
                't:%(torrents_done)d '
                'cc:%(copying)d '
                'tq:%(torrents_queued)d '
                'tw:%(torrents_waiting)d '
                'tc:%(torrents_checking)d '
                'td:%(torrents_downloading)d'
                ']'
            )
            counts_label.set_state(state)

            widgets = [
                '  [', pb.AnimatedMarker(), ']', ' Downloading ', counts_label, ' ', size_label,
                ' [', pb.FileTransferSpeed(), '] ',
                pb.Bar(marker='='), ' [', pb.ETA(), '] ', pb.Percentage(), '  '
            ]
            self._progressBar = pb.ProgressBar(widgets=widgets, maxval=total_bytes, poll=0.1).start()

        self._progressBar.widgets[4].set_state(state)
        self._progressBar.update(done_bytes)

    def run(self):
        import api.copier

        if len(self.argv) != 1:
            print >> sys.stderr, 'Error: Invalid number of arguments.\nSee -h for more info'
            return 1

        if self.options.deduplicate != 'No' and not self.options.user:
            print >> sys.stderr, \
                'Error: When using -D/--deduplicate with hardlink/symlink modes, ' \
                '-u option must always be specified.\nSee -h for more info'
            return 1

        resid = self.argv[0]

        if self.options.timeout is not None:
            try:
                self.options.timeout = float(self.options.timeout)
                assert self.options.timeout >= 0.0
            except Exception:
                print >> sys.stderr, (
                    'Error: Invalid argument for timeout (-t): got {0!r}'.format(
                        self.options.timeout
                    )
                )
                return 1

        # We will ask for results, but with very small timeout (0.1 secs) if
        # we dont want wait at all. This is needed to catch early errors from remote job.
        # If timeout specified -- we convert 0 to None, because if somebody wants async operation
        # he will specify nowait instead :).
        # Timeout values:
        #  - 0: nonblock mode -- try get result and return immidiately if not yet available
        #  - None: wait untill resource will be available
        #  > 0: wait this amount of seconds to complete
        timeout = (self.options.timeout or None)

        priority = getattr(api.copier.Priority, self.options.priority)
        if self.options.network:
            network = getattr(api.copier.Network, self.options.network)
        else:
            network = None
        deduplicate = getattr(api.copier.Deduplicate, self.options.deduplicate)

        try:
            if self.options.progress:
                self.create_handle_progress(version=self.options.progress_version)

            handle = api.copier.Copier().handle(resid)

            if self.options.progress:
                self.create_handle_progress(version=self.options.progress_version, done=True)

            handle_get = handle.get(
                dest=self.options.dir,
                user=self.options.user,
                priority=priority,
                network=network,
                deduplicate=deduplicate,
                max_dl_speed=self.options.max_dl_speed,
                max_ul_speed=self.options.max_ul_speed,
                tag=self.options.tag,
                sign=self.options.sign_key,
                **self.opts
            )

            if self.options.progress:
                self._handle_progress(
                    handle_get, timeout,
                    freq=self.options.progress_report_freq,
                    version=self.options.progress_version
                )

            handle_get.wait(timeout=timeout)
        except Exception as err:
            if self._progressBar:
                sys.stderr.write('\n')
            code = self.analyze_error(err)
            return self.print_error(err, code)

        return 0
