# -*- coding: utf-8 -*-
try:
    __import__('pkg_resources').require('requests')
except BaseException:
    pass

import msgpack
import six

from collections import defaultdict
from .helpers import cleanHostsDict


DEFAULT_REQUEST_TIMEOUT = 15


class CloudResolver(object):
    def __init__(self, cloud_source, timeout=DEFAULT_REQUEST_TIMEOUT):
        self.url = '{}/?term='.format(cloud_source)
        self.timeout = timeout

    def resolveHosts(self, command):
        """
        Получить список хостов из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: set
        :return: множество найденных доменных имён
        """
        resolved_hosts, _ = self._do_request(command)
        return set(resolved_hosts)

    def resolveShards(self, command):
        """
        Получить список шардов из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: dict
        :return: словарь, где ключами являются имена машин, а
                 значениями — множества, содержащие строки с именами шардов на соответствующей машине
        """
        resolved_instances = self.resolveInstances(command)
        for host, value in six.iteritems(resolved_instances):
            resolved_instances[host] = set(map(lambda x: x[0], value)) - set(['none'])
        return cleanHostsDict(resolved_instances)

    def resolveInstances(self, command):
        """
        Получить список инстансов базового поиска из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: dict
        :return: словарь, где ключами являются имена машин, а
                 значениями — множества, содержащие пары вида (шард, инстанс)
        """
        result = defaultdict(set)
        _, resolved_instances = self._do_request(command)

        for host, d in six.iteritems(resolved_instances):
            for shard, instance in six.iteritems(d):
                result[host].add((instance, shard))

        return cleanHostsDict(result)

    def _do_request(self, command):
        import requests
        r = requests.get(self.url+command, timeout=self.timeout)

        hosts, instances = msgpack.loads(r.content)

        return hosts, instances
