import socket
from itertools import cycle

import six


def hostsDictsIntersection(d1, d2):
    result = {}
    hosts = set(d1) & set(d2)
    for host in hosts:
        value = d1[host] & d2[host]
        if value:
            result[host] = value

    return result


def hostsDictsUnion(d1, d2):
    result = {}
    hosts = set(d1) | set(d2)
    for host in hosts:
        value = d1.get(host, set()) | d2.get(host, set())
        if value:
            result[host] = value

    return result


def check_host_fqdn(name):
    errno_noname = {
        _ for _ in (
            getattr(socket, name, None)
            for name in ('EAI_NONAME', 'EAI_NODATA')
        )
        if _ is not None
    }

    ex = None
    for i in range(3):
        try:
            addrs = [addr[4][0] for addr in socket.getaddrinfo(name, None, 0, socket.SOCK_DGRAM, 0)]
        except socket.error as e:
            ex = e
            if ex.errno in errno_noname:
                # Dont retry NONAME errors, just raise as is
                raise
            # Force continue searching
            # print('RETRY 1:%d %s for %s' % (i, ex, name))
            continue
        # Stop searching, everything ok
        break
    else:
        # Will happen only if "Force continue searching" block was executed
        raise ex

    if not addrs:
        raise socket.gaierror('no address associated with hostname')

    for addr in addrs:
        for i in range(3):
            try:
                name = socket.gethostbyaddr(addr)[0]
            except socket.herror:
                # Dont make more retries for h_error
                break
            except socket.error as ex:
                if ex.errno in errno_noname:
                    # Dont make more retries if dns returned NONAME
                    break

                # Retry DNS errors
                # print('RETRY 2:%d %s for %s for %s' % (i, ex, addr, name))
                continue

            if name == 'any.yandex.ru':
                raise socket.gaierror(socket.EAI_NONAME, 'unknown host (resolved to any.yandex.ru)')

            return name

    # We dont found any reverse record for thoose ips
    # So, return first one
    return addrs[0]


def cleanHostsDict(hostDict):
    return dict([(k, v) for k, v in six.iteritems(hostDict) if v])


def getAlignedKwargs(listKeysListsPairs):
    """ getAlignedKwargs(listKeysListsPairs) -> listKwargs

    Returns aligned list of kwargs for requests to CMS.

    Example:
    >>>getAlignedKwargs([('conf', set(['A', 'B'])), ('shardTagName', set(['x']))])
    [{'conf': 'A', 'shardTagName': 'x'}, {'conf': 'B', 'shardTagName': 'x'}]
    """

    count = max(map(len, map(lambda x: x[1], listKeysListsPairs)))
    listKeysListsPairs = filter(lambda x: True if x[1] else False, listKeysListsPairs)
    listKeysListsPairs = map(lambda x: (x[0], cycle(x[1])), listKeysListsPairs)

    result = []
    for _ in six.moves.xrange(count):
        result.append(dict([(x, y.next())for x, y in listKeysListsPairs]))

    return result


def hostToFQDN(host):
    if '.' not in host:
        return host + '.yandex.ru'
    return host


def instancesToHostsDict(instances):
    hostsDict = {}
    for i in instances:
        hostsDict.setdefault(hostToFQDN(i.host), set()).add((i.shard, i.name))
    return hostsDict
