from api.conductor import Conductor
from api.gencfg import GenCfg, GenCfgNotFoundError, GenCfgHttpError

# FIXME: remove as soon as MacOS version updated to new API
try:
    from api.gencfg import GenCfgBadRequest
except ImportError:
    # old version of GenCfg API does not provide this exception
    GenCfgBadRequest = GenCfgHttpError

try:
    from api.walle import Walle
except ImportError:
    Walle = None

try:
    from api.hq import HQResolver, HQServiceNotFoundError
except ImportError:
    # Skynet before 16.1.0 may not have this module
    HQResolver = None
    HQServiceNotFoundError = None

try:
    from api.qloud_dns import QloudDNS
except ImportError:
    # Skynet before 15.21.x may not have this module
    QloudDNS = None

try:
    from api.netmon import NetmonResolver
except ImportError:
    # Skynet before 16.3.x may not have this module
    NetmonResolver = None

try:
    from api.samogon import SamogonResolver
except ImportError:
    # Skynet before 16.7.x may not have this module
    SamogonResolver = None

try:
    from api.yp import YPResolver
except ImportError:
    # Skynet before 16.10.x may not have this module
    YPResolver = None

try:
    from api.qloud_dns import QloudHostResolver
except ImportError:
    # Skynet before 15.21.x may not have this module
    QloudHostResolver = None


class HostSource(object):
    def __init__(
        self,
        conductor=None,
        gencfg=None,
        hq=None,
        qloud_dns=None,
        walle=None,
        netmon=None,
        samogon=None,
        yp=None,
        qloud=None,
        yp_use_sd_resolver=None,
        use_service_resolve_policy=None,
    ):
        self.conductor_resolver = conductor or Conductor()
        self.gencfg_resolver = gencfg or GenCfg()
        try:
            self.hq_resolver = hq or (HQResolver(use_service_resolve_policy=use_service_resolve_policy) if HQResolver else None)
        except TypeError:  # no support for use_service_resolve_policy
            self.hq_resolver = hq or (HQResolver() if HQResolver else None)
        self.qloud_dns_resolver = qloud_dns or (QloudDNS() if QloudDNS else None)
        self.walle_resolver = walle or (Walle() if Walle else None)
        self.netmon_resolver = netmon or (NetmonResolver() if NetmonResolver else None)
        self.samogon_resolver = samogon or (SamogonResolver() if SamogonResolver else None)
        try:
            self.yp_resolver = yp or (YPResolver(use_sd_resolver=yp_use_sd_resolver) if YPResolver else None)
        except TypeError:  # no support for use_sd_resolver
            self.yp_resolver = yp or (YPResolver() if YPResolver else None)

        self.qloud_resolver = qloud or (QloudHostResolver() if QloudHostResolver else None)

        self._host_sources = {
            'K': self.conductor_resolver.hosts_by_group,
            'k': self.conductor_resolver.hosts_by_tag,
            'P': self.conductor_resolver.hosts_by_project,
            'D': self.conductor_resolver.hosts_by_dc,
            'G': self._resolve_gencfg,
            'W': self.walle_resolver.get_hosts if self.walle_resolver else self._dummy_resolve,
            'w': self.walle_resolver.get_hosts_by_tag if self.walle_resolver else self._dummy_resolve,
            't': self._resolve_switch if self.walle_resolver and hasattr(
                self.walle_resolver, 'get_hosts_by_switch') else self._dummy_resolve,
            'f': self._resolve_family if self.hq_resolver else self._dummy_resolve,
            'd': self.netmon_resolver.get_hosts_by_dc if self.netmon_resolver else self._dummy_resolve,
            'l': self.netmon_resolver.get_hosts_by_line if self.netmon_resolver else self._dummy_resolve,
            'C': self._resolve_conf if self.hq_resolver else self._dummy_resolve,
            'M': self._resolve_mtn,
            'm': self._resolve_slave_mtn,
            'Q': self.qloud_dns_resolver.resolve_hosts if self.qloud_dns_resolver else self._dummy_resolve,
            'q': self._resolve_qloud if self.qloud_resolver else self._dummy_resolve,
            'z': self._resolve_samogon if self.samogon_resolver else self._dummy_resolve,
            'Y': self.yp_resolver.get_pods if self.yp_resolver else self._dummy_resolve,
            'p': self.yp_resolver.get_pods_ex if self.yp_resolver and hasattr(self.yp_resolver,
                                                                              'get_pods_ex') else self._dummy_resolve,
            'b': self.yp_resolver.get_boxes if self.yp_resolver and hasattr(self.yp_resolver,
                                                                            'get_boxes') else self._dummy_resolve,
            # keep old CMS prefix for backward compatibility (return empty set always)
            'I': self._dummy_resolve,
            'S': self._dummy_resolve,
            's': self._dummy_resolve
        }
        self._cache = {}

    @staticmethod
    def _dummy_resolve(_):
        return []

    def _resolve_gencfg(self, gc):
        group, tags = self._process_tags(gc)
        # do we have configuration?
        parts = group.split(':')
        return self.gencfg_resolver.get_hosts(parts[0], parts[1] if len(parts) > 1 else None, tags)

    def _resolve_conf(self, c):
        conf, tags = self._process_tags(c)
        pos = conf.rfind('-')
        f = conf[0:pos] if pos > 0 else conf
        return self.hq_resolver.get_hosts(f, conf, tags)

    def _resolve_mtn(self, mtn):
        group, tags = self._process_tags(mtn)
        # do we have configuration?
        parts = group.split(':')

        if self.hq_resolver:
            # add simple heuristics: upper case -> GenCfg, else -> HQ
            if parts[0] == parts[0].upper():
                # try GenCfg first
                resolver = self.gencfg_resolver
                fallback_resolver = self.hq_resolver
            else:
                # try HQ first
                resolver = self.hq_resolver
                fallback_resolver = self.gencfg_resolver
        else:
            resolver = self.gencfg_resolver
            fallback_resolver = None

        try:
            result = resolver.get_mtns(parts[0], parts[1] if len(parts) > 1 else None, tags)
        except (GenCfgNotFoundError, GenCfgBadRequest, GenCfgHttpError, HQServiceNotFoundError) as ex:
            if fallback_resolver:
                result = fallback_resolver.get_mtns(parts[0], parts[1] if len(parts) > 1 else None, tags)
            else:
                raise ex

        return result

    def _resolve_slave_mtn(self, mtn):
        group, tags = self._process_tags(mtn)
        # do we have configuration?
        parts = group.split(':')

        return self.gencfg_resolver.get_slave_mtns(parts[0], parts[1] if len(parts) > 1 else None, tags)

    # q@ext.hw_seg:state?dc
    def _resolve_qloud(self, q):
        rest, dc = self._process_location(q)
        items = rest.split(':')
        state = items[1] if len(items) > 1 else None
        parts = items[0].split('.')
        installation = parts[0]
        hw_segment = parts[1] if len(parts) > 1 else None

        return self.qloud_resolver.resolve_hosts(installation, hw_segment, state, dc)

    def _resolve_samogon(self, s):
        service, location = self._process_location(s)
        items = service.split(':')
        try:
            return self.samogon_resolver.get_hosts(items[0], items[1] if len(items) > 1 else None, location)
        except TypeError:
            # old API without location
            return self.samogon_resolver.get_hosts(items[0], items[1] if len(items) > 1 else None)

    def _resolve_family(self, f):
        family, tags = self._process_tags(f)
        items = family.split(':')
        return self.hq_resolver.get_hosts(items[0], items[1] if len(items) > 1 else None, tags)

    def _resolve_switch(self, s):
        items = s.split(':')
        return self.walle_resolver.get_hosts_by_switch(items[0], int(items[1]) if len(items) > 1 else None)

    def get_hosts_by_groups(self, groups):
        result = []

        for g in groups:
            # check cache first
            hosts = self._cache.get(str(g), None)

            if hosts is None:
                # print str(g), 'MISS'
                # cache missed
                func = self._host_sources[g.prefix]
                if func is None:
                    raise Exception("Impossible happened: HostSource got unknown prefix '{0}'".format(g.prefix))

                hosts = func(g.name)
                self._cache[str(g)] = hosts
            # else:
            #     print str(g), 'HIT'

            result.extend(hosts)

        return set(result)

    @staticmethod
    def _process_tags(value):
        # do we have tag?
        parts = value.split('?')
        return parts[0], set(parts[1].split(',')) if len(parts) > 1 else None

    @staticmethod
    def _process_location(value):
        # do we have location?
        parts = value.split('?')
        return parts[0], parts[1] if len(parts) > 1 else None

    getHostsByGroups = get_hosts_by_groups
