try:
    __import__('pkg_resources').require('ply')
except Exception:
    pass

import ply
import ply.lex
import ply.yacc

from .expression import Dnf, Clause, Literal
from .errors import HrSyntaxError


def formatSyntaxError(value, lexpos, data, error_context=15):
    lpos = 0
    rpos = len(data)
    posarrow = '{0}^'.format(' ' * lexpos)

    if len(data) > error_context * 2:
        if lexpos > error_context:
            lpos = lexpos - error_context
        else:
            rpos = error_context * 2
        if len(data) - lexpos > error_context:
            rpos = lexpos + error_context
        else:
            lpos = len(data) - error_context * 2

        data = data[lpos:rpos] + '...'
        if lpos != 0:
            data = '...' + data
            lpos -= 3
        posarrow = '{0}^'.format(' ' * (lexpos - lpos))

    return 'Unexpected symbol "{0}" in position {1}:\n{2}\n{3}'.format(value, lexpos, data, posarrow)


class Lexer(object):
    tokens = ('PREFIX', 'NAME', 'MINUS', 'TIMES', 'LBRACKET', 'RBRACKET')

    t_NAME = r'[-=_.,:?*a-zA-Z0-9]+'
    t_LBRACKET = r'\[|\('
    t_RBRACKET = r'\]|\)'

    t_ignore = ' \t\n'

    def __init__(self, **kwargs):
        self.lexer = ply.lex.lex(module=self, debug=0, **kwargs)

    def t_PREFIX(self, t):
        r"""(
              (C|CONF|conf)
            | (h|host)
            | (I|INSTANCETAG|ITAG|instancetag|itag)
            | (S|SHARDTAG|STAG|shardtag|stag)
            | (s|shard)
            | (K|CONDUCTORGROUP|conductorgroup)
            | (k|CONDUCTORTAG|conductortag)
            | (P|CONDUCTORPROJECT|conductorproject)
            | (D|CONDUCTORDC|conductordc)
            | (G|GENCFG|gencfg)
            | (W|WALLE|walle)
            | (w|WALLETAG|walletag)
            | (t|TOR|tor)
            | (f|HQ|family)
            | (Q|QLOUD|qloud)
            | (q|QLOUDHW|qloudhw)
            | (M|MTN|mtn)
            | (m|SLAVEMTN|slavemtn)
            | (d|dc)
            | (l|line)
            | (z|SAMOGON|SG|samogon|sg)
            | (Y|YP|yp)
            | (p|PODS|pods)
            | (b|BOXES|boxes)
        )[@]"""

        t.value = t.value[:-1]
        t.value = {
            'conf': 'C',
            'host': 'h',
            'instancetag': 'I',
            'itag': 'I',
            'shardtag': 'S',
            'stag': 'S',
            'shard': 's',
            'conductorgroup': 'K',
            'conductortag': 'k',
            'conductorproject': 'P',
            'conductordc': 'D',
            'gencfg': 'G',
            'walle': 'W',
            'walletag': 'w',
            'tor': 't',
            'family': 'f',
            'hq': 'f',
            'qloud': 'Q',
            'qloudhw': 'q',
            'mtn': 'M',
            'slavemtn': 'm',
            'dc': 'd',
            'line': 'l',
            'samogon': 'z',
            'sg': 'z',
            'yp': 'Y',
            'pods': 'p',
            'boxes': 'b',
        }.get(t.value.lower(), t.value)

        return t

    def t_MINUS(self, t):
        r""" \- """
        return t

    def t_TIMES(self, t):
        r""" \. """
        return t

    def t_error(self, t):
        raise HrSyntaxError(formatSyntaxError(t.value[0], t.lexer.lexpos, self.data))

    def token(self):
        return self.lexer.token()

    def input(self, data):
        self.data = data
        if data.strip() == '':
            raise HrSyntaxError('Empty input data')
        self.lexer.input(data)
        self.lexer.lineno = 1


class Parser(object):
    def __init__(self, lexer=None, error_context=15, **kwargs):
        self.lexer = lexer or Lexer(**kwargs)
        self.tokens = self.lexer.tokens
        # disable precompilation (write_tables=0), because our grammer is simple (<10 rules)
        self.parser = ply.yacc.yacc(module=self, debug=0, write_tables=0)
        self.error_context = error_context

    def parse(self, data, **kwargs):
        self.data = data
        self.lexer.input(data.strip())
        result = self.parser.parse(**kwargs)
        if result is None:
            raise HrSyntaxError()
        if not result.has_prefix:
            result.set_prefix()
        return result

    def p_expression_plus(self, p):
        """expression : expression term"""
        p[0] = p[1] + p[2]

    def p_expression_minus(self, p):
        """expression : expression MINUS term"""
        p[0] = p[1] - p[3]

    def p_expression_term(self, p):
        """expression : term"""
        p[0] = p[1]

    def p_term_times(self, p):
        """term : term TIMES factor"""
        p[0] = p[1] * p[3]

    def p_term_factor(self, p):
        """term : factor"""
        p[0] = p[1]

    def p_factor_prefix(self, p):
        """factor : PREFIX factor"""
        p[2].set_prefix(p[1])
        p[0] = p[2]

    def p_factor_expression(self, p):
        """factor : LBRACKET expression RBRACKET"""
        p[0] = p[2]

    def p_factor_name(self, p):
        """factor : NAME"""
        literal = Literal(p[1])
        clause = Clause(set([literal]))
        p[0] = Dnf(set([clause]))

    def p_error(self, p):
        if p:
            raise HrSyntaxError(formatSyntaxError(p.value, p.lexpos, self.data))
        raise HrSyntaxError('Unexpected end of command')
