# -*- coding: utf-8 -*-
from __future__ import absolute_import

from threading import Lock

from .resolverbase import ResolverBase
from .helpers import check_host_fqdn
from .hostsource import HostSource


class Resolver(ResolverBase):
    """
    Due to limitations in ply/yacc used for parsing:
        - Make Resolver just before using it::

            Resolver().resolveHosts(...)

        - Do not resolve in parallel in threads, use lock at least::

            from threading import Lock
            l = Lock

            def _resolve_in_threads(*args):
                with l:
                    return Resolver().resolveHosts(*args)
    """

    lock = Lock()

    def __init__(self, check_hosts=True, no_tag=False, use_hq=False, yp_use_sd_resolver=None, use_service_resolve_policy=None):
        with self.lock:  # ply/yacc initialization is not threadsafe
            super(Resolver, self).__init__(
                HostSource(yp_use_sd_resolver=yp_use_sd_resolver, use_service_resolve_policy=use_service_resolve_policy),
                check_host_fqdn if check_hosts else lambda x: x,
                no_tag,
                use_service_resolve_policy
            )

    def resolveHosts(self, command):
        """
        Получить список хостов из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: set
        :return: множество найденных доменных имён
        """
        return super(Resolver, self).resolve_hosts(command)

    def resolveShards(self, command):
        """
        Получить список шардов из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: dict
        :return: словарь, где ключами являются имена машин, а
                 значениями — множества, содержащие строки с именами шардов на соответствующей машине
        """
        return super(Resolver, self).resolve_shards(command)

    def resolveInstances(self, command):
        """
        Получить список инстансов базового поиска из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: dict
        :return: словарь, где ключами являются имена машин, а
                 значениями — множества, содержащие пары вида (шард, инстанс)
        """
        return super(Resolver, self).resolve_instances(command)

    def resolveSlots(self, command):
        """
        Получить список слотов из выражения в синтаксисе калькулятора Блинова.

        :param str command: выражение для разбора
        :rtype: dict
        :return: множества, содержащие тройки вида (хост, слот, конфигурация)
        """
        return super(Resolver, self).resolve_slots(command)
