from __future__ import print_function
from unittest import TestCase

from library.sky.hostresolver.hostsource import HostSource
from api.gencfg import GenCfgNotFoundError


class MockGroup(object):
    def __init__(self, prefix, name):
        self.prefix, self.name = prefix, name

    def __str__(self):
        return '{}@{}'.format(self.prefix, self.name)


GROUP_HOSTS = {
    'K1': 'h1.ru',
    'k1': 'h2.ru',
    'P1': 'h3.ru',
    'D1': 'h4.ru',
    'G1': 'h5.ru',
    'G2': 'h6.ru',
    'M1': 'h7.ru',
    'M2': 'h8.ru',
    'MHQ1': 'h9.ru',
    'MHQ2': 'h10.ru',
    'FHQ1': 'h11.ru',
    'FHQ2': 'h12.ru',
    'FHQ3': 'h13.ru',
    'Q1': 'h14.ru',
    'W1': 'h15.ru',
    'WTAG1': 'h16.ru',
    'd1': 'h17.ru',
    'l1': 'h18.ru',
    's1': 'h19.ru',
    's2': 'h20.ru',
    'YP': 'h21.ru',
}


class MockConductor(object):
    def hosts_by_group(self, group):
        assert group == 'K1'
        return [GROUP_HOSTS[group]]

    def hosts_by_tag(self, tag):
        assert tag == 'k1'
        return [GROUP_HOSTS[tag]]

    def hosts_by_project(self, project):
        assert project == 'P1'
        return [GROUP_HOSTS[project]]

    def hosts_by_dc(self, dc):
        assert dc == 'D1'
        return [GROUP_HOSTS[dc]]


class MockGencfg(object):
    def get_hosts(self, group, tag, instance_tags=None):
        assert group == 'G1' or (group == 'G2', tag == 'tag')
        return [GROUP_HOSTS[group]]

    def get_mtns(self, group, tag, instance_tags=None):
        if 'MHQ' in group:
            print('try HQ')
            raise GenCfgNotFoundError('ya.ru')
        assert group == 'M1' or (group == 'M2', tag == 'tag')
        return [GROUP_HOSTS[group]]


class MockHq(object):
    def get_hosts(self, group, rev, tags=None):
        assert group == 'FHQ1' or (group == 'FHQ2', rev == '123456') or (group == 'FHQ3', rev == '123456')
        return [GROUP_HOSTS[group]]

    def get_mtns(self, group, rev, tags=None):
        assert group == 'MHQ1' or (group == 'MHQ2', rev == '123456')
        return [GROUP_HOSTS[group]]


class MockQloudDns(object):
    def resolve_hosts(self, group):
        assert group == 'Q1'
        return [GROUP_HOSTS[group]]


class MockQloud(object):
    def resolve_hosts(self, installation, hardware_segment, state, dc):
        assert installation == 'EXT'
        return [GROUP_HOSTS[installation]]


class MockWalle(object):
    def get_hosts(self, group):
        assert group == 'W1'
        return [GROUP_HOSTS[group]]

    def get_hosts_by_tag(self, tag):
        assert tag == 'WTAG1'
        return [GROUP_HOSTS[tag]]


class MockNetmon(object):
    def get_hosts_by_dc(self, dc):
        assert dc == 'd1'
        return [GROUP_HOSTS[dc]]

    def get_hosts_by_line(self, line):
        assert line == 'l1'
        return [GROUP_HOSTS[line]]


class MockSamogon(object):
    def get_hosts(self, group, servant):
        assert group == 's1' or (group == 's2', servant == 'servant')
        return [GROUP_HOSTS[group]]


class MockYP(object):
    def get_pods(self, podset):
        assert podset == 'YP'
        return [GROUP_HOSTS[podset]]


class HostGroupsTest(TestCase):
    def test_groups_basic(self):
        hs = HostSource(
            conductor=MockConductor(),
            gencfg=MockGencfg(),
            hq=MockHq(),
            qloud_dns=MockQloudDns(),
            walle=MockWalle(),
            netmon=MockNetmon(),
            samogon=MockSamogon(),
            yp=MockYP(),
            qloud=MockQloud()
        )
        gset = [
            MockGroup('K', 'K1'),
            MockGroup('k', 'k1'),
            MockGroup('P', 'P1'),
            MockGroup('D', 'D1'),
            MockGroup('G', 'G1'),
            MockGroup('G', 'G2:tag'),
            MockGroup('M', 'M1'),
            MockGroup('M', 'M2:tag'),
            MockGroup('M', 'MHQ1'),
            MockGroup('M', 'MHQ2:123456'),
            MockGroup('f', 'FHQ1'),
            MockGroup('f', 'FHQ2:123456'),
            MockGroup('C', 'FHQ3-123456'),
            MockGroup('Q', 'Q1'),
            MockGroup('W', 'W1'),
            MockGroup('w', 'WTAG1'),
            MockGroup('d', 'd1'),
            MockGroup('l', 'l1'),
            MockGroup('z', 's1'),
            MockGroup('z', 's2:servant'),
            MockGroup('Y', 'YP'),
        ]

        hosts = hs.get_hosts_by_groups(gset)
        print(hosts)
        assert len(hosts) == len(GROUP_HOSTS)
