import unittest
from library.sky.hostresolver import parser
from library.sky.hostresolver.errors import HrSyntaxError


class TestParserFunctions(unittest.TestCase):
    def setUp(self):
        self.parser = parser.Parser()

    def test_simple1(self):
        self.assertEqual(str(self.parser.parse('ws35-035')), 'h@ws35-035')

    def test_simple2(self):
        self.assertEqual(str(self.parser.parse('[[[ ws35-035 ]]]')), 'h@ws35-035')

    def test_simple3(self):
        self.assertEqual(str(self.parser.parse('ws35-035 RusTier0')), 'h@RusTier0 | h@ws35-035')

    def test_simple4(self):
        self.assertEqual(str(self.parser.parse('ws35-035 S@RusTier0')), 'S@RusTier0 | h@ws35-035')

    def test_simple5(self):
        self.assertEqual(str(self.parser.parse('A [ S@B - C ]')), 'S@B & ~h@C | h@A')

    def test_simple6(self):
        self.assertEqual(str(self.parser.parse('[ A - B ] [ C - D ]')), 'h@A & ~h@B | h@C & ~h@D')

    def test_simple7(self):
        self.assertEqual(str(self.parser.parse('[A B] . [C D]')), 'h@A & h@C | h@A & h@D | h@B & h@C | h@B & h@D')

    def test_simple8(self):
        self.assertEqual(str(self.parser.parse('d@A dc@B l@C line@D')), 'd@A | d@B | l@C | l@D')

    def test_complex1(self):
        self.assertEqual(
            str(self.parser.parse('C@[HEAD DEBUG] . [ ws35-035 S@RusTier0 ]')),
            'C@DEBUG & S@RusTier0 | C@DEBUG & h@ws35-035 | C@HEAD & S@RusTier0 | C@HEAD & h@ws35-035'
        )

    def test_complex2(self):
        self.assertEqual(str(self.parser.parse('C@[HEAD - [DEBUG - TEST]] . [ ws35-035 S@RusTier0 ]')),
                         'C@HEAD & C@TEST & S@RusTier0 | C@HEAD & C@TEST & h@ws35-035'
                         ' | C@HEAD & S@RusTier0 & ~C@DEBUG | C@HEAD & h@ws35-035 & ~C@DEBUG')

    def test_ipv4(self):
        self.assertEqual(str(self.parser.parse('127.0.0.1')), 'h@127.0.0.1')

    def test_ipv6(self):
        self.assertEqual(str(self.parser.parse('ae12:123::a')), 'h@ae12:123::a')

    def test_syntax_error1(self):
        self.failUnlessRaises(HrSyntaxError, self.parser.parse, '[ - ]')

    def test_syntax_error2(self):
        self.failUnlessRaises(HrSyntaxError, self.parser.parse, '$')

    def test_syntax_error3(self):
        self.failUnlessRaises(HrSyntaxError, self.parser.parse, '    ')

    def test_syntax_error4(self):
        self.failUnlessRaises(HrSyntaxError, self.parser.parse, '[[[[ a b ]]]]]')


if __name__ == '__main__':
    unittest.main()
