from library.sky.hostresolver.resolverbase import ResolverBase
import unittest

from library.sky.hostresolver.helpers import hostsDictsIntersection
from library.sky.hostresolver.parser import Parser
from library.sky.hostresolver.expression import transform_literals, Literal


class MockSourceCms(object):
    def __init__(self, mockData=None):
        self.mock_data = mockData or {}

    def getInstancesByHosts(self, kwargsDict):
        key_map = {
            'conf': 'C',
            'shardTagName': 'S',
            'instanceTagName': 'I',
            'shard': 's',
            'host': 'h',
            'conductorproject': 'K',
            'dc': 'd',
            'line': 'l',
        }
        it = kwargsDict.iteritems()
        k, v = it.next()
        resultInstances = self.mock_data.get((key_map[k], v), {})
        for k, v in it:
            resultInstances = hostsDictsIntersection(
                resultInstances,
                self.mock_data.get((key_map[k], v), {})
            )

        return resultInstances

    def get_hosts_by_groups(self, groups):
        hosts = set()
        for g in groups:
            hosts.update(self.mock_data.get((g.prefix, g.name.upper()), ()))
        return hosts


class TestResolverFunctions(unittest.TestCase):
    def setUp(self):
        source_cms = MockSourceCms({
            ('h', 'ws35-001.yandex.ru'): {'ws35-001.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('h', 'ws35-002.yandex.ru'): {'ws35-002.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('h', 'ws35-003.yandex.ru'): {'ws35-003.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('h', 'ws35-004.yandex.ru'): {'ws35-004.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('h', 'ws35-006.yandex.ru'): {'ws35-006.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('h', 'ws35-035.yandex.ru'): {'ws35-035.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('h', 'ws35-040.yandex.ru'): {'ws35-040.yandex.ru': set([('primus1', 'a'), ('primus2', None)])},
            ('C', 'HEAD'): {
                'ws35-001.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
                'ws35-002.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
                'ws35-003.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
                'ws35-004.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
                'ws35-006.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
                'ws35-035.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
                'ws35-040.yandex.ru': set([('primus1', 'a'), ('primus2', None)]),
            },
            ('K', 'G3'): set([
                'ws35-001.yandex.ru',
                'ws35-002.yandex.ru',
                'ws35-035.yandex.ru',
                'ws35-040.yandex.ru',
            ]),
            ('K', 'G4'): set([
                'ws35-003.yandex.ru',
                'ws35-004.yandex.ru',
                'ws35-040.yandex.ru',
            ]),
            ('K', 'G1'): set([
                'ws35-001.yandex.ru',
                'ws35-002.yandex.ru',
                'ws35-035.yandex.ru',
                'ws35-040.yandex.ru'
            ]),
            ('K', 'G2'): set([
                'ws35-003.yandex.ru',
                'ws35-004.yandex.ru',
                'ws35-040.yandex.ru',
            ]),
            ('d', 'G1'): set([
                'ws35-001.yandex.ru',
                'ws35-002.yandex.ru',
                'ws35-035.yandex.ru',
                'ws35-040.yandex.ru'
            ]),
            ('l', 'G1'): set([
                'ws35-001.yandex.ru',
                'ws35-002.yandex.ru',
                'ws35-035.yandex.ru',
                'ws35-040.yandex.ru',
            ]),
        })

        def normalize_host_name(name):
            if '.' not in name:
                return name + '.yandex.ru'
            else:
                return name

        self.resolver = ResolverBase(source_cms, normalize_host_name, True)

    def test_hosts1(self):
        result = self.resolver.resolve_hosts('ws35-035 ws35-002.yandex.ru')
        self.assertEquals(set(result), set(['ws35-035.yandex.ru', 'ws35-002.yandex.ru']))

    def test_hosts3(self):
        result = self.resolver.resolve_hosts('K@G3 - ws35-035')
        self.assertEquals(set(result), set(['ws35-001.yandex.ru', 'ws35-002.yandex.ru', 'ws35-040.yandex.ru']))

    def test_hosts4(self):
        result = self.resolver.resolve_hosts('K@G3 - K@G4')
        self.assertEquals(set(result), set(['ws35-001.yandex.ru', 'ws35-035.yandex.ru', 'ws35-002.yandex.ru']))

    def test_hosts5(self):
        result = self.resolver.resolve_hosts('K@G3 . K@G4')
        self.assertEquals(set(result), set(['ws35-040.yandex.ru']))

    def test_hosts6(self):
        result = self.resolver.resolve_hosts('K@G1 . K@G4')
        self.assertEquals(set(result), set(['ws35-040.yandex.ru']))

    def test_hosts7(self):
        result = self.resolver.resolve_hosts('K@G3 - K@G1')
        self.assertEquals(set(result), set())

    def test_hosts8(self):
        result = self.resolver.resolve_hosts('K@G1 - K@G3')
        self.assertEquals(set(result), set())

    def test_hosts6_2(self):
        result = self.resolver.resolve_hosts('d@G1 . K@G4')
        self.assertEquals(set(result), set(['ws35-040.yandex.ru']))

    def test_hosts7_2(self):
        result = self.resolver.resolve_hosts('K@G3 - d@G1')
        self.assertEquals(set(result), set())

    def test_hosts8_2(self):
        result = self.resolver.resolve_hosts('d@G1 - K@G3')
        self.assertEquals(set(result), set())

    def test_hosts6_3(self):
        result = self.resolver.resolve_hosts('l@G1 . K@G4')
        self.assertEquals(set(result), set(['ws35-040.yandex.ru']))

    def test_hosts7_3(self):
        result = self.resolver.resolve_hosts('K@G3 - l@G1')
        self.assertEquals(set(result), set())

    def test_hosts8_3(self):
        result = self.resolver.resolve_hosts('l@G1 - K@G3')
        self.assertEquals(set(result), set())

    def test_hosts9(self):
        self.assertEquals(set(self.resolver.resolve_hosts('K@g3')), set(self.resolver.resolve_hosts('K@G3')))

    def test_brackets(self):
        result = self.resolver.resolve_hosts('(([[K@G4 . K@G3))]]')
        self.assertEquals(result, {'ws35-040.yandex.ru'})


class TestTransformLiterals(unittest.TestCase):
    def setUp(self):
        self.parser = Parser()

    def test_add_suffix(self):
        def add_suffix_to_host(literal):
            if literal.prefix == 'h':
                return Literal(literal.name + ".suffix", literal.prefix)
            return literal

        (dnf1, _) = transform_literals(self.parser.parse("bla-bla K@cms yada-yada"), add_suffix_to_host)
        dnf2 = self.parser.parse("bla-bla.suffix K@cms yada-yada.suffix")
        self.assertEquals(dnf1, dnf2)

    def test_add_suffix_fail(self):
        error_text = "error text"

        def add_suffix_to_host(literal):
            if literal.prefix == 'h':
                if literal.name == "bla-bla":
                    raise Exception(error_text)
                return Literal(literal.name + ".suffix", literal.prefix)
            return literal

        (dnf1, errors) = transform_literals(self.parser.parse("bla-bla K@cms yada-yada"), add_suffix_to_host)
        dnf2 = self.parser.parse("bla-bla K@cms yada-yada.suffix")
        self.assertEquals(dnf1, dnf2)
        self.assertEquals(len(errors), 1)
        self.assertEquals(errors["h@bla-bla"], error_text)


if __name__ == '__main__':
    unittest.main()
