# -*- coding: utf-8 -*-
import re
import itertools
import warnings


class AutoBlinovError(Exception):
    pass


def braceExpansion(argv, asString=False):
    """
    This is a simple bash brace expansion function. It does not supports nested braces but is pretty
    useful for expand hosts list packed by previously executed command.
    :param argv:        A list of expression to be processed.
    :param asString:    Perform result concatenation into a single string joined with space symbol.
    :return:            Expanded list or string depending on `asString` argument.
    """
    r = re.compile(r'''
        ([^{]*)
        {
            (?:(?:((?:0*)(\d+))\.\.((?:0*)(\d+)))
            |
            ([^{}\s]+))
        }
        ([^{}]*)
    ''', re.VERBOSE)
    hosts = []
    for arg in itertools.chain(*(s.split() for s in argv)):
        pos = 0
        addon = []
        while pos < len(arg):
            m = r.match(arg, pos)
            if not m:
                break
            prev = pos
            pos = m.end()
            if m.group(3) and m.group(5):  # Range
                mlen = max(len(m.group(i)) for i in [2, 4])
                chunk = map(
                    lambda x: '{0}{1:0{len}d}{2}'.format(m.group(1), x, m.group(7), len=mlen),
                    range(int(m.group(3)), int(m.group(5)) + 1)
                )
            elif m.group(6):
                chunk = map(
                    lambda x: ''.join([m.group(1), x, m.group(7)]),
                    m.group(6).split(',')
                )
            else:
                chunk = [arg[prev:pos]]
            addon = \
                chunk \
                if not addon else \
                list(itertools.chain.from_iterable(map(lambda x: x + s, addon) for s in chunk))
        hosts += addon if addon else [arg]

    return hosts if not asString else ' '.join(hosts)


def detectResolver(args):
    oldPrefixes = [
        (prefix + sign)
        for prefix in ['', 'i', 's', 'shard']
        for sign in ['-', '+']
    ]

    newPrefixes = [
        prefix + '@'
        for prefix in [
            'C', 'CONF', 'conf',
            'h', 'host',
            'I', 'INSTANCETAG', 'ITAG', 'instancetag', 'itag',
            'S', 'SHARDTAG', 'STAG', 'shardtag', 'stag',
            's', 'shard',
            'K', 'CONDUCTORGROUP', 'conductorgroup',
            'k', 'CONDUCTORTAG', 'conductortag',
            'P', 'CONDUCTORPROJECT', 'conductorproject',
            'D', 'CONDUCTORDC', 'conductordc',
            'G', 'GENCFG', 'gencfg',
            'W', 'WALLE', 'walle',
            'w', 'WALLETAG', 'walletag',
            't', 'TOR', 'tor',
            'f', 'FAMILY', 'HQ', 'family',
            'Q', 'QLOUD', 'qloud',
            'q', 'QLOUDHW', 'qloudhw',
            'M', 'MTN', 'mtn',
            'm', 'SLAVEMTN', 'slavemtn',
            'd', 'dc',
            'l', 'line',
            'z', 'SG', 'SAMOGON', 'sg', 'samogon',
            'Y', 'YP', 'yp',
            'p', 'PODS', 'pods',
            'b', 'BOXES', 'boxes',
        ]
    ]
    newMarkers = [
        '[', ']', '(', ')'
    ]

    item = None
    for rawitem in args:
        if ' ' in rawitem:
            items = rawitem.split()
        else:
            items = [rawitem]

        for item in items:
            item = item.strip()

            for name, markers, prefixes in (
                ('yr', [], oldPrefixes),
                ('blinov', newMarkers, newPrefixes)
            ):
                for marker in markers:
                    if marker in item:
                        return name

                for prefix in prefixes:
                    if prefix == item:
                        continue
                    if item.startswith(prefix):
                        return name

    # we failed to detect resolver, so we check last item for plain hostnames, which can only be in blinov calc
    if item and '@' not in item and '+' not in item:
        return 'blinov'

    return ''


def resolveHosts(args, check_hostnames=True):
    warnings.warn('Using library.sky.hosts.resolveHosts() is deprecated, please use library.sky.hostresolver.Resolver()!')
    # invert logic:
    # +SEARCH1 - +ws1-100  =  +SEARCH1 -ws1-100
    # +SEARCH1 - +ws1-100 -ws6-100 + +ws6-200  =  +SEARCH1 -ws1-100 +ws6-100 +ws6-200
    # +ws1-100 - +ws1-100 -ws1-101 + -ws1-101  =  None

    # +ws6-100 +ws6-101 => ws6-100 ws6-101
    # +CMS -CMS +CMS => (CMS CMS) - (CMS)
    tempArgs = []
    invert = False
    for a in braceExpansion(args):
        if a == '-':
            invert = True
            continue
        elif a == '+':
            invert = False
            continue
        if invert:
            if a[0] == '+':
                tempArgs.append('-' + a[1:])
            elif a[0] == '-':
                tempArgs.append('+' + a[1:])
        else:
            tempArgs.append(a)
    args = tempArgs
    del tempArgs

    def isGroupName(name):
        return name.isupper()

    positiveGroup = set()
    negativeGroup = set()
    positiveConfGroup = set()
    negativeConfGroup = set()

    cfg = None

    def translate(item):
        """
        Translate hosts from old syntax to new:
          +host -> h@host
          +GROUP -> H@GROUP
          i+testws -> I@testws
          i+C1 -> I@priemka
          s+shardtag -> S@shardtag
          shard+name -> s@name
        """

        for postfix, groups in (
                ('+', (positiveGroup, positiveConfGroup)),
                ('-', (negativeGroup, negativeConfGroup)),
        ):
            for typA, typB in (
                    ('s', 'S@'),
                    ('i', 'I@'),
                    ('shard', 's@'),
                    ('', ('', 'H@'))
            ):
                prefix = typA + postfix
                if item.startswith(prefix):
                    item = item[len(prefix):]
                    if typA == '':  # special case for +host +GROUP
                        if isGroupName(item):
                            item = typB[1] + item
                        else:
                            item = typB[0] + item
                        groups[0].add(item)
                    else:
                        if typA == 'i':  # special case to handle i+R1 i+C1
                            if item == 'R1':
                                item = 'testws'
                            elif item == 'C1':
                                item = 'priemka'
                            item = typB + item
                        else:
                            item = typB + item
                        groups[1].add(item)
                    return

        raise Exception(
            'Unsupported host name format "%s". '
            'Please use Blinov\'s Calc syntax' % (item, )
        )

    for arg in args:
        if ' ' in arg:
            items = arg.split()
        else:
            items = [arg]

        for item in items:
            if item.startswith('cfg='):
                if cfg is not None:
                    raise Exception('Specifing cfg= multiple times is meaningless in old calculator')
                cfg = item[len('cfg='):]
            else:
                translate(item)

    from library.sky.hostresolver.errors import HrSyntaxError, HrHostCheckFailedError
    from library.sky.hostresolver.resolver import Resolver

    # Construct positive and negative clauses here.
    # First resolve instancetags, shards and shardtags. If configuration was specified -- intersect them
    # with specified configuration. If only configuration was specified and no instancetags, shards nor
    # shardtags -- dump full configuration (weird legacy behaviour!)
    # So, at the end we will have something like these:
    #   C@CONF (only for positive clause)
    #   (I@ITAG S@STAG)
    #   C@CONF . (I@ITAG S@STAG)
    #   '' (empty, yep!)

    if not positiveConfGroup and not negativeConfGroup:
        if cfg:
            positiveClause = 'C@' + cfg
        else:
            positiveClause = ''
        negativeClause = ''
    else:
        positiveClause = negativeClause = ''
        if positiveConfGroup:
            if cfg:
                positiveClause = 'C@' + cfg + ' . '
            positiveClause += '(' + ' '.join(positiveConfGroup) + ')'
        if negativeConfGroup:
            if cfg:
                negativeClause = 'C@' + cfg + ' . '
            negativeClause += '(' + ' '.join(negativeConfGroup) + ')'

    # After that add to positive and negative clauses plain hostnames
    # E.g.
    #   C@CONF -> C@CONF h@host
    #   C@CONF . (I@ITAG S@STAG) -> C@CONF . (I@ITAG S@STAG) h@host
    if positiveGroup:
        if positiveClause:
            positiveClause += ' '
        positiveClause += ' '.join(positiveGroup)
    if negativeGroup:
        if negativeClause:
            negativeClause += ' '
        negativeClause += ' '.join(negativeGroup)

    # Finally, construct full clause
    #   (POSITIVE)
    #   (POSITIVE) - (NEGATIVE)
    if not positiveClause and not negativeClause:
        raise Exception('Please use Blinov\'s Calc syntax')
    elif not positiveClause:
        # Without any positive clause we will have empty set
        return set(), set()
    else:
        clause = '(%s)' % (positiveClause, )
        if negativeClause:
            clause = '%s - (%s)' % (clause, negativeClause, )

    try:
        return Resolver(check_hostnames).resolveHosts(clause), set()
    except (HrSyntaxError, HrHostCheckFailedError) as ex:
        raise AutoBlinovError(clause, ex)
