# -*- coding: utf-8 -*-
import subprocess
import os
import socket
import signal

from library.tasks.shell import userDefaultShell


class CommandRunner(object):
    """
    Простой запуск шелл-команды.
    Команда запускается на машине, собирает stdout, stderr и код возврата,
    после чего возвращает их пользователю в качестве результата.

    :param str cmd: строка с шелл-командой
    :param bool noWait: если флаг выставлен, задача не будет
                        дожидаться выполнения команды, а сразу вернёт кортеж ("", "", 0)
    :param dict env: опционально можно передать список переменных окружения, с которыми
                     необходимо запустить процесс
    """
    def __init__(self, cmd, noWait=False, env=None):
        super(CommandRunner, self).__init__()
        self.cmd = cmd
        self.noWait = noWait

        self.env = env

    def __str__(self):
        return self.cmd

    def startCommand(self, dupStderr=False, env=None):
        if self.noWait:
            stdout = open(os.devnull, "w")
            stderr = subprocess.STDOUT
        else:
            stdout, stderr = (subprocess.PIPE, subprocess.STDOUT if dupStderr else subprocess.PIPE)

        procEnv = dict(self.env) if self.env is not None else os.environ
        procEnv['SHELL'] = userDefaultShell()
        if env is not None:
            procEnv.update(env)

        def preexec():
            if self.noWait:
                os.setpgrp()
            signal.signal(signal.SIGPIPE, signal.SIG_DFL)

        return subprocess.Popen(
            self.cmd,
            bufsize=-1,
            env=procEnv,
            shell=True,
            executable=userDefaultShell(),
            stdout=stdout,
            stderr=stderr,
            close_fds=True,
            preexec_fn=preexec
        )

    def run(self, env=None):
        process = self.startCommand(env=env)
        if self.noWait:
            return "", "", 0
        out = process.communicate()
        return out[0], out[1], process.returncode


class IterCommandRunner(CommandRunner):
    """
    Запуск шелл-команды с постепенным получением результатов.
    Команда полностью аналогична :class:`~.CommandRunner`, с тем
    отличием, что задача должна быть запущена в iter-режиме и в
    качестве результатов по мере выполнения будут yield'иться
    пары вида (mark, line), где line — это строка вывода, а
    mark — тип потока вывода, с которого получена строка (один из
    :attr:`~.IterCommandRunner.STDOUT` и :attr:`~.IterCommandRunner.STDERR`).

    :param str cmd: строка с шелл-командой
    """
    STDOUT = 1  #: метка, соответствующая выводу на stdout
    STDERR = 2  #: метка, соответствующая выводу на stderr

    def __call__(self):
        import fcntl
        import select
        import errno

        process = self.startCommand()

        ios = [process.stdout.fileno(), process.stderr.fileno()]
        collectedData = [[], []]

        for fd in ios:
            curFlags = fcntl.fcntl(fd, fcntl.F_GETFL)
            fcntl.fcntl(fd, fcntl.F_SETFL, curFlags | os.O_NONBLOCK)

        while ios:
            ready = select.select(ios, [], [])[0]

            for data, mark, stream in (
                (collectedData[0], IterCommandRunner.STDOUT, process.stdout),
                (collectedData[1], IterCommandRunner.STDERR, process.stderr)
            ):
                if stream.fileno() not in ready:
                    continue

                try:
                    piece = stream.read()
                except EnvironmentError as err:
                    if err[0] == errno.EBADF:
                        return
                    elif err[0] == errno.EAGAIN:
                        continue
                    else:
                        raise

                if not piece:
                    if data:
                        yield (mark, ''.join(data))
                        del data[:]
                    ios.remove(stream.fileno())
                    continue
                elif '\n' in piece:
                    piece = (''.join(data) + piece).split('\n')
                    del data[:]
                    for x in piece[:-1]:
                        yield(mark, x + '\n')
                    if piece[-1]:
                        data.append(piece[-1])
                else:
                    data.append(piece)

    run = __call__

yandexSuffix = '.yandex.ru'


def hostSpecificData(dct):
    hostName = socket.gethostname()
    data = dct.get(hostName, None)
    if data is None:
        if hostName.endswith(yandexSuffix):
            hostName = hostName[:-len(yandexSuffix)]
            return dct.get(hostName, [])
        else:
            return []
    else:
        return data


class ShardCommandRunner(CommandRunner):
    """
    Запускает команду «пошардово».
    Команда последовательно будет запущена для каждого
    из шардов этого хоста.
    В качестве результата выполнения команды будет возвращён не кортеж,
    как у :class:`~.CommandRunner`, а словарь вида ("имя шарда" → кортеж).

    .. envvar:: SHARD
        В этой переменной окружения при каждом запуске будет находиться имя
        соответствующего шарда, например, "primus034-028-1400583685".

    :param dict shardsByHosts: словарь, где ключом выступает полное доменное имя хоста,
                               а значением — список имён шардов для этого хоста
    :param str cmd: строка с шелл-командой
    :param bool noWait: если флаг выставлен, задача не будет
                        дожидаться выполнения команды, а сразу вернёт кортеж ("", "", 0)
    :param dict env: опционально можно передать список переменных окружения, с которыми
                     необходимо запустить процесс
    """
    def __init__(self, shardsByHosts, cmd, noWait=False, env=None):
        super(ShardCommandRunner, self).__init__(cmd, noWait=noWait, env=env)
        self.shardsByHosts = shardsByHosts

    def run(self):
        shards = hostSpecificData(self.shardsByHosts)
        results = {}
        for shard in shards:
            results[shard] = super(ShardCommandRunner, self).run(env={'SHARD': shard})
        return results

    __call__ = run


class InstanceCommandRunner(CommandRunner):
    """
    Запускает команду «поинстансово».
    Команда последовательно будет запущена для каждого
    из инстансов базового поиска на этом хосте.
    В качестве результата выполнения команды будет возвращён не кортеж,
    как у :class:`~.CommandRunner`, а словарь вида ("инстанс" → кортеж).

    .. envvar:: INSTANCE_FULL
        В этой переменной окружения при каждом запуске будет находиться полное имя
        экземпляра базового поиска, например, "ws1-123:8044@HEAD"

    .. envvar:: INSTANCE
        В этой переменной окружения при каждом запуске будет находиться короткое имя
        экземпляра базового поиска, например, "ws1-123:8044".

    :param dict instanceByHosts: словарь, где ключом выступает полное доменное имя хоста,
                               а значением — список имён шардов для этого хоста
    :param str cmd: строка с шелл-командой
    :param bool noWait: если флаг выставлен, задача не будет
                        дожидаться выполнения команды, а сразу вернёт кортеж ("", "", 0)
    :param dict env: опционально можно передать список переменных окружения, с которыми
                     необходимо запустить процесс
    """
    def __init__(self, instancesByHosts, cmd, noWait=False, env=None):
        super(InstanceCommandRunner, self).__init__(cmd, noWait=noWait, env=env)
        self.instancesByHosts = instancesByHosts

    def run(self):
        instances = hostSpecificData(self.instancesByHosts)
        results = {}
        for shard, instance in instances:
            if instance is None:
                continue
            env = {'INSTANCE_FULL': instance,
                   'INSTANCE': instance[:instance.index('@')],
                   'INSTANCE_PORT': instance[instance.index(':') + 1:instance.index('@')],
                   }
            results[instance] = super(InstanceCommandRunner, self).run(env=env)
        return results

    __call__ = run
