import os
import subprocess
import signal
import select
import traceback

from .shell import ShellRunner, TelnetHandler


class RunGdb:
    def __init__(self, command):
        self.command = command

    def run(self):
        gdb = None
        for s in ['/bin/gdb', '/usr/bin/gdb', '/usr/local/bin/gdb']:
            if os.path.exists(s):
                gdb = s
                break
        if not gdb:
            raise Exception('no gdb on remote machine')
        gdb_command = None
        if self.command:
            import shlex
            gdb_command = shlex.split(self.command)
        command = [gdb, '-q']
        if gdb_command:
            command += ['--args'] + gdb_command
        subprocess.Popen(command, shell=False, close_fds=True, bufsize=4096).wait()


class GdbRunner(ShellRunner):
    def __init__(self, term_type, command, process_timeout=0):
        ShellRunner.__init__(self, term_type, command)
        self.process_timeout = process_timeout

    @staticmethod
    def run_gdb_process(pty_fd):
        def _safe_read(src, timeout=None, bufferSize=4096):
            rr, ww, xx = select.select([src], [], [], timeout)
            if src in rr:
                data = os.read(src, bufferSize)
                return data

        def _safe_write(src, data, timeout=None):
            rr, ww, xx = select.select([], [src], [], timeout)
            if src in ww:
                bytesWriten = os.write(src, data)
                return bytesWriten

        showPromptLabel = 'Gdb\'s prompt is '
        _safe_write(pty_fd, 'show prompt\r')
        data = ''
        while True:
            data += _safe_read(pty_fd)
            if showPromptLabel in data:
                break

        for line in data.split('\n'):
            if showPromptLabel in line:
                gdbPrompt = line
                break
        import re
        pattern = '.+?{0}"(.*)"\..+?'.format(showPromptLabel)
        currentGdbPrompt = re.match(pattern, gdbPrompt).group(1)

        _safe_write(pty_fd, 'set history size 1\r')
        data = ''
        while True:
            data += _safe_read(pty_fd)
            if data.count(currentGdbPrompt) == 2:
                break

        newGdbPrompt = '(<<____new__gdb__prompt__{0}___>>) '.format(pty_fd)
        _safe_write(pty_fd, 'set prompt {0}\r'.format(newGdbPrompt))
        data = ''
        while True:
            data += _safe_read(pty_fd)
            if data.count(newGdbPrompt) == 1:
                break

        _safe_write(pty_fd, 'run\r')
        data = ''
        while True:
            data += _safe_read(pty_fd)
            if data.count(newGdbPrompt) == 2:
                break

        gdbProcessOutput = data
        gdbProcessOutput = gdbProcessOutput.replace(newGdbPrompt, currentGdbPrompt)

        successfulStopLabel = 'Program exited normally.'
        if successfulStopLabel in gdbProcessOutput:
            return (gdbProcessOutput, True)

        _safe_write(pty_fd, 'set prompt {0}\r'.format(currentGdbPrompt))
        data = ''
        while True:
            data += _safe_read(pty_fd)
            if data.count(currentGdbPrompt) == 2:
                break

        _safe_write(pty_fd, 'set history size 0\r')
        data = ''
        while True:
            data += _safe_read(pty_fd)
            if data.count(currentGdbPrompt) == 1:
                break

        return (gdbProcessOutput, False)

    def run(self):
        if self.process_timeout:
            signal.alarm(self.process_timeout)

        self.create_socket()

        handler = TelnetHandler(None, self.term_type, self.command)
        handler.setup_environment()
        handler.fork_shell()

        import datetime
        executionTime = datetime.datetime.now()
        gdbProcessOutput, gdbProcessSuccess = GdbRunner.run_gdb_process(handler.pty_fd)
        executionTime = datetime.datetime.now() - executionTime

        if gdbProcessSuccess:
            return (None, gdbProcessOutput, executionTime)

        pid = os.fork()
        if pid != 0:
            self.server_sock.close()
            return (self.telnet_port, gdbProcessOutput, executionTime)

        sock = self.accept_socket()
        signal.alarm(0)

        telnet = handler.create_telnet(sock)

        err = None
        try:
            handler.telnet_session(telnet)
        except:
            err = traceback.format_exc()
        finally:
            handler.destroy()
        return err
