# -*- coding: utf-8 -*-
import os
import subprocess
import tempfile
import shutil
import api.copier


class LaunchRunner(object):
    """
    Запустить локальный файл на удалённой машине.
    Файл будет скопирован на машину при помощи торрента, после
    чего запущен по аналогии с :class:`~library.tasks.command.CommandRunner`.

    :param str remoteWD: рабочий каталог на удалённой машине, в котором должен
                         происходить запуск. NB: это не тот каталог, в котором
                         будет лежать скачанный файл!
    :param str localBin: путь к локальному исполняемому файлу, который необходимо
                         запустить
    :param list executionParams: список аргументов, с которыми должна быть запущена
                                 команда
    :param bool noWait: если флаг выставлен, задача не будет
                        дожидаться выполнения команды, а сразу вернёт кортеж ("", "", 0)
    """
    def __init__(self, remoteWD, localBin, executionParams, noWait=False):
        self.remoteWD = remoteWD
        self.localBin = os.path.abspath(localBin)
        self.executionParams = executionParams
        self.noWait = noWait

        localBinDir = os.path.dirname(self.localBin)
        localBinName = os.path.basename(self.localBin)
        self.binResId = api.copier.Copier().create([localBinName], cwd=localBinDir).resid()

    def __str__(self):
        return 'Launch [{} {}]'.format(self.localBin, self.executionParams)

    def run(self):
        tmpdir = tempfile.mkdtemp()
        try:
            os.chmod(tmpdir, 0o755)
            api.copier.Copier().get(self.binResId, dest=tmpdir).wait()

            remoteBin = os.path.join(tmpdir, os.path.basename(self.localBin))
            shellCommand = self.executionParams[:]
            shellCommand.insert(0, remoteBin)

            stdout, stderr = (subprocess.PIPE, ) * 2
            if self.noWait:
                stdout = open(os.devnull, "w")
                stderr = subprocess.STDOUT
            process = subprocess.Popen(shellCommand, shell=False, stdout=stdout, stderr=stderr, cwd=self.remoteWD)
        except:
            shutil.rmtree(tmpdir)
            raise

        if self.noWait:
            return "", "", 0

        try:
            out = process.communicate()
            return out[0], out[1], process.returncode
        finally:
            shutil.rmtree(tmpdir)
