# coding: utf-8
from __future__ import print_function

import os
import sys
import socket

from library.auth.sign import FileKeysSignManager, ChainSignManager
from library.auth.signagent import SignAgentClient
from library.tasks.shell import TelnetClient
from library.portoshell import make_message, sign_message, establish_connection

from kernel.util.config import Config


def create_signer(identity_file=None):
    fksm = FileKeysSignManager(
        userKeyDirs=['{installRoot}/etc/auth/{userName}', '{userHome}/.ssh'],
        keyFiles=['.*private$', 'id_rsa', 'id_dsa'],
    )
    fksm.load()

    if identity_file is not None:
        try:
            keys = list(fksm.loadKeys(open(identity_file, 'rb'), identity_file))
        except EnvironmentError as e:
            print("Can't load keys from '%s': %s" % (identity_file, e), file=sys.stderr)
            raise SystemExit(1)

        for key in keys:
            fksm.addKey(key)

    sa = SignAgentClient()
    sa.load()

    return ChainSignManager([fksm, sa])


def portoshell(slot,
               server_port,
               user,
               identity_file=None,
               extra_env=None,
               unset_env=None,
               configuration_id='',
               hostname=None,
               command=None,
               interactive_cmd=False,
               inactivity_timeout=3600,
               ):
    sm = create_signer(identity_file)

    if hostname:
        host = hostname
    elif '@' in slot:  # canonical format service@host
        host = slot.split('@', 1)[1]
    elif ':' in slot:
        host = slot.split(':', 1)[0]
    else:
        print("Cannot deduce hostname from the slot name %r. Use option -H to specify remote host." % (slot,),
              file=sys.stderr)
        raise SystemExit(1)

    files = {}
    home = os.path.expanduser('~')
    for filename in cfg.Deploy:
        name = os.path.expanduser(filename)
        basename = name[len(home) + 1:] if name.startswith(home) else name
        try:
            if os.path.exists(name) and os.path.isfile(name):
                files[basename] = open(name).read()
                continue
        except EnvironmentError:
            pass
        if basename in cfg.DefaultFiles:
            files[basename] = cfg.DefaultFiles[basename]

    msg = make_message(
        user,
        slot,
        configuration_id=configuration_id,
        extra_env=extra_env,
        unset_env=unset_env,
        command=command,
        interactive_cmd=interactive_cmd,
        inactivity_timeout=inactivity_timeout,
        files=files,
    )
    sign_message(msg, sm)

    try:
        sock = establish_connection(host, server_port, msg)
    except socket.gaierror as e:
        print("Host resolve failed: " + str(e), file=sys.stderr)
        raise SystemExit(1)
    except EnvironmentError as e:
        print(e.message or e, file=sys.stderr)
        raise SystemExit(1)

    telnet = TelnetClient()
    telnet.sock = sock
    telnet.interact()


cfg = Config('ya.skynet.portoshell', """
# Список файлов и каталогов, которые необходимо скопировать на удалённую машину
Deploy : ['~/.bashrc', '~/.profile', '~/.vimrc', '~/.inputrc', '~/profile', '~/.zshrc']
# Стандартные файлы, которые нужно использовать, если на текущей машине их нет
DefaultFiles:
  .bashrc : |
    export PATH=$PATH:/skynet/tools:/Berkanavt/bin
    export PS1="\[\\033[34;01m\]\\u@\h:\w\[\\033[m\]\\\\$ "
    export HISTFILE=
    export HISTCONTROL=ignoreboth
    export LESSHISTFILE=-
    export LANG=en_US.UTF-8
    alias ls="ls -G"
    alias l="ls -l"
    alias ll="ls -la"

  .inputrc : |
    set input-meta on
    set output-meta on
    $if mode=emacs
    "\e[1~": beginning-of-line
    "\e[4~": end-of-line
    "\e[3~": delete-char
    "\e[2~": quoted-insert
    "\e[5~": beginning-of-history
    "\e[6~": end-of-history
    "\e[1;5C": forward-word
    "\e[1;5D": backward-word
    "\e[5C": forward-word
    "\e[5D": backward-word
    "\e\e[C": forward-word
    "\e\e[D": backward-word
    $if term=rxvt
    "\e[8~": end-of-line
    "\eOc": forward-word
    "\eOd": backward-word
    $endif
    $endif

  .vimrc: |
    set nocompatible nobackup noswapfile noerrorbells backspace=2 autoread autowrite nowrap
    set incsearch ignorecase smartcase hlsearch tabstop=4 shiftwidth=4 expandtab autoindent
    set ruler showcmd nowrap wildmenu formatoptions+=r display+=lastline,uhex
    set laststatus=2 statusline=%<%f%h%m%r%=%b=0x%B\ \ %l,%c%V\ %P
    set fileencodings=ucs-bom,utf-8,cp1251,default,latin1
    set viminfo=""
    syntax on
""")
