# -*- coding: utf-8 -*-
from library.tasks.remote_object_wrapper import RemoteObjectWrapper


class TimeOutTask(RemoteObjectWrapper):
    """
    Выполнить задачу с заданным таймаутом.
    Если задача исполняется через CQueue, для
    ограничения её времени выполнения используется
    встроенная функциональность, в противном случае
    используется более общий механизм, когда задача
    запускается в побочном потоке, а основной поток
    следит за временем и при необходимости прерывает
    выполнение.

    :param slave: задача для выполнения
    :param int timeout: лимит времени выполнения в секундах
    :param bool useCqueue: использовать ли встроенные механизмы
                           CQueue для ограничения времени выполнения
    """
    def __init__(self, slave, timeout, useCqueue=False):
        super(TimeOutTask, self).__init__(slave)
        self.timeout = timeout
        self.__useCqueue = useCqueue
        self.__sessionTimeout = timeout
        self.__discoverTimeout = timeout
        self.__orphanTimeout = timeout

    def __str__(self):
        return '(-t {0}): {1}'.format(self.timeout, str(self.slave))

    def run(self):
        if self.__useCqueue:
            return self.slave.run()
        else:
            return self._runWithTimeout(self.timeout)

    def _runWithTimeout(self, timeout):
        from threading import Thread
        from Queue import Queue, Empty

        q = Queue()

        def func():
            try:
                res = (self.slave.run(), None)
            except Exception as e:
                res = (None, e)

            q.put(res)

        Thread(target=func).start()

        try:
            (res, err) = q.get(block=True, timeout=timeout)
        except Empty:
            raise Exception('timeout')

        if err:
            raise err

        return res

    @property
    def sessionTimeout(self):
        return self.__sessionTimeout

    @property
    def discoverTimeout(self):
        return self.__discoverTimeout

    @property
    def orphanTimeout(self):
        return self.__orphanTimeout
