# -*- coding: utf-8 -*-
from __future__ import print_function
from .sysinfo import SysInfo

import six


if six.PY3:
    long = int


class StatCollector(object):
    """
    Получение информации о системе.
    В качестве результата возвращается словарь с основными
    показателями загрузки системы.

    Пример такого словаря:

        .. runblock:: pycon

            from pprint import pprint
            from library.tasks.top import StatCollector
            pprint(StatCollector().run())
    """
    def run(self):
        info = SysInfo()

        ret = {
            'LOAD': info.loadAvg(),
            'TMP': info.tmpFreeSpace(),
            'SPACE': info.freeSpace(),
            'SOCKETS': info.sockets(),
            'IDLE': info.idle(),
            'FAULTS': info.pageFaults(),
            'NCPU': info.nCpu(),
            'NORMLOAD': info.normalizedLoadAvg(),
        }

        return ret


def formatStorage(value):
    value = long(float(value))
    last = 1024

    for p in ('B', 'K', 'M', 'G', 'T', 'P'):
        if value < last:
            break

        last *= 1024

    return ('%.1f' % (value * 1024.0 / last)) + p


def formatField(field, value):
    if field == value:
        return value

    if value is None:
        return ''

    if field in ('TMP', 'SPACE'):
        return formatStorage(value)

    if field in ('IDLE',):
        return '{0:.1f}%'.format(value)

    if field in ('NORMLOAD',):
        return '{0:.2f}%'.format(100 * value)

    if isinstance(value, float):
        return '%.2f' % value

    return str(value)


def collectData(samples):
    for host, res in samples:
        res['HOST'] = host

        yield res


def sortSamples(samples, field):
    def key(x):
        return x.get(field, None)

    return sorted(collectData(samples), key=key)


def allFields(data):
    ret = {}

    for x in data:
        for key, value in x.items():
            l = len(formatField(key, value))

            if key in ret:
                ret[key] = max(ret[key], l)
            else:
                ret[key] = max(max(l, len(key)), 8)

    return [x for x in ret.items()]


def formatString(fields):
    ret = ''
    i = 0

    for field, maxlen in fields[:-1]:
        ret += '{%d:>%d}| ' % (i, maxlen)
        i += 1

    ret += '{%d}' % i

    return ret


def getFields(item, fields):
    ret = []

    for field, maxlen in fields:
        ret.append(formatField(field, item.get(field, None)))

    return ret


def formatItem(item, fields):
    fmt = formatString(fields)
    fld = getFields(item, fields)

    return fmt.format(*fld)


def printTop(samples, sortby):
    if len(samples) == 0:
        return

    samples = sortSamples(samples, sortby)
    fields = allFields(samples)

    order = ['HOST', 'LOAD', 'TMP', 'SPACE', 'SOCKETS', 'IDLE', 'FAULTS', 'ERROR', 'NCPU', 'NORMLOAD']

    def orderkey(x):
        return order.index(x[0])

    fields = sorted(fields, key=orderkey)

    footer = {}

    for field, maxlen in fields:
        footer[field] = field

    print(formatItem(footer, fields))

    for item in samples:
        print(formatItem(item, fields))
