#!/skynet/python/bin/python

from __future__ import print_function, division, absolute_import

try:
    from pkg_resources import require
    _ = require
except ImportError:
    pass

import fcntl
import argparse
import os
import sys
import subprocess as subproc
import shutil
import cgi
import textwrap
import select
import time
import tarfile
import gevent
import gevent.socket
import gevent.select
import threading
from os import path

from kernel.util.errors import formatException

from cStringIO import StringIO


class BuildError(Exception):
    pass


class SubProcError(Exception):
    def __init__(self, msg, args, exitcode):
        self.args = args
        self.exitcode = exitcode
        Exception.__init__(self, msg)


class ColorStdStream(object):
    RESET_SEQ = '\033[0m'
    LIGHT_GREEN = '\033[1;32m'
    LIGHT_GRAY = '\033[0;37m'
    LIGHT_RED = '\033[1;31m'

    def wrap_stdout(self, s):
        return ''.join((
            self.LIGHT_GRAY,
            s,
            self.RESET_SEQ
        ))

    def wrap_stderr(self, s):
        return ''.join((
            self.LIGHT_RED,
            s,
            self.RESET_SEQ
        ))

    class _StreamProxy(object):
        def __init__(self, stream, wrapper):
            self.stream = stream
            self.wrapper = wrapper

        def write(self, s):
            self.stream.write(self.wrapper(s))
            self.flush()

        def fileno(self):
            return self.stream.fileno()

        def flush(self):
            self.stream.flush()

    def __enter__(self):
        self.stdout, self.stderr = sys.stdout, sys.stderr
        if self.stdout.isatty():
            sys.stdout = self._StreamProxy(self.stdout, self.wrap_stdout)
        if self.stderr.isatty():
            sys.stderr = self._StreamProxy(self.stderr, self.wrap_stderr)

    def __exit__(self, *args):
        sys.stdout, sys.stderr = self.stdout, self.stderr


class HtmlStream(ColorStdStream):
    def __init__(self, full_html=True, jsappend=False, stream=sys.stdout):
        self.full_html = full_html
        self.indent_level = 0
        self.need_indent = True
        self.jsappend = jsappend
        self.stream = stream

    def wrap_html(self, s, klass):
        result = []
        breaks = s.count('\n')
        for idx, line in enumerate(s.split('\n')):
            if line:
                result.append('<span class="%s">%s</span>' % (klass, cgi.escape(line)))
            if idx != breaks:
                result.append('<br/>\n')

        if self.jsappend:
            for idx, line in enumerate(result):
                if line and line[-1] == '\n':
                    add_br = True
                    line = line[:-1]
                else:
                    add_br = False

                result[idx] = '<script type="text/javascript">$(\'%s\').append(\'%s\')</script>' % (
                    self.jsappend,
                    line.replace("'", "\\'")
                )

                if add_br:
                    result[idx] += '\n'

        return ''.join(result)

    def wrap_stdout(self, s):
        return self.indent(self.wrap_html(s, 'stdout'))

    def wrap_stderr(self, s):
        return self.indent(self.wrap_html(s, 'stderr'))

    def indent(self, s):
        if not self.indent_level:
            return s

        if not self.need_indent:
            self.need_indent = s and s[-1] == '\n'
            return s

        res = '\n'.join([('        ' + l if l else '') for l in s.split('\n')])
        self.need_indent = res and res[-1] == '\n'
        return res

    def header(self):
        self.stream.write(textwrap.dedent('''
            <html>
                <head>
                    <style type="text/css">
                        body {
                            background: black;
                            color: white;
                            font-family: ProFontWindows, Dejavu Sans Mono, monospace;
                            font-size: 10px;
                        }
                        .stdout { white-space: pre; color: lightgreen; font-weight: normal }
                        .stderr { white-space: pre; color: red; font-weight: bold }
                    </style>
                </head>
                <body>
        ''').lstrip('\n'))
        self.indent_level = 2

    def footer(self):
        self.stream.write(textwrap.dedent('''
                </body>
            </html>
        ''').lstrip('\n'))
        self.indent_level = 0

    def __enter__(self):
        if self.full_html:
            self.header()
        self.stdout, self.stderr = sys.stdout, sys.stderr
        sys.stdout = self._StreamProxy(self.stream, self.wrap_stdout)
        sys.stderr = self._StreamProxy(self.stream, self.wrap_stderr)
        return self

    def __exit__(self, *args, **kwargs):
        if self.full_html:
            self.footer()

        super(HtmlStream, self).__exit__(*args, **kwargs)


class RemoteChild(object):
    def __init__(self, svnurl, svnrevision):
        assert svnurl
        assert svnrevision

        self.svnurl = svnurl
        self.svnrevision = svnrevision

    def sendmsg(self, data):
        import msgpack
        import struct

        msg = msgpack.dumps(('MSG', data))
        sys.stdout.write(struct.pack('!I', len(msg)))
        sys.stdout.write(msg)
        sys.stdout.flush()

    def cleanup(self):
        for f in os.listdir('.'):
            if f == '.downloads':
                try:
                    shutil.rmtree('.downloads/zeromq-arc')
                except Exception:
                    pass
                continue
            if f == '.ccache':
                continue
            if os.path.isdir(f):
                shutil.rmtree(f)
            else:
                os.unlink(f)

    def runproc(self, args, env=None):
        lk = threading.Lock()

        def _fw_stream(name, stream):
            while True:
                line = stream.readline()
                if not line:
                    break

                with lk:
                    self.sendmsg(('STREAM', name, line.strip()))

        if env is None:
            env = os.environ.copy()

        proc = subproc.Popen(args, stdout=subproc.PIPE, stderr=subproc.PIPE, env=env)

        thr1 = threading.Thread(target=lambda: _fw_stream('stdout', proc.stdout))
        thr1.start()

        thr2 = threading.Thread(target=lambda: _fw_stream('stderr', proc.stderr))
        thr2.start()

        proc.wait()
        if proc.returncode != 0:
            raise Exception('Process %r exited with %r' % (args, proc.returncode))

    def main(self):
        self.sendmsg(('INIT', ))

        data = sys.stdin.read()

        self.sendmsg(('BUILD', ))

        os.chdir('/var/tmp/skydeps_build')
        self.cleanup()

        with open('skynet_packages.tgz', 'wb') as f:
            f.write(data)

        self.runproc(['tar', '-xzf', 'skynet_packages.tgz'])

        if not os.path.exists('python-dists'):
            os.makedirs('python-dists')

        env = os.environ.copy()
        if self.svnurl and self.svnrevision:
            env['SVNURL'] = self.svnurl
            env['SVNREVISION'] = self.svnrevision

        self.runproc(['./build.py'], env=env)

        try:
            distname = os.listdir('python-dists')[0]
        except (OSError, IndexError):
            distname = None

        self.sendmsg(('RESULT', distname, open(os.path.join('python-dists', distname), 'rb').read()))

        os.chdir('/var/tmp/skydeps_build')
        self.cleanup()

        self.sendmsg(('DONE', ))


def pipe_reader(pipe, stream):
    while 1:
        data = pipe.readline()
        if not data:
            break
        stream.write(data)


def run_subproc(args, catch_stdout=False, passthru_stdout=True, env=None):
    proc = subproc.Popen(args, stdout=subproc.PIPE, stderr=subproc.PIPE, close_fds=True, env=env)
    poller = select.poll()
    poller_fds = {}

    if catch_stdout:
        stdout = []

    for stream, target_stream in ((proc.stdout, sys.stdout), (proc.stderr, sys.stderr)):
        fcntl.fcntl(
            stream.fileno(),
            fcntl.F_SETFL,
            fcntl.fcntl(
                stream.fileno(),
                fcntl.F_GETFL
            ) | os.O_NONBLOCK
        )
        poller.register(stream.fileno(), select.POLLIN)
        poller_fds[stream.fileno()] = target_stream

    while 1:
        for fd, ev in poller.poll():
            if ev & select.POLLIN:
                data = os.read(fd, 4096)

                if len(data) > 0:
                    if poller_fds[fd] == sys.stdout:
                        if passthru_stdout:
                            poller_fds[fd].write(data)
                            poller_fds[fd].flush()
                        if catch_stdout:
                            stdout.append(data)
                    else:
                        poller_fds[fd].write(data)
                        poller_fds[fd].flush()
                else:
                    # as well as OpenBSD and some other OSes, cygwin does not set POLLHUP event in case of EOF
                    # so we set it here manually for further processing
                    ev |= select.POLLHUP

            if ev & select.POLLHUP:
                poller.unregister(fd)
                poller_fds.pop(fd)

        if not poller_fds:
            break

    proc.wait()
    if catch_stdout:
        return proc, ''.join(stdout)
    return proc


def hostlog(host, msg, *args, **kwargs):
    err = kwargs.pop('err', False)
    stream = kwargs.pop('stream', None)
    if stream:
        file_ = stream
        kwargs['nohost'] = True
    else:
        file_ = sys.stdout if not err else sys.stderr

    if kwargs.pop('nohost', False):
        print(msg % args, file=file_, **kwargs)
    else:
        print('%-12s: %s' % (host, msg % args, ), file=file_, **kwargs)


def build_remote_host_ssh(host, data, svnurl, svnrevision, output, logfile=None):
    import msgpack
    import struct

    user = 'robot-skybot'

    try:
        if logfile:
            logfp = open(logfile, 'w')
            hstream = HtmlStream(stream=logfp)
            hstream.header()

            def this_hostlog(host, msg, *args, **kwargs):
                err = kwargs.pop('err', False)
                wrapper = hstream.wrap_stdout if not err else hstream.wrap_stderr
                hstream.stream.write(wrapper(msg % args))
                hstream.stream.write(wrapper(kwargs.pop('end', '\n')))
                hstream.stream.flush()
        else:
            this_hostlog = hostlog

        proc = subproc.Popen([
            'ssh', '-l', user,
            host,
            'mkdir -p /var/tmp/skydeps_build && '
            'cat > /var/tmp/skydeps_build/build.py && '
            'chmod +x /var/tmp/skydeps_build/build.py'
        ], stdin=subproc.PIPE)

        proc.stdin.write(open(sys.argv[0], mode='rb').read())
        proc.stdin.close()
        proc.wait()

        if proc.returncode != 0:
            raise Exception('proc #1: return code %r' % (proc.returncode, ))

        proc = subproc.Popen([
            'ssh', '-l', user,
            host,
            'cd /var/tmp/skydeps_build && '
            './build.py --remote-child '
            '--remote-child-svnurl "%s" '
            '--remote-child-svnrev "%s"' % (svnurl, svnrevision),
        ], stdin=subproc.PIPE, stdout=subproc.PIPE)

        def getmsg():
            gevent.socket.wait_read(proc.stdout.fileno())
            llen = proc.stdout.read(4)

            if len(llen) < 4:
                raise Exception('EOF')

            msglen = struct.unpack('!I', llen)[0]

            gevent.socket.wait_read(proc.stdout.fileno())
            raw = proc.stdout.read(msglen)
            if len(raw) < msglen:
                raise Exception('EOF')

            msg = msgpack.loads(raw)

            assert msg[0] == 'MSG', 'Invalid message: %r' % (msg, )
            return msg[1]

        msg = getmsg()
        assert msg[0] == 'INIT', 'Invalid msg %r' % (msg, )

        this_hostlog(host, 'Sending data...')

        proc.stdin.write(data)
        proc.stdin.close()

        assert getmsg()[0] == 'BUILD'

        this_hostlog(host, 'Builder started')

        distrib = None
        distrib_name = None

        while True:
            raw = getmsg()
            typ = raw[0]
            msg = raw[1:]

            if typ == 'EOF':
                raise Exception('EOF')

            if typ == 'STREAM':
                if (msg[0] == 'stdout'):
                    this_hostlog(host, msg[1])
                else:
                    this_hostlog(host, msg[1], err=True)

            if typ == 'RESULT':
                distrib_name, distrib = msg[0:2]

            if typ == 'DONE':
                break

        if distrib_name:
            if not os.path.exists(output):
                os.makedirs(output)

            this_hostlog(host, 'writing data to %s/%s...', output, distrib_name)
            open('%s/%s' % (output, distrib_name), 'wb').write(distrib)
            this_hostlog(host, 'done!')
            return True

        return

    finally:
        if logfile:
            hstream.footer()
            logfp.close()


def build_remote_ssh(hosts, data, svnurl, svnrevision, output, logdir):
    grns = []
    some_failed = False

    for host in hosts:
        grns.append((
            host,
            gevent.spawn(
                build_remote_host_ssh,
                host, data,
                svnurl, svnrevision,
                output,
                os.path.join(logdir, host + '.buildlog.html') if logdir else None
            )
        ))

    for host, grn in grns:
        try:
            res = grn.get()
            if not res:
                some_failed = True
        except Exception:
            hostlog(host, 'remote build greenlet failed: %s', formatException(), err=True)
            some_failed = True

    if some_failed:
        return 1

    return 0


def selective_checkout(name, arcadia_path, deps=(), rev=1):

    print('Checking out {0} from {1}'.format(name, arcadia_path))

    root = path.join(os.getcwd(), 'libraries', name)

    # ensure root is clean
    if path.exists(root):
        shutil.rmtree(root)
    os.makedirs(root)

    def run(*args):
        print('  executing: %s' % (' '.join(args), ))
        return run_subproc(args, catch_stdout=True, passthru_stdout=False)[1]

    # bootstrap ya
    yatool = path.join(root, 'yatool')
    run(
        'svn', 'export',
        '-r8166323',
        'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya',
        yatool
    )
    os.chmod(yatool, 0o755)

    # bootstrap arcadia
    arcadia = path.join(root, 'arcadia')
    run(
        yatool,
        'clone',
        '-r%d' % (rev, ),
        arcadia
    )

    # selective checkout: this command doesn't build anything
    run(
        yatool,
        'make', '--checkout', '-DALLOCATOR=SYSTEM',
        path.join(arcadia, arcadia_path),
        '-j0'
    )

    # checkout additional non-referenced deps (temporary workaround)
    for dep in deps:
        run(
            yatool,
            'make', '-xx', '--checkout', '-DALLOCATOR=SYSTEM',
            path.join(arcadia, dep),
            '-j0'
        )


def prepare():
    env = os.environ.copy()
    env.pop('LD_LIBRARY_PATH', None)

    deps_netlibus = (
        'contrib/tools/python',
        'contrib/tools/ragel6',
        'contrib/libs/cppdemangle',       # required coz we are checkouting on linux, but building on darwin
        'contrib/libs/asmglibc',          # same as above
    )

    deps_tvm = deps_netlibus + (
        'tools/fix_elf',
        'build/platform/python',
        'contrib/libs/protoc',
        'contrib/libs/protobuf',
        'contrib/python/enum34',
        'library/cpp/malloc/system',
        'library/cpp/getopt/small',
        'library/cpp/cppparser',
        'tools/rescompiler',
        'tools/enum_parser/parse_enum',
    )

    selective_checkout('netlibus', 'infra/netlibus/pymodule', deps=deps_netlibus, rev=8166323)

    selective_checkout('ticket_parser2', 'library/python/deprecated/ticket_parser2/so', deps=deps_tvm, rev=8166323)


def build(args):
    os.chdir(os.path.dirname(__file__))

    if not args.remote_run:
        if sys.platform == 'darwin':
            os.environ['PATH'] = os.getenv('PATH', '') + ':/usr/local/bin'

        print('Running local build')

        print('-' * 72)

        args = ['./make-python.sh']

        proc = run_subproc(args)
        print('-' * 72)

        if proc.returncode != 0:
            print('Build failed with exit code: %d' % (proc.returncode, ), file=sys.stderr)
            return 1
        else:
            print('Build completed with exitcode: 0')
            return 0
    else:
        print('Running remote build')

        prepare()

        if os.path.exists('.svn') or os.path.exists('../.svn') or os.path.exists('../../.svn'):
            env = os.environ.copy()
            env.pop('LD_LIBRARY_PATH', None)

            env['LANG'] = env['LANGUAGE'] = 'C'  # Force svn/hg to print english messages

            proc, stdout = run_subproc(
                ['svn', 'ls', '--depth', 'infinity', '.'],
                catch_stdout=True, passthru_stdout=False,
                env=env
            )
            filelist = stdout.strip().split('\n')

            proc, svninfo = run_subproc(
                ['svn', 'info', '.'], catch_stdout=True, passthru_stdout=False, env=env
            )

        elif os.path.exists('.hg') or os.path.exists('../.hg'):
            proc, stdout = run_subproc(
                ['hg', 'status', '-ncma', '.'],
                catch_stdout=True, passthru_stdout=False
            )
            filelist = stdout.strip().split('\n')

            proc, svninfo = run_subproc(
                ['hg', 'svn', 'info'], catch_stdout=True, passthru_stdout=False
            )
        elif (
            os.path.exists('../../.arc')  # mountpoint/skynet/packages
            or os.path.exists('../../../.arc')  # mountpoint/trunk/skynet/packages
            or os.path.exists('../../../../../.arc')  # mountpoint/branches/skynet/release-18.0/skynet/packages
        ):
            proc, stdout = run_subproc(
                ['arc', 'ls-files', '.'],
                catch_stdout=True, passthru_stdout=False,
            )
            filelist = stdout.strip().split('\n')

            proc, svninfo = run_subproc(
                ['arc', 'info'],
                catch_stdout=True, passthru_stdout=False,
            )

        else:
            print('Error: dont know how to grab filelist here (hg or svn or arc not found)', file=sys.stderr)
            return 1

        svnurl = None
        svnrevision = None

        filelist.append('libraries/netlibus')
        filelist.append('libraries/ticket_parser2')

        for line in svninfo.split('\n'):
            if ': ' not in line:
                continue
            key, value = line.split(': ', 1)
            if key == 'URL':
                svnurl = value.strip()
            elif key == 'Last Changed Rev':
                svnrevision = value.strip()
            elif key == 'hash' and svnurl is None:
                if svnrevision is None:
                    svnrevision = value.strip()
                svnurl = 'arcadia-arc:/#' + value.strip()

        print('Building svn: %r:%r' % (svnurl, svnrevision))

        tarfile.grp = tarfile.pwd = None

        ts = time.time()

        datatar_fp = StringIO()
        datatar = tarfile.open(fileobj=datatar_fp, mode='w|gz')

        try:
            for fn in filelist:
                datatar.add(fn)
        finally:
            datatar.close()

        print('Created data tar (%d bytes) in %0.4fs' % (datatar_fp.tell(), time.time() - ts))

        if args.html:
            logdir = args.logdir
        else:
            logdir = None

        return build_remote_ssh(
            args.remote_run,
            datatar_fp.getvalue(),
            svnurl, svnrevision,
            output=args.output,
            logdir=logdir
        )


def comma_or_space_separated_list(string):
    sep = None
    strip = False

    for psep in ',', ';':
        if psep in string:
            sep = psep
            strip = True
            break

    res = string.split(sep)
    if strip:
        return [s.strip() for s in res]
    return res


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-R', '--remote-run', type=comma_or_space_separated_list)
    parser.add_argument('--html', action='store_true')
    parser.add_argument('--logdir', action='store', default=os.curdir)
    parser.add_argument('--output', action='store', default='python-dists')
    parser.add_argument('--prepare', action='store_true')
    parser.add_argument('--remote-child', action='store_true')
    parser.add_argument('--remote-child-svnurl')
    parser.add_argument('--remote-child-svnrev')

    args = parser.parse_args()

    if args.html:
        sys.stderr = sys.stdout
        if args.remote_run:
            with HtmlStream(full_html=False, jsappend='#common_log') as writer:
                writer.stream.write(textwrap.dedent('''
                    <html>
                        <head>
                            <style type="text/css">
                                body {
                                    font-size: 10px;
                                    font-family: ProFontWindows, Dejavu Sans Mono, monospace;
                                    background: black;
                                    color: white
                                }
                                .stdout { white-space: pre; color: lightgreen; font-weight: normal; }
                                .stderr { white-space: pre; color: red; font-weight: bold; }
                                .commonlog {
                                    border-bottom: 2px solid gray;
                                    padding: 0.5em;
                                    margin-bottom: 1em;
                                }
                                .hostlog {
                                    height: 400px;
                                    width: 390px;
                                    float: left;
                                    border: 1px solid gray;
                                    padding: 1em
                                }
                                .hostlog .log {
                                    height: 100%;
                                    overflow: hidden;
                                }
                                .hostlog > .title > a {
                                    font-weight: bold;
                                    color: white;
                                }
                            </style>
                            <script src="https://ajax.googleapis.com/ajax/libs/jquery/1.7.2/jquery.js"></script>
                        </head>
                        <body>
                            <div id="common_log" class="commonlog"></div>
                ''').lstrip('\n'))
                for host in args.remote_run:
                    writer.stream.write(
                        ' ' * 8 +
                        '<div id="%s_log" class="hostlog">'
                        '<span class="title"><a href="%s.buildlog.html">%s</a></span>'
                        '<div class="log"></div>'
                        '</div>\n' % (host, host, host)
                    )

                writer.indent_level = 2

                writer.stream.write(writer.indent(textwrap.dedent('''
                    <script type="text/javascript">
                        finished = false

                        function tailer() {
                            $('div.hostlog').each(function(idx, el) {
                                var host = /(.*)_log/.exec(el.id)[1]
                                var hostfile = host + '.buildlog.html'

                                tail(hostfile, $(el), 0)
                            })
                        }

                        function tail(doc, target, prevlen) {
                            target.find('> .log').load(
                                doc, function(response, status, xhr) {
                                    var reload_in = 500
                                    if (status != 'success') {
                                        reload_in = 3000
                                        target.find('> .log').empty()
                                        target.find('> .log').append('<span class="stderr">error</span>')
                                    }

                                    target.find('> .log').scrollTop(999999)
                                    if (!finished) {
                                        setTimeout(function() { tail(doc, target, 0) }, reload_in)
                                    }
                                }
                            )
                        }

                        tailer()
                    </script>
                ''').lstrip('\n')))

                writer.stream.flush()

                res = build(args)

                writer.stream.write(textwrap.dedent('''
                            <script type="text/javascript">finished = true</script>
                        </body>
                    </html>
                ''').lstrip('\n'))

                return res
        else:
            with HtmlStream():
                return build(args)
        # This indicated what build func failed with unhandled
        # exception, so we should return non-zero exit code
        return 1

    elif args.remote_child:
        return RemoteChild(args.remote_child_svnurl, args.remote_child_svnrev).main()

    else:
        with ColorStdStream():
            try:
                if args.prepare:
                    prepare()
                    return 0
                else:
                    return build(args)
            except KeyboardInterrupt:
                print('Interrupted', file=sys.stderr)
                return 1


if __name__ == '__main__':
    raise SystemExit(main())
