import argparse
import subprocess as subproc
import py
import re
import os

from pprint import pprint as pp  # noqa


GOOD_LIBS = {
    '/System/Library/Frameworks/ApplicationServices.framework/Versions/A/ApplicationServices',
    '/System/Library/Frameworks/Carbon.framework/Versions/A/Carbon',
    '/System/Library/Frameworks/CoreFoundation.framework/Versions/A/CoreFoundation',
    '/System/Library/Frameworks/CoreServices.framework/Versions/A/CoreServices',
    '/System/Library/Frameworks/Foundation.framework/Versions/C/Foundation'
    '/System/Library/Frameworks/IOKit.framework/Versions/A/IOKit',
    '/System/Library/Frameworks/SystemConfiguration.framework/Versions/A/SystemConfiguration',
    '/System/Library/Frameworks/Tcl.framework/Versions/8.5/Tcl',
    '/System/Library/Frameworks/Tk.framework/Versions/8.5/Tk',
    '/usr/lib/libgcc_s.1.dylib',
    '/usr/lib/libstdc++.6.dylib',
    '/usr/lib/libresolv.9.dylib',
    '/usr/lib/libSystem.B.dylib',
}


def run_otool(path):
    proc = subproc.Popen(['otool', '-L', path.strpath], stdout=subproc.PIPE)
    data = proc.stdout.read()
    proc.wait()

    if 'is not an object file' in data:
        return False

    libs = set()

    for line in data.split('\n'):
        if not line.startswith('\t'):
            continue
        lib = re.search('^\t([0-9a-zA-Z\.\_\-\+\@\\\/]+) \(', line)
        if not lib:
            print('not matched', line)
        else:
            library = lib.groups()[0]
            if library in GOOD_LIBS:
                continue
            if '@loader_path' in library:
                continue
            library = py.path.local(library)
            if library.check(exists=0):
                print('Unable to find library %s (in %s)' % (library, path))
            else:
                libs.add(library)

    return libs


def fix_rpath(paths, library, libdir):
    print('Fixing rpath for library %s for %d items' % (library, len(paths)))

    if 'ports/lib' in library.strpath:
        newpath = library.strpath[library.strpath.find('ports/lib'):]
        newpath = newpath[6:]

        if 'lib/libgcc/' in newpath:
            pos = newpath.find('lib/libgcc/')
            newpath = newpath[:pos] + newpath[pos + 10:]

        newlibrary = libdir.join(newpath)
    else:
        newlibrary = library

    for path in paths:
        print('  %s' % (path, ))
        path.chmod(path.stat().mode | 0o600)
        relpath = '@loader_path/' + path.dirpath().bestrelpath(newlibrary)
        print('     old: %s' % (library, ))
        print('     new: %s' % (relpath, ))
        os.system('install_name_tool -change "%s" "%s" "%s"' % (library, relpath, path))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--libdir')
    parser.add_argument('--bindir')

    args = parser.parse_args()

    fix_libs = {}

    for path in py.path.local(args.bindir).visit():
        if not path.check(file=1):
            continue

        libs = run_otool(path)
        if libs:
            for lib in libs:
                fix_libs.setdefault(lib, []).append(path)

    for path in py.path.local(args.libdir).visit():
        if not path.check(file=1):
            continue
        if path.check(link=1):
            continue
        if '.' not in path.strpath:
            continue
        ext = path.strpath.rsplit('.', 1)[1]
        if ext not in ('dylib', 'so'):
            continue

        libs = run_otool(path)
        if libs:
            for lib in libs:
                fix_libs.setdefault(lib, []).append(path)

    for library, paths in sorted(fix_libs.iteritems()):
        fix_rpath(paths, library, libdir=py.path.local(args.libdir))

    return 0


if __name__ == '__main__':
    raise SystemExit(main())
