copy_missing_libraries() {
    info "Copy missing libraries"
    push

    mkdir -p "${LIBDIR}"

    # Little bit tricky, but we need libstdc++ ship with us
    if [ $(uname) = "FreeBSD" ]; then
        if $(command -v gcc44 >/dev/null 2>&1); then
            run cp /usr/local/lib/gcc44/libstdc++.so.6 "${LIBDIR}"
        else
            run cp /usr/local/lib/gcc47/libstdc++.so.6 "${LIBDIR}"
        fi
    elif [ "$(uname -sp)" = "Darwin arm" ]; then
        info "Not looking for libstdc++ on Darwin arm64"
    elif [ "$(uname)" = "Darwin" ] || [ "$(uname)" = "Linux" ]; then
        run rm -f "${LIBDIR}/"libstdc++.so*
        echo 'int main() {return 0;}' > "${WORKDIR}/findstdc++.cpp"
        if [ "$(uname)" != "Darwin" ]; then LINKOPT="-Wl,--no-as-needed"; fi
        ${CXX} ${LINKOPT} -lstdc++ -o "${WORKDIR}/findstdc++" "${WORKDIR}/findstdc++.cpp"
        if [ "$(uname)" = "Darwin" ]; then
            libstdcpppath=$(otool -L "${WORKDIR}/findstdc++" | grep -v findstd | grep libc++ | awk '{print $1}')
        else
            libstdcpppath=$(ldd "${WORKDIR}/findstdc++"  | grep stdc++ | awk '{ print $3 }')
        fi
        if [ -z "${libstdcpppath}" ]; then
            echo "Cant locate libstdc++.so"
            exit 1
        fi
        run rm "${WORKDIR}/findstdc++" "${WORKDIR}/findstdc++.cpp"
        run install -m 0755 "${libstdcpppath}" "${LIBDIR}"
        info "Found libstdc++ at ${libstdcpppath}"
    fi

    # if [ $(uname) = "Darwin" ]; then
    #     run install -m 0755 "/Users/skynet/ports/lib/libintl.8.dylib" "${LIBDIR}"
    #     run install -m 0755 "/Users/skynet/ports/lib/libglib-2.0.0.dylib" "${LIBDIR}"
    #     run install -m 0755 "/Users/skynet/ports/lib/libgcc/libgcc_s.1.dylib" "${LIBDIR}"
    #     run install -m 0755 "/Users/skynet/ports/lib/libgcc/libstdc++.6.dylib" "${LIBDIR}"
    # fi
    pop
    info "Done"
}


little_test() {
    cd "${WORKDIR}/skynet/python"

    # Now few tests to ensure everything needed is compiled fine
    info_c "Checking python module zlib (zlib.so)"
    bin/python -c 'import zlib'

    info_c "Checking python module bz2 (bz2.so)"
    bin/python -c 'import bz2'

    info_c "Checking python module readline (readline.so)"
    bin/python -c 'import readline'

    info_c "Checking python module ncurses (_curses.so)"
    bin/python -c "import curses"

    info_c "Checking python module sqlite3 (_sqlite3.so)"
    bin/python -c 'import _sqlite3; import sqlite3'

    if [ "$(uname -sp)" != "Darwin arm" ] && [ "$(uname -p)" != "aarch64" ]; then
        info_c "Checking python module \"Crypto.PublicKey._fastmath\" (_fastmath library)"
        bin/python -c 'from Crypto.PublicKey import _fastmath'
    fi

    info_c "Checking python module \"lxml.etree\" (lxml library)"
    bin/python -c '__import__("pkg_resources").require("lxml"); import lxml.etree'

    if [ "$(uname)" = "Linux" ] || [ "$(uname)" = 'FreeBSD' ]; then
        info_c "Checking iconv support in \"lxml.etree\""
        bin/python -c 'from lxml import etree; etree.XML("<?xml version=\"1.0\" encoding=\"cp1251\"?><root></root>")'
    fi

    info_c "Checking python module \"yaml._yaml\" (yaml C extension)"
    bin/python -c 'from yaml import _yaml, cyaml'
}


update_shebangs() {
    info "Fixing shebangs"
    push

    for f in $(ls ${BINDIR}); do
        f="$BINDIR/$f"
        if [ -x "$f" ] && [ -f "$f" ] && [ ! -z "$(file $f | grep script)" ]; then
            if [[ "$(head -n1 $f)" = "#!${BINDIR}/python"* ]] ||
               [[ "$(head -n1 $f)" = "#!/bin/python"* ]]; then
                info "Fix file $f"
                cp -p "$f" "$f"__new_shebang
                echo "#!/skynet/python/bin/python" > "$f"__new_shebang
                tail -n +2 "$f" >> "$f"__new_shebang
                mv -f "$f"__new_shebang "$f"
            fi
        fi
    done

    pop
    info "Done"
}


remake_pyco() {
    cd "${WORKDIR}/skynet"

    info "Drop old *.py[co] files..."
    find ./ -iname '*.py[co]' -delete

    # Alternative to this is to provide -x '\/test\/' for compileall
    info "Drop test folders (they won't compile)"
    push

    while read path; do
        info "rm -rf $path..."
        rm -rf "$path"
    done < <(find "python/lib/python${PYTHONMAJORVERSION}" -type d '(' -iname test -or -iname tests ')' -print)

    pop
    info "Done"

    info "Compile pyc..."
    "$BINDIR/python" -m compileall -d skynet -q -f .
}

detect_version() {
    if [ -z "$SVNURL" ] && [ -z "$SVNREVISION" ]; then
        local svninfo
        svninfo=$(svn info; true)
        if [ -z "$svninfo" ]; then
            local _bk=$PYTHONNOUSERSITE
            unset PYTHONNOUSERSITE
            svninfo=$(hg svn info .; true)
            if [ -z "$svninfo" ]; then
                echo "Unable to determine svn revision!" >&2
                exit 1
            fi
            export PYTHONNOUSERSITE=$_bk

        fi

        BUILD_SVN_URL=$(echo "$svninfo" | grep -m 1 URL | awk -F ': ' '{ print $2 }')
        BUILD_SVN_REVISION=$(echo "$svninfo" | grep Last\ Changed\ Rev | awk -F ': ' '{ print $2 }')
    else
        BUILD_SVN_URL="$SVNURL"
        BUILD_SVN_REVISION="$SVNREVISION"
    fi
    BUILD_VERSION="$BUILD_SVN_URL@$BUILD_SVN_REVISION"

    info "Build info"
    info "    Url      : $BUILD_SVN_URL"
    info "    Revision : $BUILD_SVN_REVISION"
    info "    Version  : $BUILD_VERSION"
}

add_extra() {
    cp "$ROOT/build_v2/extra/bin/py27gdb" "$PREFIX/bin/"

    mkdir -p "$PREFIX/etc/gdb"
    cp "$ROOT/build_v2/extra/etc/gdb/py27init" "$PREFIX/etc/gdb/"
    cp "$ROOT/build_v2/extra/etc/gdb/py27cmd" "$PREFIX/etc/gdb/"

    local CAFILE=$LIBDIR/python${PYTHONMAJORVERSION}/site-packages/certifi/cacert.pem

    set +e
    grep 'Yandex Internal' "$CAFILE" >/dev/null
    local exist=$?
    set -e

    if [ ! $exist -eq 0 ]; then
        {
            echo ''
            echo 'Yandex Internal'
            echo '==============='
            cat "$ROOT/build_v2/extra/YandexInternalRootCA.crt"
        } >> "$CAFILE"
    fi
}

make_distributive() {
    cd "${CWD}"

    mkdir -p python-dists

    {
        echo "Version: $BUILD_VERSION"
        echo "SVN:"
        echo "  URL: $BUILD_SVN_URL"
        echo "  Revision: $BUILD_SVN_REVISION"
    } > "${WORKDIR}/skynet/python/.info"

    local DIST=skynet-deps_${ARCH}.tar.gz
    run tar -czf "python-dists/${DIST}" -C "${WORKDIR}/skynet" $(ls "${WORKDIR}/skynet")
    info "Done: python-dists/${DIST}"
}

cleanup_workdir() {
    run rm -rf "${WORKDIR}"
}


fixrpath() {
    if [ "$(uname)" = Darwin ] || [[ "$(uname)" = 'CYGWIN'* ]]; then
        return
    fi
    "$PREFIX/bin/chrpath" -r '$ORIGIN/../lib' "$BINDIR"/python* >/dev/null 2>&1 || echo

    "$PREFIX/bin/python2" - "$BINDIR/chrpath" "$LIBDIR" "$BINDIR" << PYEND
from __future__ import print_function

import py
import sys
import os
import subprocess as subproc

chrpath = py.path.local(sys.argv[1]).realpath()
libdir = py.path.local(sys.argv[2]).realpath()
bindir = py.path.local(sys.argv[3]).realpath()

def fixrpath(libdir, path):
    seen = set()
    for item in path.visit():
        if not item.check(file=1) or item.check(link=1):
            continue

        if not os.access(item.strpath, os.X_OK):
            continue

        if item.ext in ('.py', '.pyc', '.pyo'):
            continue

        item = item.realpath()
        if item.strpath in seen:
            continue

        seen.add(item.strpath)

        relpath = item.bestrelpath(libdir)
        if relpath == '..':
            relpath = '\$ORIGIN'
        elif relpath.startswith('../'):
            relpath = '\$ORIGIN/' + relpath[3:]

        item.chmod(0o755)
        
        args = [chrpath.strpath, '-r', relpath, item.strpath]
        proc = subproc.Popen(
            args,
            stdout=subproc.PIPE, stderr=subproc.STDOUT
        )
        proc.stdout.read()
        proc.wait()

fixrpath(libdir, bindir)
fixrpath(libdir, libdir)
PYEND
}

fix_darwin_rpath() {
    if [ "$(uname)" != "Darwin" ]; then
        return
    fi

    # special case for openssl, which builds in absolute path to /skynet starting from 1.1
    for openssllib in libssl.1.1.dylib libcrypto.1.1.dylib; do
        find "$LIBDIR/" -name "*.dylib" -o -name "*.so" -mindepth 1 -maxdepth 1 -exec install_name_tool -change "/skynet/python/lib/$openssllib" "@loader_path/$openssllib" {} \;
        find "$LIBDIR/" -name "*.dylib" -o -name "*.so" -mindepth 2 -maxdepth 2 -exec install_name_tool -change "/skynet/python/lib/$openssllib" "@loader_path/../$openssllib" {} \;
        find "$LIBDIR/" -name "*.dylib" -o -name "*.so" -mindepth 3 -maxdepth 3 -exec install_name_tool -change "/skynet/python/lib/$openssllib" "@loader_path/../../$openssllib" {} \;
        find "$LIBDIR/" -name "*.dylib" -o -name "*.so" -mindepth 4 -maxdepth 4 -exec install_name_tool -change "/skynet/python/lib/$openssllib" "@loader_path/../../../$openssllib" {} \;
        find "$LIBDIR/" -name "*.dylib" -o -name "*.so" -mindepth 5 -maxdepth 5 -exec install_name_tool -change "/skynet/python/lib/$openssllib" "@loader_path/../../../../$openssllib" {} \;
    done
    # no idea, why we have to do it again and explicitly here, but we do
    install_name_tool -change "/skynet/python/lib/libcrypto.1.1.dylib" "@loader_path/libcrypto.1.1.dylib" "$LIBDIR/libssl.1.1.dylib"

    "$BINDIR/python" "$ROOT/build_v2/_fix_darwin_rpath.py" --bindir "$BINDIR" --libdir "$LIBDIR"
}
