# IMPORTS {{{
from __future__ import print_function

__import__('pkg_resources').require('PyYAML', 'msgpack-python')

import argparse
import contextlib
import errno
import fcntl
import grp
import itertools
import msgpack
import os
import pwd
import py
import py.error
import Queue
import re
import requests
import select
import signal
import simplejson
import socket
import stat as osstat
import subprocess as subproc
import sys
import tarfile
import tempfile
import textwrap
import threading
import datetime
import time
import six
import yaml
# IMPORTS }}}

# HACKS {{{
_tarfile_copyfileobj = tarfile.copyfileobj


fsync_queue = Queue.Queue()
fsync_threads = []
fsync_read_binary_mode = 'rb+' if sys.platform == 'cygwin' else 'rb'


def _copyfileobj(src, dst, length=None):
    ret = _tarfile_copyfileobj(src, dst, length)
    fsync_queue.put(dst.name)
    return ret


def _fsync_thread():
    while True:
        fn = fsync_queue.get()
        if fn is None:
            return
        try:
            with open(fn, fsync_read_binary_mode) as fp:
                os.fsync(fp.fileno())
        except IOError:
            pass


if hasattr(os, 'fsync'):
    tarfile.copyfileobj = _copyfileobj
    for i in range(16):
        fsync_thread = threading.Thread(target=_fsync_thread)
        fsync_thread.daemon = True
        fsync_thread.start()
        fsync_threads.append(fsync_thread)

# HACKS }}}


# GLOBAL DEFINES AND SETTINGS {{{
SKYNET_BIN_GENERATION = 4
FN_DEPS = 'skynet-deps_%s.tgz'
FN_SKYNET = 'skynet.tgz'
FN_SKYCORE = 'skycore.tgz'
FN_SKYINSTALL = 'skyinstall.py'
FN_DOCS = 'docs.tgz'
FN_IGNORE = (
    'services',
    'skyinstall.py',
    'selftest.py',
)
HB_REPORT_SEND_TIMEOUT = 30
HB_REST_API_BASE_URL = 'http://api.heartbeat.yandex.net/api/v1.0'
HB_REST_API_SERVER_ID_HDR = 'X-Heartbeat-Server'
PRIMARY_VERSION_META = ['version', 'revision', 'url', 'date', 'installed_by', 'skynet.bin gen', 'skycore rev']

SHEBANG_TPL = '#!/skynet/python/bin/python'

ALL_HOSTS = object()
FAIL_REASON = None
NO_LOCK_SKYCORE = True
RESTART_SKYCORE_SERVICES = ''
# GLOBAL DEFINES AND SETTINGS }}}


class HelpFormatter(argparse.RawDescriptionHelpFormatter, argparse.ArgumentDefaultsHelpFormatter):  # {{{
    pass
# }}}


class Version(object):  # {{{
    """
    The class has been copied from
        svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/infra/gosky/src/gosky/version.py
    with light modifications (`Version` and `LooseVersion` classes has been merged into single one).
    See the original version for documentation, comments and tests.
    """
    component_re = re.compile(r'(\d+ | [a-z]+ | \.)', re.VERBOSE)

    def __init__(self, vstring=None):
        if vstring:
            self.parse(vstring)
        else:
            self.version, self.vstring = [], None

    def __cmp__(self, other):
        return self._cmp(other)

    def __str__(self):
        return self.vstring

    def __repr__(self):
        return "Version ('%s')" % str(self)

    def parse(self, vstring):
        self.vstring = vstring
        components = [x for x in self.component_re.split(vstring) if x and x not in ('.', '-', '_')]
        for i, obj in enumerate(components):
            try:
                components[i] = int(obj)
            except ValueError:
                pass
        self.version = components

    def _cmp(self, other):
        if isinstance(other, str):
            other = Version(other)
        if self.version == other.version:
            return 0

        for idx in range(max(len(self.version), len(other.version))):
            v = [x[idx] if idx < len(x) else None for x in (self.version, other.version)]

            lower = -1
            for (a, b) in (v, reversed(v)):
                if a is None:
                    a = 0
                if b is None:
                    b = 0
                if a == b:
                    break

                higher = lower * -1
                if type(a) == type(b):
                    return lower if a < b else higher
                if isinstance(a, int):
                        return higher
                lower *= -1

        return 0
# }}}


class Path(py.path.local):  # {{{
    class Checkers(py.path.local.Checkers):
        def listable(self):
            """ Check what we can list directory contents. """
            try:
                self.path.listdir()
            except:
                return False

            return True

        def changeable(self):
            """ Check what we can create/remove files and dirs here. """
            if not self.path.check(dir=1):
                return False

            namer = tempfile._RandomNameSequence()

            check_file = None
            for seq in range(100):
                check_file = self.path.join('check' + namer.next())
                if not self.path.check(listable=1):
                    break
                if check_file.check(exists=1):
                    check_file = None
                else:
                    break

            if check_file is None:
                return False

            try:
                check_file.ensure(file=1).remove()
            except:
                return False

            return True

    class PathConverter(object):
        def __init__(self, path):
            self._path = path
            self._old_strpath = None

        def __enter__(self):
            if sys.platform == 'cygwin':
                    self._old_strpath = self._path.strpath
                    self._path.strpath = self._path.strpath.replace('/', '\\')

        def __exit__(self, exc_type, exc_val, exc_tb):
            if self._old_strpath:
                self._path.strpath = self._old_strpath

    def ensure_perms(self, uid, gid, chmod, mask=None):
        stat = self.lstat()
        errors = []

        # avoid second stat() call
        islink = osstat.S_ISLNK(stat.mode)

        if not islink:
            if isinstance(chmod, (list, tuple)):
                chmod_x, chmod_n = chmod
            else:
                chmod_x, chmod_n = (chmod, chmod)

            if mask is not None:
                chmod_x = chmod_x - (chmod_x & mask)
                chmod_n = chmod_n - (chmod_n & mask)

        if stat.uid != uid or stat.gid != gid:
            try:
                with Path.PathConverter(self):
                    if islink:
                        os.lchown(self.strpath, uid, gid)
                    else:
                        self.chown(uid, gid)
            except Exception as err:
                errors.append(
                    '%s has invalid owner/group %d:%d, cant set to %d:%d: %s' % (
                        self, stat.uid, stat.gid, uid, gid, err
                    )
                )

        if not islink:
            if stat.mode & 0o100:
                chmod = chmod_x
            else:
                chmod = chmod_n

            if (stat.mode & 0o777) ^ chmod != 0:
                try:
                    self.chmod(chmod)
                except Exception as err:
                    errors.append('%s has invalid mode %04o, cant set to %04o: %s' % (stat.mode, chmod, err))

        if errors:
            raise Exception('\n'.join(errors))

        return self
# }}}


# Utils (indent, logging) {{{
@contextlib.contextmanager
def indent(ctx, increase=1, quiet_prefix=None):
    try:
        ctx['indent'] += increase
        if quiet_prefix:
            ctx['quiet_prefix'].append(quiet_prefix)
        yield ctx['indent']
    finally:
        ctx['indent'] -= increase
        if quiet_prefix:
            ctx['quiet_prefix'].pop()
        if ctx['indent'] < 0:
            warn(ctx, 'Indent < 0!')
            ctx['indent'] = 0


def log(ctx, message, *args, **kwargs):
    severity = kwargs.get('severity', 'info')
    color = kwargs.get('color', 'white')
    stream = kwargs.get('stream', sys.stdout)
    full = kwargs.get('full', not ctx['args'].batch)
    always = kwargs.get('always', False)

    if not always and ctx['args'].quiet and severity in ('info', 'debug'):
        return

    colors = {
        'reset': '\033[0m',
        'black': '\033[30m',
        'red': '\033[31m',
        'green': '\033[32m',
        'yellow': '\033[33m',
        'blue': '\033[34m',
        'magenta': '\033[35m',
        'cyan': '\033[36m',
        'white': '\033[37m',
    }

    for name, seq in colors.items():
        if not seq.endswith(';1m'):
            colors['bold' + name] = seq[:-1] + ';1m'

    assert color in colors, 'No such color'

    try:
        message = message % args
    except:
        message = str(message)

    if severity == 'warning':
        severity = 'WARN'
    elif severity == 'error':
        severity = 'ERRR'
    elif severity == 'debug':
        severity = 'DEBG'

    if ctx['indent'] > 0 and (not ctx['args'].quiet or always):
        message = (ctx['indent'] * '  ') + message

    if ctx['args'].quiet and not always:
        for prefix in reversed(ctx['quiet_prefix']):
            message = '%s%s' % (prefix, message)

    if full:
        msg = '[%s]  ' % (time.strftime('%Y-%m-%d %H:%M:%S'), )
    else:
        msg = ''

    msg = '%s[%s]  %s' % (msg, severity.upper(), message)
    if stream.isatty():
        msg = colors[color] + msg + colors['reset']
    stream.write(msg + '\n')
    stream.flush()


def info(ctx, message, *args, **kwargs):
    onlyfile = kwargs.pop('onlyfile', False)
    if not onlyfile:
        log(ctx, message, *args, color='white', severity='info', stream=sys.stdout)
    if ctx['log']:
        log(ctx, message, *args, color='white', severity='info', stream=ctx['logfp'], full=True, always=True)


def debug(ctx, message, *args, **kwargs):
    onlyfile = kwargs.pop('onlyfile', False)
    if ctx['args'].debug and not onlyfile:
        log(ctx, message, *args, color='boldblack', severity='debug', stream=sys.stdout)
    if ctx['log']:
        log(ctx, message, *args, color='white', severity='debug', stream=ctx['logfp'], full=True, always=True)


def warn(ctx, message, *args, **kwargs):
    onlyfile = kwargs.pop('onlyfile', False)
    if not onlyfile:
        log(ctx, message, *args, color='boldyellow', severity='warning', stream=sys.stdout)
    if ctx['log']:
        log(ctx, message, *args, color='white', severity='warning', stream=ctx['logfp'], full=True, always=True)


def error(ctx, message, *args, **kwargs):
    onlyfile = kwargs.pop('onlyfile', False)
    if not onlyfile:
        log(ctx, message, *args, color='boldred', severity='error', stream=sys.stdout)
    if ctx['log']:
        log(ctx, message, *args, color='white', severity='error', stream=ctx['logfp'], full=True, always=True)


@contextlib.contextmanager
def syscall_timeout(seconds):
    def handler(signum, frame):
        pass

    orig_handler = signal.signal(signal.SIGALRM, handler)

    try:
        signal.alarm(seconds)
        yield
    finally:
        signal.alarm(0)
        signal.signal(signal.SIGALRM, orig_handler)
# Utils (indent, logging) }}}


def runproc(ctx, cmd, timeout, nice=0, get_out=False, env=None):  # {{{
    info(ctx, 'Calling %s...', subproc.list2cmdline(cmd))

    if nice == 0:
        preexec_fn = lambda: None
    else:
        preexec_fn = lambda: os.nice(nice)

    with indent(ctx, quiet_prefix='%s: ' % (subproc.list2cmdline(cmd), )):
        try:
            proc = subproc.Popen(cmd, stdout=subproc.PIPE, stderr=subproc.PIPE, preexec_fn=preexec_fn, env=env)
        except Exception as ex:
            warn(ctx, 'failed with: %s', str(ex))
            if not get_out:
                return 1
            return 1, '', ''

        deadline = time.time() + timeout

        ob, eb = [], []

        for fd in (proc.stdout.fileno(), proc.stderr.fileno()):
            fcntl.fcntl(fd, fcntl.F_SETFL, fcntl.fcntl(fd, fcntl.F_GETFL) | os.O_NONBLOCK)

        if get_out:
            out = {'stderr': [], 'stdout': []}

        while proc.poll() is None:
            of, ef = proc.stdout.fileno(), proc.stderr.fileno()
            rlist, wlist, xlist = select.select([of, ef], [], [], 1)

            for fd in rlist:
                buff = ob if of == fd else eb
                while 1:
                    try:
                        data = os.read(fd, 8192)
                        buff.append(data)
                        if get_out and data:
                            if fd == of:
                                out['stdout'].append(data)
                            else:
                                out['stderr'].append(data)
                    except OSError as ex:
                        if ex.errno == errno.EAGAIN:
                            break
                        raise

                    if '\n' in data or not data:
                        raw = ''.join(buff)
                        if not raw:
                            break

                        lines = []
                        pidx = 0
                        while 1:
                            idx = raw.find('\n', pidx)
                            if idx == -1:
                                lines.append(raw[pidx:])
                                break
                            else:
                                idx += 1
                                lines.append(raw[pidx:idx])
                                pidx = idx

                        for line in lines:
                            if not line.endswith('\n') and data:
                                buff.append(data)
                            else:
                                if not data:
                                    # If this is final lines from stream, dont cut \r, just print as is
                                    line = line.rstrip('\r')

                                line = line.rsplit('\r', 1)[-1].rstrip('\n')

                                if fd == of:
                                    info(ctx, 'stdout: %s', line)
                                else:
                                    warn(ctx, 'stderr: %s', line)

                        buff[:] = []

            if time.time() >= deadline:
                warn(ctx, 'timedout, killing...')
                proc.kill()
                deadline = time.time() + 10

        proc.wait()

        if proc.returncode != 0:
            warn(ctx, 'failed with exitcode %d', proc.returncode)
        else:
            info(ctx, 'finished with exitcode 0')

        if not get_out:
            return proc.returncode

        return proc.returncode, ''.join(out['stdout']), ''.join(out['stderr'])
# }}}


def _dump_version_meta(ctx, meta, deps_meta, prefix):  # {{{
    info(ctx, prefix)
    with indent(ctx, quiet_prefix=prefix):
        keys = ['Version:', 'Deps Version:', 'Installation Date:', 'Installed by:']
        maxlen = max(len(x) for x in keys)
        info(
            ctx,
            '{0: <{1}} {2} [{3}@{4}]'.format(
                keys[0], maxlen, *(meta[k] for k in ['version', 'url', 'revision']))
        )
        if deps_meta is not None:
            info(ctx, '{0: <{1}} {2} [{3}@{4}]'.format(
                keys[1], maxlen,
                deps_meta.get('Version', 'unknown'),
                deps_meta.get('SVN', {}).get('URL', 'unknown'),
                deps_meta.get('SVN', {}).get('Revision', 'unknown')
            ))
        else:
            info(ctx, '{0: <{1}} info not found'.format(keys[1], maxlen))

        info(ctx, '{0: <{1}} {2}'.format(keys[2], maxlen, meta['date']))
        info(ctx, '{0: <{1}} {2}'.format(keys[3], maxlen, meta['installed_by']))
# }}}


def detect_hostname():  # {{{
    hostname = socket.gethostname()

    if sys.platform == 'cygwin' and '.' not in hostname:
        hostname = socket.getfqdn(hostname).split()[0]

    ips = set()

    not_found_errs = (
        getattr(socket, 'EAI_NONAME', None),
        getattr(socket, 'EAI_NODATA', None),
    )

    try:
        addrinfo = socket.getaddrinfo(hostname, 0, 0, socket.SOCK_STREAM, 0)
    except socket.gaierror as ex:
        if ex.errno in not_found_errs:
            # Unable to resolve ourselves
            return hostname
        raise
    else:
        for ipinfo in addrinfo:
            ips.add(ipinfo[4][0])

        fqdns = set()

        for ip in ips:
            try:
                fqdn = socket.gethostbyaddr(ip)[0]
                if sys.platform == 'cygwin':
                    # on cygwin gethostbyaddr can return all DNS prefixes as an address, so we cut it here
                    fqdn = fqdn.split()[0]
                fqdns.add(fqdn)
            except socket.gaierror as ex:
                if ex.errno in not_found_errs:
                    continue
                raise

        fqdns = list(fqdns)

        if hostname in fqdns:
            # Found hostname in fqdns
            return hostname
        elif len(fqdns) == 1 and fqdns[0].startswith(hostname):
            # Got only 1 fqdn
            return fqdns[0]
        else:
            # Got many fqdns, dont know how to choose one
            return hostname
# }}}


def _get_skycore_services(cfg):
    skycore_services = set()
    for service_name, service_info in cfg.items():
        if service_info['config']:
            # dirty hack for copier migration
            if service_name == 'skybone':
                service_name = 'copier'
            skycore_services.add(service_name)

    return skycore_services


def get_base_config(ctx):  # {{{
    ctx['hostname'] = os.getenv('SKYINSTALL_HOSTNAME')
    if not ctx['hostname']:
        ctx['hostname'] = detect_hostname()
    info(ctx, 'Hostname: %s', ctx['hostname'], onlyfile=ctx['args'].batch)

    uri = 'http://api.genisys.yandex-team.ru/v2/hosts/%s/skynet?fmt=json' % (ctx['hostname'], )

    def _print_cfg(c, prefix=None):
        if prefix:
            debug(ctx, prefix)
        with indent(ctx):
            for key, value in c.items():
                if isinstance(value, dict):
                    debug(ctx, '%s:' % (key, ))
                    _print_cfg(value)
                elif isinstance(value, list):
                    debug(ctx, '%s:' % (key, ))
                    with indent(ctx):
                        for item in value:
                            debug(ctx, '- %s' % (item, ))
                else:
                    if not isinstance(value, basestring):
                        value = repr(value)
                    debug(ctx, '%s: %s' % (key, value))

    info(ctx, 'Grabbing config from genisys')
    with indent(ctx):
        debug(ctx, 'requesting %s' % (uri, ))

        headers = {'User-Agent': 'SkyInstall/1.0'}
        req = requests.get(uri, headers=headers)
        debug(ctx, 'done, status code %r, %d bytes', req.status_code, len(req.text) if req.text else -1)

        assert req.status_code == 200, 'Genisys returned bad status code (%r)' % (req.status_code, )

        cfg = simplejson.loads(req.text)
        base_cfg = cfg['subsections']['base']
        _print_cfg(base_cfg, 'dumped base config:')

        assert 'config' in base_cfg, 'no "config" in cfg'

        base_cfg = base_cfg['config']

        assert 'v1' in base_cfg, 'no "v1" in cfg'

        base_cfg = base_cfg['v1']

        skycore_cfg = (
            cfg['subsections']['skycore']['subsections']['namespaces']['subsections']['skynet']['subsections']
        )

        debug(ctx, 'Skycored services:')
        with indent(ctx):
            for key, value in skycore_cfg.items():
                debug(ctx, '{}: {}'.format(key, 'Enabled' if value.get('config', None) else 'Disabled'))

        skycore_services = _get_skycore_services(skycore_cfg)

        ctx['cfg'] = {}

        if not ctx['args'].prefix:
            ctx['cfg']['prefix'] = base_cfg['prefix']
        else:
            ctx['cfg']['prefix'] = ctx['args'].prefix
# }}}


def check_system(ctx):  # STEP {{{
    global FAIL_REASON

    # Perform all system checks before install
    info(ctx, 'Checking system')

    with indent(ctx):
        host, arch = os.uname()[0], os.uname()[4]
        host = host.lower()
        if host == 'freebsd':
            host = '%s%s' % (host, os.uname()[2][0])

        if arch in ('i686', 'i586', 'i486', 'i386'):
            arch = 'x86'
        elif arch in ('amd64', 'x86_64'):
            arch = 'amd64'
        else:
            arch = arch

        ctx['platform'] = '%s_%s' % (host, arch)

        if ctx['args'].arch and ctx['args'].arch != ctx['platform']:
            warn(ctx, 'we detected "%s" platform, but forced to "%s" by args', ctx['platform'], ctx['args'].arch)
            ctx['platform'] = ctx['args'].arch

        debug(ctx, 'Checking niceness')
        with indent(ctx, quiet_prefix='check niceness: '):
            current_niceness = os.nice(0)
            debug(ctx, 'current: %3d', current_niceness)

            fix_niceness = 0 - current_niceness
            debug(ctx, 'fix    : %3d', fix_niceness)

            if fix_niceness != 0:
                r, w = os.pipe()
                pid = os.fork()
                try:
                    if pid == 0:
                        try:
                            fixed_niceness = os.nice(fix_niceness)
                            if fixed_niceness != 0:
                                raise Exception('didnt set')
                            else:
                                os.write(w, '1')
                        except BaseException as ex:
                            os.write(w, '0')
                            os.write(w, str(ex)[:2048])
                        finally:
                            os.close(w)
                            os._exit(0)

                    # We wait for pid for simplicity here. Since we are writing to pipes
                    # max 2049 bytes, this should not deadlock.
                    pid, exitStatus = os.waitpid(pid, 0)

                    if exitStatus != 0:
                        error(ctx, 'Niceness checker process died with exit status: %s', exitStatus)
                        FAIL_REASON = 'Nice check fail'
                        raise SystemExit(1)

                    result = os.read(r, 1)
                    if result == '0':
                        reason = os.read(r, 2048)
                        error(ctx, 'Unable to fix niceness to zero: %s', reason)
                        FAIL_REASON = 'Nice fix fail'
                        raise SystemExit(1)
                    else:
                        debug(ctx, 'We can fix niceness before running skynet')

                except Exception as ex:
                    error(ctx, 'We unable to set niceness back to 0: %s', str(ex))
                    FAIL_REASON = 'Nice set fail'
                    raise SystemExit(1)

                finally:
                    os.close(r)
                    try:
                        os.close(w)
                    except:
                        pass

            ctx['nice'] = fix_niceness
# }}}


def check_paths(ctx):  # STEP {{{
    info(ctx, 'Checking paths')

    with indent(ctx, quiet_prefix='check paths: '):
        # Scheme looks like this (if dest is "/"):
        # prefix          /Berkanavt
        # runtime            /supervisor
        # services                /services
        # managed                 /managed
        # var                     /var
        # varskycore              /skycore
        # base                    /base
        # active                      /active -> {ACTIVE}
        # real                        /{active}
        # mainlink        /skynet -> /Berkanavt/supervisor/base/active
        # skyctl              /startup/skyctl
        # sky                 /tools/sky
        # cygps               /tools/cygps [CYGWIN ONLY]
        # global_bin      /usr/local/bin
        # skylink             /sky -> {SKYLINK}
        # cygpslink           /cygps -> {CYGPSLINK} [CYGWIN ONLY]

        paths = ctx['paths'] = {}

        if ctx['args'].dest:
            paths['prefix'] = ctx['args'].dest
        else:
            paths['prefix'] = Path(ctx['cfg']['prefix'])

        if ctx['args'].local:
            paths.update({
                'mainlink': paths['prefix'].join('skynet'),
                'global_bin': None,
                'global_etc': None,
            })
        else:
            paths.update({
                'mainlink': Path('/').join('skynet'),
                'global_bin': Path('/').join('usr', 'local', 'bin'),
                'global_etc': Path('/').join('etc'),
            })

        paths['runtime'] = paths['prefix'].join('supervisor')

        # Under prefix
        paths.update({
            'base':     paths['runtime'].join('base'),      # noqa
            'services': paths['runtime'].join('services'),  # noqa
            'managed':  paths['runtime'].join('managed'),   # noqa
            'var':      paths['runtime'].join('var'),       # noqa
            'etc':      paths['runtime'].join('etc'),       # noqa
        })

        # Misc
        paths.update({
            'active':     paths['base'].join('active'),     # noqa
            'skyctl':     paths['mainlink'].join('startup', 'skyctl'), # noqa
            'varskycore': paths['runtime'].join('skycore'), # noqa
            'sky':        paths['mainlink'].join('tools', 'sky'), # noqa
        })

        if paths['global_bin']:
            paths.update({
                'skylink': paths['global_bin'].join('sky')
            })
            paths['skyctllink'] = paths['global_bin'].join('skyctl')
        else:
            paths['skylink'] = None
            paths['skyctllink'] = None

        if sys.platform == 'cygwin':
            paths.update({
                'cygps': paths['mainlink'].join('tools', 'cygps'),
            })
            if paths['global_bin']:
                paths.update({
                    'cygpslink': paths['global_bin'].join('cygps')
                })
            else:
                paths['cygpslink'] = None

        if (
            not ctx['args'].local and paths['global_etc'] and paths['global_bin']
        ):
            if ctx['platform'].startswith('linux'):
                paths['initd'] = paths['global_etc'].join('init.d')
                paths['upstart'] = paths['global_etc'].join('init')
                paths['systemd'] = paths['global_etc'].join('systemd', 'system')
                paths['initd_skycore_script_src'] = paths['active'].join('startup', 'linux_skycored.sh')
                paths['upstart_skycore_job_src'] = paths['active'].join('startup', 'linux_upstart_skycored')
                paths['systemd_skycore_job_src'] = paths['active'].join('startup', 'linux_systemd_skycored')
                paths['crond'] = paths['global_etc'].join('cron.d')
                paths['crond_skynetd'] = paths['crond'].join('skynetd')
                paths['crond_skynetd_src'] = paths['active'].join('startup', 'linux_cron')
                paths['bash_completiond'] = paths['global_etc'].join('bash_completion.d')
                paths['sky_bash_completion'] = paths['bash_completiond'].join('sky')
                paths['sky_bash_completion_src'] = paths['active'].join('tools', 'bash_completion', 'sky')
                paths['skyctl_bash_completion'] = paths['bash_completiond'].join('skyctl')
                paths['skyctl_bash_completion_src'] = paths['active'].join('tools', 'bash_completion', 'skyctl')
            elif ctx['platform'].startswith('freebsd'):
                paths['initd'] = Path('/').join('usr', 'local', 'etc', 'rc.d')
                paths['crontab'] = paths['global_etc'].join('crontab')
            elif ctx['platform'].startswith('darwin'):
                paths['crontab'] = Path('/').join('usr', 'lib', 'cron', 'tabs', 'root')
                paths['launchd_skycored'] = paths['active'].join('startup', 'skycored.plist')
                paths['launchd_skycored_autostart'] = paths['active'].join('startup', 'skycored-autostart.plist')
                paths['launchd'] = Path('/').join('Library', 'LaunchDaemons')

        if 'initd' in paths:
            paths['initd_skycore_script'] = paths['initd'].join('skycored')
        if 'upstart' in paths:
            paths['upstart_skycore_job'] = paths['upstart'].join('skycored.conf')
        if 'systemd' in paths:
            paths['systemd_skycore_job'] = paths['systemd'].join('skycored.service')

        # To remove
        if 'initd' in paths:
            paths['deprecated_initd_script'] = paths['initd'].join('skynetd')
        paths['deprecated_uppy'] = paths['mainlink'].join('startup', 'up.py')

        links = ctx['links'] = {
            'mainlink': (
                paths['mainlink'].dirpath().bestrelpath(paths['active'])
                if ctx['args'].local else
                paths['active'].strpath
            ),

            # This is just to allow link in paths check, link itself will be determined later
            'active': None,
        }

        if paths['skylink']:
            links['skylink'] = paths['sky']
        if paths['skyctllink']:
            links['skyctllink'] = paths['skyctl']

        if sys.platform == 'cygwin':
            if paths['cygpslink']:
                links['cygpslink'] = paths['cygps']

        link_allowed = ('prefix', 'runtime', 'skylink', 'skyctllink')
        ignore = (  # ignore auto-checking
            'skyctl', 'sky', 'cygps',
            'crontab',
            'crond_skynetd', 'crond_skynetd_src',
            'bash_completiond',
            'sky_bash_completion','sky_bash_completion_src',
            'skyctl_bash_completion','skyctl_bash_completion_src',
            'deprecated_initd_script', 'deprecated_uppy',
            'initd_skycore_script', 'initd_skycore_script_src',
            'upstart_skycore_job_src', 'upstart_skycore_job',
            'systemd_skycore_job_src', 'systemd_skycore_job',
            'launchd', 'launchd_skycored', 'launchd_skycored_autostart',
        )
        required = ('prefix', 'base', 'varskycore')  # required for unpacking
        if not ctx['args'].local:
            required += ('global_bin', 'global_etc')

        if sys.platform == 'cygwin':
            files = ('skyctllink', 'skylink', 'cygpslink', )
        else:
            files = ('skyctllink', 'skylink', )

        # Make a deep check of all required paths.
        # Main idea is simple: if we will want to change something (e.g. create dir or change symlink)
        # we need what parent path is changeable by us. If symlink/dir already exists and is OK -- dont
        # scream.
        checked = set()
        for name, path in sorted(paths.items(), key=lambda s: (s[1].strpath if s[1] else s[1])):
            if path is None:
                continue

            if name in ignore:
                debug(ctx, 'Path (%-14s) "%s" ignored', name, path)
                continue

            debug(ctx, 'Path (%-14s) "%s": checking parts', name, path)

            for part in path.parts()[1:]:
                islink = False
                islink_allowed = False
                isfile = False

                if part == path:
                    islink = name in links
                    islink_allowed = name in link_allowed
                    isfile = name in files
                else:
                    # If we already checked this and this is not target path, skip.
                    if part in checked:
                        continue

                parent = part.dirpath()

                if parent.check(exists=1) and parent.check(dir=1):
                    if not parent.check(listable=1):
                        error(ctx, 'We are not able to list contents of %s directory', parent)
                        raise SystemExit(1)

                    if not parent.check(changeable=1):
                        if islink:
                            if not part.check(link=1) or part.readlink() != links[name]:
                                error(
                                    ctx, 'We are not able to modify "%s" symlink in "%s": is not changeable by us',
                                    part.basename, parent
                                )
                                with indent(ctx, quiet_prefix='symlink "%s": ' % (part, )):
                                    if part.check(link=1):
                                        error(ctx, 'current: %s', part.readlink(), onlyfile=ctx['args'].batch)
                                    else:
                                        error(ctx, 'current is not exists at all')
                                    if links[name] is not None:
                                        error(ctx, 'we want: %s', links[name])
                                raise SystemExit(1)
                        else:
                            if not part.check(exists=1, dir=1):
                                error(
                                    ctx, 'We are not able to create "%s" dir in "%s": is not changeable by us',
                                    part.basename, parent
                                )
                                raise SystemExit(1)
                    else:
                        if not islink:
                            if not part.check(exists=1, dir=1):
                                debug(ctx, 'Path (%-14s) "%s" is not exists: will create dir later', name, part)

                    # If not states specially, links are not allowed. We try to drop them, asuming nothing
                    # sensitive will be removed.
                    if not islink:
                        if not islink_allowed:
                            if part.check(link=1):
                                # Ok, we found a symlink there dir was expected.
                                # Allow to drop symlink and recreate dir only if that path is under
                                # /Berkanavt.

                                msg = 'Path (%-14s) "%s" is a symlink to "%s", but we expect dir and ' \
                                      'link is not allowed there' % (name, part, part.readlink())

                                if part.strpath.startswith(paths['runtime'].strpath):
                                    warn(ctx, '%s, removing...', msg)
                                    part.remove()
                                else:
                                    if part.check(dir=1):
                                        debug(
                                            ctx, 'Path (%-14s) "%s" is a symlink to "%s", which is directory and ok',
                                            name, part, part.readlink()
                                        )
                                    else:
                                        error(ctx, msg)
                                        raise SystemExit(1)
                    else:
                        if part.check(exists=1, link=0):
                            error(ctx, 'Path (%-14s) "%s" is a not a symlink, we want symlink', name, part)
                            raise SystemExit(1)
                        else:
                            debug(ctx, 'Path (%-14s) "%s" is a symlink and we expect it', name, part)

                    # Files are not allowed anywhere, except specially stated. But we dont drop them.
                    if not isfile:
                        if part.check(file=1):
                            warn(ctx, 'Path (%-14s) "%s" is a file, but we expect dir', name, part)
                            raise SystemExit(1)
                        else:
                            debug(ctx, 'Path (%-14s) "%s" is not a file and we expect this', name, part)

                checked.add(part)

            if name in required and path.check(exists=0):
                info(ctx, 'Creating missing "%s"...', path)

                with indent(ctx):
                    while path.check(link=1):
                        opath = path
                        path = py.path.local(os.path.realpath(path.strpath))
                        info(ctx, 'which actually pointing to "%s"', path)
                        if path == opath:
                            error(ctx, 'oops! recursive symlink: "%s"', path)

                path.ensure(dir=1)

        # Prepare name for this skynet destination install path
        version, revision = ctx['args'].version, ctx['args'].revision
        if version and revision:
            real_name_base = '%s.r%s' % (version, revision)
        elif version:
            real_name_base = version
        elif revision:
            real_name_base = 'unknown.r%s' % (revision, )
        else:
            real_name_base = 'unknown'

        for i in range(-1, 100):
            if i != -1:
                real_name = '%s.%02d' % (real_name_base, i)
            else:
                real_name = real_name_base

            real_path = paths['base'].join(real_name)
            if real_path.check(exists=1) or real_path.check(link=1):
                continue

            paths['real'] = real_path
            break

        # Determine future path of active link
        links['active'] = paths['active'].dirpath().bestrelpath(paths['real'])

        # If active points to non existent path, we can have strange situation here if it already points to us.
        # In this case after extracting tarballs it will point to this skynet and we will fail to stop it.
        if paths['active'].check(link=1) and paths['active'].check(exists=0):
            info(ctx, 'Current active link points to non-existent path %s, removing', paths['active'].readlink())
            paths['active'].remove()

        info(ctx, 'Paths:', onlyfile=ctx['args'].batch)
        with indent(ctx):
            info(ctx, 'prefix: %s', paths['prefix'], onlyfile=ctx['args'].batch)
            info(ctx, 'runtime: %s', paths['runtime'], onlyfile=ctx['args'].batch)
            info(ctx, 'real: %s', paths['real'], onlyfile=ctx['args'].batch)

        if 'STOP_AFTER_CHECKING_PATHS' in os.environ:
            raise SystemExit(0)
# }}}


def move_prefix(ctx, old_prefix, new_prefix):  # {{{
    oldvar = old_prefix.join('supervisor', 'var')
    newvar = new_prefix.join('supervisor', 'var')

    oldskycore = old_prefix.join('supervisor', 'skycore')
    newskycore = new_prefix.join('supervisor', 'skycore')

    oldlayout = old_prefix.join('supervisor', '.layout')
    newlayout = new_prefix.join('supervisor', '.layout')

    if (
        ctx['paths']['skyctl'].check(exists=1, file=1) and
        old_prefix.join('supervisor', 'services').check(exists=1, dir=1) and
        old_prefix.join('supervisor', 'managed').check(exists=1, dir=1) and
        ctx['prev']['skycore rev']
    ):
        stop_skycore(ctx, "unable to stop old skynet", 180, fail=False)

        if ctx['paths']['deprecated_uppy'].check(exists=1, file=1):
            if not manage_skynet(ctx, ['stop'], 180, 'deprecated_uppy'):
                error(ctx, 'unable to stop old skynet')
                raise Exception('unable to stop old skynet')
    else:
        info(ctx, 'skipping skynet stopping: %s not exists', ctx['paths']['skyctl'])

    install_log = oldvar.join('log', 'install.log')

    for old, new in (
        (oldvar, newvar),
        (oldskycore, newskycore),
        (oldlayout, newlayout),
    ):
        if old.check(dir=1, exists=1):
            for path in old.visit():
                if path == install_log:
                    debug(ctx, 'skip: %s', path)
                    continue

                newpath = new.join(path.strpath[len(old.strpath):])

                if path.check(link=1):
                    debug(ctx, 'link: %s', newpath)
                    newpath.dirpath().ensure(dir=1)
                    if newpath.check(exists=1):
                        newpath.remove()
                    newpath.mksymlinkto(path.readlink())
                elif path.check(dir=1):
                    debug(ctx, 'dir: %s', newpath)
                    newpath.ensure(dir=1)
                    newpath.chmod(path.stat().mode)
                    newpath.chown(path.stat().uid, path.stat().gid)
                elif path.check(file=1):
                    debug(ctx, 'file: %s', newpath)
                    newpath.dirpath().ensure(dir=1)
                    path.copy(newpath)
                    newpath.chmod(path.stat().mode)
                    newpath.chown(path.stat().uid, path.stat().gid)
                else:
                    debug(ctx, 'special (skipped): %s', newpath)

        elif old.check(file=1, exists=1, link=0):
            debug(ctx, 'file: %s', new)
            new.dirpath().ensure(dir=1)
            old.copy(new)
            new.chmod(old.stat().mode)
            new.chown(old.stat().uid, old.stat().gid)

    old_sup = old_prefix.join('supervisor')
    for i in range(10):
        try:
            if old_sup.check(dir=1, exists=1):
                old_sup.remove()
        except py.error.ENOTEMPTY:
            if i == 9:
                raise
# }}}


def check_old_skynet(ctx):  # {{{
    info(ctx, 'Check old skynet installation')
    with indent(ctx, quiet_prefix='Check old skynet: '):
        debug(ctx, 'Fetch previous installation metadata.')

        # Generic way to find .info file -- use active path
        info_path = ctx['paths']['active'].join('.info')

        # Another way, using symlink /skynet. Works during prefix migration
        info_path_2 = ctx['paths']['mainlink'].join('.info')

        prev = ctx['prev'] = dict.fromkeys(PRIMARY_VERSION_META)
        try:
            if info_path.check(exists=1, file=1):
                data = yaml.load(info_path.open('rb'))
            else:
                data = yaml.load(info_path_2.open('rb'))
            prev.update(dict(filter(lambda x: x[1] is not None, [(k, data.get(k, None)) for k in prev.iterkeys()])))
        except Exception as ex:
            warn(ctx, 'Unable to parse %r metadata file: %s', info_path.strpath, str(ex))

        if not isinstance(prev['skynet.bin gen'], int):
            prev['skynet.bin gen'] = 0

        deps_info_path = ctx['paths']['active'].join('python', '.info')
        deps_info_path_2 = ctx['paths']['mainlink'].join('python', '.info')

        try:
            if deps_info_path.check(exists=1, file=1):
                prev_deps = yaml.load(deps_info_path.open('rb'))
            else:
                prev_deps = yaml.load(deps_info_path_2.open('rb'))
        except Exception as ex:
            prev_deps = None
            warn(ctx, 'Unable to parse %r metadata file: %s', deps_info_path.strpath, str(ex))

        ctx['prev']['deps'] = prev_deps

        # Read mainlink

        mainlink = ctx['paths']['mainlink']
        if mainlink.check(link=1):
            mainlink_link = mainlink.dirpath().join(mainlink.readlink())

            old_prefix = None
            for part in mainlink_link.parts():
                if part.basename == 'supervisor':
                    old_prefix = part.dirpath()
                    break
            else:
                warn(ctx, 'Unable to find proper old skynet prefix')

            ctx['paths']['old_prefix'] = old_prefix
        else:
            ctx['paths']['old_prefix'] = None

        _dump_version_meta(ctx, prev, prev_deps, 'Current installation metadata: ')

    info(ctx, 'Migrating prefix')
    with indent(ctx):
        prefix = ctx['paths']['prefix']
        old_prefix = ctx['paths']['old_prefix']

        debug(ctx, 'prefix (old): %s', old_prefix)
        debug(ctx, 'prefix (new): %s', prefix)

        if old_prefix is not None and old_prefix != prefix:
            move_prefix(ctx, old_prefix, prefix)
        else:
            debug(ctx, 'migration not needed')
# }}}


def symlink_change_atomic(ctx, path, link):  # {{{
    info(ctx, 'Switch symlink at %s to %s', path, link)
    with indent(ctx, quiet_prefix='symlink switch: '):
        clink = None
        npath = path.dirpath().join(path.basename + '.new')

        if path.check(link=1):
            clink = path.readlink()
            info(ctx, 'Old: %s', clink)
        elif path.check(exists=1):
            info(ctx, 'Old: not a link, removing')
            path.remove()
        else:
            info(ctx, 'Old: not exists')

        info(ctx, 'New: %s', link)

        if clink:
            if link != clink:
                info(ctx, 'We need to change it')
                if npath.check(link=1) or npath.check(exists=1):
                    npath.remove()

                info(ctx, 'Tmp: %s -> %s', npath, link)

                npath.mksymlinkto(link)
                npath.move(path)

                info(ctx, 'Symlink atomically changed')
                return clink
            else:
                info(ctx, 'Symlink is okay')
                return clink
        else:
            info(ctx, 'We need to create it')
            path.mksymlinkto(link)

            info(ctx, 'Symlink created')
            return None
# }}}


def update_symlinks(ctx):  # STEP {{{
    info(ctx, 'Switch skynet to new one')
    with indent(ctx, quiet_prefix='switch skynet: '):
        active_path = ctx['paths']['active']
        active_link = ctx['links']['active']
        mainlink_path = ctx['paths']['mainlink']
        mainlink_link = ctx['links']['mainlink']

        old_active_link = symlink_change_atomic(ctx, active_path, active_link)

        try:
            symlink_change_atomic(ctx, mainlink_path, mainlink_link)
        except Exception as ex:
            ex_info = sys.exc_info()
            info(ctx, 'Failed to change symlink %s -> %s: %s', mainlink_path, mainlink_link, str(ex))

            # Attempt to revert everything back
            if isinstance(old_active_link, basestring):
                symlink_change_atomic(ctx, active_path, old_active_link)
            six.reraise(*ex_info)
# }}}


def _extract_simple(ctx, tar, tarinfo, name=None, subpath=None):  # {{{
    if name is None:
        name = tarinfo.name

    info(ctx, name)
    with indent(ctx, quiet_prefix='%s: ' % (name, )):
        subtar = tarfile.open(mode='r|gz', fileobj=tar.extractfile(tarinfo))

        version = None
        for subtarinfo in subtar:
            if '..' in subtarinfo.name or subtarinfo.name.startswith('/'):
                warn(ctx, 'ignored %s (not secured path)', subtarinfo.name)
                continue

            if name == FN_SKYNET and subtarinfo.name == '.version':
                fh = subtar.extractfile(subtarinfo)
                version = fh.read()
                info(ctx, 'Paths.Real Switch: ')
                with indent(ctx, quiet_prefix='Paths.Real Switch: '):
                    info(ctx, 'Old: %s', ctx['paths']['real'])
                    newpath = ctx['paths']['real'].dirpath().join(version)

                    for epoch in range(0x400):
                        if newpath.check(exists=0):
                            break
                        newpath = ctx['paths']['real'].new(basename='%s.%02d' % (version, epoch))
                    else:
                        error(ctx, 'Unable to determine suitable base placement in 1KiB retries.')
                        raise SystemExit(1)

                    info(ctx, 'New: %s', newpath)
                    ctx['paths']['real'] = newpath
                    ctx['links']['active'] = ctx['paths']['active'].dirpath().bestrelpath(newpath)
                continue

            # Force root perms on files (SKYDEV-1343)
            subtarinfo.uname = 'root'
            subtarinfo.gname = 'root'
            subtarinfo.uid = 0
            subtarinfo.gid = 0

            targetpath = ctx['paths']['real']
            if subpath:
                targetpath = targetpath.join(subpath)

            subtar.extractall(targetpath.strpath, [subtarinfo])

        if version:
            with ctx['paths']['real'].join('.version').open(mode='wb') as fp:
                fp.write(version)
                if hasattr(os, 'fsync'):
                    os.fsync(fp.fileno())
# }}}


def _eval(ctx, code, input=None, attempts=6, active=False, critical=True):  # {{{
    res = []
    failures = []
    real = ctx['paths']['real'] if not active else ctx['paths']['active']

    procs = []
    threads = []

    def _reader(proc):
        try:
            stdout, stderr = proc.communicate(input=input)
            if proc.returncode == 0:
                res[:] = [msgpack.loads(stdout)]
            else:
                failures.append(stderr)
        except:
            pass

    for i in range(attempts, 0, -1):
        executable = real.join('python', 'bin', 'python').strpath
        debug(ctx, 'try in subprocess %r...', executable)

        cproc = subproc.Popen(
            [executable, '-c', textwrap.dedent(code)],
            stdin=subproc.PIPE, stdout=subproc.PIPE, stderr=subproc.PIPE
        )
        procs.append(cproc)

        cthr = threading.Thread(target=_reader, args=(cproc, ))
        threads.append(cthr)

        cthr.start()

        wait_untill = time.time() + 20
        for idx, thr in enumerate(threads):
            tout = wait_untill - time.time()
            if tout > 0:
                thr.join(timeout=tout)

            if res:
                break

        if res:
            break

    # Wait until all procs will die for max 5 minutes.
    deadline = time.time() + 300

    while time.time() < deadline:
        goon = False
        for proc in procs:
            if proc.poll() is None:
                goon = True
                try:
                    proc.kill()
                except:
                    pass
        if not goon:
            break

        time.sleep(0.5)

    if not res:
        msg = 'Failed to evaluate Python code with %s Python distribution' % ('current' if active else 'nested')
        if not critical:
            raise Exception('\n'.join([msg] + failures))
        error(ctx, msg)
        if failures:
            for fail in failures:
                if fail:
                    error(ctx, fail)
                    break
        raise SystemExit(1)
    return res
# }}}


def update_config(ctx):
    info(ctx, 'Updating configuration registry...')
    with indent(ctx, quiet_prefix='update configuration registry: '):
        _eval(
            ctx,
            '''
                import sys
                import socket
                import logging
                import msgpack
                from library import config

                logging.basicConfig(level=logging.DEBUG, stream=sys.stderr)
                config.update(host='{0}')
            '''.format(ctx['hostname'])
        )


def prepare_tarballs(ctx, fp):  # STEP {{{
    tar = tarfile.open(mode='r|', fileobj=fp)
    depsfn = FN_DEPS % (ctx['platform'], )

    info(ctx, 'Extracting...')

    with indent(ctx):
        for tarinfo in tar:
            if tarinfo.name in (depsfn, FN_SKYNET):
                name = tarinfo.name
                is_deps = name == depsfn
                if is_deps and ctx['args'].batch:
                    name = 'skynet-deps*.tgz'

                _extract_simple(ctx, tar, tarinfo, name=name)
                if is_deps:
                    # Python extracted, its time to evaluate some code with it )
                    update_config(ctx)
                    ctx['deps-changed'] = True
                    depsinfofn = ctx['paths']['real'].join('python', '.info')
                    if depsinfofn.check(file=1, exists=1):
                        depsinfo = yaml.load(depsinfofn.open(mode='rb'))
                    else:
                        depsinfo = None

            elif tarinfo.name == FN_SKYCORE:
                _extract_simple(ctx, tar, tarinfo, subpath='skycore')

            elif tarinfo.name == FN_DOCS:
                _extract_simple(ctx, tar, tarinfo, subpath='share/docs')

            elif tarinfo.name in FN_IGNORE:
                continue

            else:
                debug(ctx, 'ignored: %s', tarinfo.name)

    tar.close()

    class _Info(list):
        pass

    install_info = _Info((
        ('version', ctx['args'].version),
        ('revision', ctx['args'].revision),
        ('state', 'not_completely_installed'),
        ('url', ctx['args'].url),
        ('platform', ctx['platform']),
        ('args', ' '.join(sys.argv)),
        ('timestamp', int(time.time())),
        ('date', time.ctime()),
        ('paths', _Info(sorted(
            [
                (name_, path)
                for name_, path in ctx['paths'].items()
                if name_ not in ctx['links']
            ], key=lambda x: x[1]
        ))),
        ('links', _Info(sorted(
            [
                (ctx['paths'][name_], target)
                for name_, target in ctx['links'].items()
            ]
        ))),
        ('installed_by', pwd.getpwuid(os.getuid()).pw_name),
        ('skynet.bin gen', SKYNET_BIN_GENERATION),
        ('skycore rev', ctx['args'].skycore_rev if ctx['paths']['real'].join('skycore').check(exists=1) else ''),
    ))
    upcoming = ctx['next'] = dict(kv for kv in install_info if kv[0] in PRIMARY_VERSION_META)
    ctx['next']['deps'] = depsinfo

    _dump_version_meta(ctx, upcoming, depsinfo, 'Upcoming installation metadata: ')

    yaml.add_representer(_Info, lambda dumper, data: dumper.represent_dict(data))
    yaml.add_representer(Path, lambda dumper, data: dumper.represent_str(data.strpath))

    with ctx['paths']['real'].join('.info').open(mode='wb') as fp:
        fp.write(
            '%YAML 1.2\n'
            '---\n\n' +
            yaml.dump(install_info, default_flow_style=False)
        )
        if hasattr(os, 'fsync'):
            os.fsync(fp.fileno())
# }}}


def lock_up_py_lock(ctx, lock_file):  # {{{
    global FAIL_REASON
    if lock_file.check(exists=1):
        if lock_file.check(file=0) or lock_file.check(link=1):
            lock_file.remove()
    if not lock_file.check(file=1):
        lock_fp = lock_file.open(mode='wb+')
    else:
        lock_fp = lock_file.open(mode='wb+')

    with syscall_timeout(600):
        try:
            fcntl.flock(lock_fp, fcntl.LOCK_EX)
        except IOError as ex:
            if ex.errno != errno.EINTR:
                raise
            error(ctx, 'Cant grab "%s" lock, timed out' % (lock_file, ))
            FAIL_REASON = 'Unable to grab %s lock' % (lock_file, )
            raise SystemExit(1)

    info(ctx, 'Locked "%s"', lock_file)
    return lock_fp
# }}}


def manage_skynet(ctx, args, timeout, binname='deprecated_uppy', nice=0, getresult=False, keep_autostart=None):  # STEP {{{
    # TODO: attempt to kill skynet processes if uppy fails if asked to.  # noqa

    ctl = ctx['paths'][binname]
    info(ctx, 'Manage skynet: %s %s' % (ctl, ' '.join(args), ))

    if ctl.check(exists=1, file=1):
        with indent(ctx):
            cmd = [ctl.strpath] + args

            env = os.environ.copy()
            env['STARTUP_PARALLEL'] = 'no'
            env['SKYCTL_USE_LOCK'] = '0'
            if keep_autostart is not None:
                env['SKYCTL_KEEP_AUTOSTART'] = str(int(keep_autostart))

            if not getresult:
                returncode = runproc(ctx, cmd, timeout=timeout, nice=nice, env=env)
                return returncode == 0
            else:
                returncode, stdout, stderr = runproc(ctx, cmd, timeout=timeout, nice=nice, get_out=True, env=env)
                return returncode, stdout, stderr
    else:
        warn(ctx, 'Skipping stopping of skynet: %s not exists', ctl)
        if not getresult:
            return None
        else:
            return None, None, None
# }}}


def _get_skycore_namespaces(ctx, timeout=None):
    ret, stdout, _ = manage_skynet(ctx,
                                   (['--no-lock'] if NO_LOCK_SKYCORE else []) + ['list-ns'],
                                   timeout or ctx['args'].stop_timeout,
                                   'skyctl',
                                   getresult=True)
    if ret != 0:
        return []
    stdout = stdout.strip()
    if not stdout:
        return []
    return stdout.split('\n')


def get_perms(ctx):  # STEP {{{
    # Grab target ids
    # perm:
    #   uid: skynet uid in root mode, current user uid otherwise
    #   gid: skynet gid in root mode, current user gid otherwise
    # perm2:
    #   uid: skynet uid in root mode, current user uid otherwise
    #   gid: skywheel gid or current user gid (latter only if skywheel group not exists and we are in root mode)

    global FAIL_REASON

    if not ctx['args'].local:
        with indent(ctx, quiet_prefix='checking user "%s":' % (ctx['args'].user, )):
            try:
                userpw = pwd.getpwnam(ctx['args'].user)
            except Exception as ex:
                error(ctx, 'user %r was not found: %s', ctx['args'].user, ex)
                FAIL_REASON = 'user %s was not found' % (ctx['args'].user, )
                raise SystemExit(1)

            uname, _, uuid, ugid, _, uhome, _ = userpw

            assert uname == ctx['args'].user
            if uhome != ctx['args'].user_home:
                warn(ctx, '$HOME for "%s" user is set to "%s", should be "%s"', uname, uhome, ctx['args'].user_home)
            if not Path(uhome).check(exists=1, dir=1):
                warn(ctx, '$HOME dir "%s" for "%s" user does not exists or not a directory', uhome, uname)

            # Let's try to become target user (we will need this for start/stop)
            try:
                info(ctx, 'check what we can suid to user...')
                # there is a problem with 'skynet' user in cygwin, so we just skip this check
                if sys.platform != 'cygwin':
                    proc = subproc.Popen(
                        ['id'],
                        stdout=subproc.PIPE, stderr=subproc.PIPE,
                        preexec_fn=lambda: os.setuid(uuid)
                    )
                    stdout, stderr = proc.communicate()
            except Exception as ex:
                errortext = str(ex)
                if isinstance(ex, OSError):
                    if ex.errno == errno.EPERM:
                        errortext = 'Permission denied'

                error(ctx, 'Cant become "%s" user: %s', uname, errortext)
                FAIL_REASON = 'Cant suid to user "%s"' % (uname, )
                raise SystemExit(1)

            ctx['perm'] = uuid, 0, 0o22

        with indent(ctx, quiet_prefix='checking group "%s":' % (ctx['args'].group, )):
            try:
                grouppw = grp.getgrnam(ctx['args'].group)
            except Exception as ex:
                warn(ctx, 'group "%s" was not found: %s', ctx['args'].group, ex)
                ctx['perm2'] = 0, 0, 0o22  # if no group exists, dont allow root mode
            else:
                gname, _, ggid, _ = grouppw
                ctx['perm2'] = 0, ggid, 0o2

    else:
        ctx['perm'] = os.getuid(), os.getgid(), 0o22
        ctx['perm2'] = ctx['perm'][:]
# }}}


def fix_all_perms(ctx, path=None, descend_base=True):  # STEP {{{
    global FAIL_REASON

    info(ctx, 'Set all files proper permissions and modes...')

    with indent(ctx, quiet_prefix='set all files proper permissions: '):
        counts = {
            'total': 0,
            'dont_descend': 0,
            'ignored': 0,
        }

        if path is None:
            tpath = ctx['paths']['runtime']
        else:
            tpath = path

        # /Berkanavt/supervisor/var is known to has bad perms sometimes
        # (e.g. on DISTBUILD), so fix it first or it will not allow to visit all
        # inner paths for us later.
        uid, gid, mask = ctx['perm']
        for path in (
            ctx['paths']['var'],
            ctx['paths']['var'].join('log')
        ):
            try:
                path.ensure_perms(uid, gid, (0o777, 0o666), mask=mask)
            except py.error.ENOENT:
                pass

        def _old_base_flt(p, m):
            _basepath = ctx['paths']['base'].strpath
            _realpath = ctx['paths']['real'].strpath
            _activepath = ctx['paths']['active'].strpath
            _varpath = ctx['paths']['var'].strpath

            if m == 0:
                if p.check(link=1):
                    return False

                # Do not descend into path if:
                # 1) it is under supervisor/base
                # 2) and it is not supervisor/base, but deeper
                # 3) and it is not our base or do not descend is asked
                nothandle = (
                    p.dirpath() == _basepath and
                    (
                        not descend_base or
                        p != _realpath
                    )
                )

                # Do not descrend into specific service's dirs
                nothandle = nothandle or p == ctx['paths']['var'].join('skycore')
                nothandle = nothandle or p == ctx['paths']['runtime'].join('skycore')
                nothandle = nothandle or p == ctx['paths']['var'].join('copier')
                nothandle = nothandle or p == ctx['paths']['var'].join('copier-mds')
                nothandle = nothandle or p == ctx['paths']['var'].join('procman')
            else:
                # Dont touch files if:
                # 1) it is under supervisor/base
                # 2) and it is not supervisor/base itself
                # 3) and it is not supervisor/base/{active,REAL}
                # 4) also dont touch socks in var if we are not in full restart mode
                # 5) dont touch some specific paths (e.g. var/procman)
                nothandle = (
                    (
                        p.strpath.startswith(_basepath) and
                        p.strpath != _basepath and
                        p.strpath != _activepath and
                        (
                            not p.strpath.startswith(_realpath + '/') and
                            p != _realpath
                        )
                    ) or (
                        p.strpath.startswith(_varpath) and
                        osstat.S_ISSOCK(p.stat().mode)
                    ) or (
                        p == ctx['paths']['var'].join('procman')  # dont change perms on var/procman folder
                    )
                )

            if nothandle:
                if m == 0:
                    counts['dont_descend'] += 1
                elif m == 1:
                    counts['ignored'] += 1

                return False

            return True

        def _safe_old_base_flt(p, m):
            try:
                return _old_base_flt(p, m)
            except py.error.ENOENT:
                return False

        for path in itertools.chain(
            [tpath],
            tpath.visit(
                rec=lambda p: _safe_old_base_flt(p, 0),
                fil=lambda p: _safe_old_base_flt(p, 1)
            )
        ):
            if path.check(link=1):
                if not path.check(exists=1):
                    # TODO in batch mode replace path with fake one here  # noqa
                    debug(ctx, 'invalid symlink: %s, removing...', path)
                    path.remove()
                    continue

            try:
                uid, gid, mask = ctx['perm2']
                for nonrootpath in (
                    ctx['paths']['var'],
                ):
                    if path.strpath.startswith(nonrootpath.strpath):
                        uid, gid, mask = ctx['perm']
                        break

                if path.strpath.startswith(ctx['paths']['etc'].join('auth').strpath):
                    # We are supervisor/etc/auth**
                    if path != ctx['paths']['etc'].join('auth'):
                        # We are supervisor/etc/auth/**
                        relpath = path.relto(ctx['paths']['etc'].join('auth'))
                        userdir = ctx['paths']['etc'].join('auth').join(relpath.split('/', 1)[0])
                        if userdir.check(dir=1):
                            mask = 0o022

                if path == ctx['paths']['var'].join('procman'):
                    # We are supervisor/var/procman
                    path.ensure_perms(uid, gid, (0o1777, 0o1666), mask=0)
                else:
                    path.ensure_perms(uid, gid, (0o777, 0o666), mask=mask)

                counts['total'] += 1
            except Exception as ex:
                error(ctx, 'Failed to fix permissions for %s: %s', path, ex)
                FAIL_REASON = 'Fail to fix perms'
                raise SystemExit(1)

        info(
            ctx, 'done for %d items, ignored %d items, didnt descended to %d dirs',
            counts['total'], counts['ignored'], counts['dont_descend']
        )
# }}}


def fix_shebang(ctx, path=None, descend_base=True):  # STEP {{{
    if not ctx['args'].local:
        info(ctx, 'Skipped shebang fixing')
        return

    info(ctx, 'Fix shebangs...')

    with indent(ctx, quiet_prefix='fix shebangs: '):
        counts = {
            'changed': 0
        }

        if path is None:
            tpath = ctx['paths']['runtime']
        else:
            tpath = path

        def _old_base_flt(p, m):
            _basepath = ctx['paths']['base'].strpath
            _realpath = ctx['paths']['real'].strpath

            if m == 0:
                if p.check(link=1):
                    return False

                # Do not descend into path if:
                # 1) it is under supervisor/base
                # 2) and it is not supervisor/base, but deeper
                # 3) and it is not our base or do not descend is asked
                nothandle = (
                    p.dirpath() == _basepath and
                    (
                        not descend_base or
                        p != _realpath
                    )
                )
            else:
                # Dont touch files if:
                # 1) it is under supervisor/base
                # 2) and it is not supervisor/base itself
                # 3) and it is not supervisor/base/{active,REAL}
                nothandle = (
                    p.strpath.startswith(_basepath) and
                    p.strpath != _basepath and
                    (
                        not p.strpath.startswith(_realpath + '/') and
                        p != _realpath
                    )
                )

            if nothandle:
                return False

            return True

        shebang_template_length = len(SHEBANG_TPL)
        shebang = '#!%s/' % (ctx['paths']['prefix'], )
        shebang = '%s%s' % (shebang, 'skynet/python/bin/python')

        for path in tpath.visit(
            rec=lambda p: _old_base_flt(p, 0),
            fil=lambda p: _old_base_flt(p, 1)
        ):
            if not path.check(file=1) or path.check(link=1):
                continue

            try:
                if path.open(mode='rb').read(shebang_template_length) == SHEBANG_TPL:
                    new_shebang = '%s%s' % (shebang, path.read(mode='rb')[shebang_template_length:])
                    with path.open(mode='wb') as fp:
                        fp.write(new_shebang)
                        fsync_queue.put(fp.name)

                    counts['changed'] += 1
            except Exception as ex:
                error(ctx, 'Failed to change shebang for %s: %s', path, ex)

        info(ctx, 'done for %s items', counts['changed'])
# }}}


def cleanup_prefix(ctx):  # STEP {{{
    cleaned = 0
    for path in ctx['paths']['runtime'].listdir():
        if path.basename not in (
            'base',
            'managed',
            'services',
            'skycore',
            'up.py.lock',
            'var',
            'etc',
            '.layout',
        ):
            path.remove()
            cleaned += 1

    if cleaned > 0:
        info(ctx, 'Cleaned %d items in runtime (supervisor)', cleaned)

    for required in (
        'base',
        'managed',
        'services',
        'skycore',
        'var',
        'etc',
    ):
        ctx['paths']['runtime'].join(required).ensure(dir=1)
# }}}


def cleanup_bases(ctx):  # STEP {{{
    info(ctx, 'Cleaning up "%s"...', ctx['paths']['base'])

    with indent(ctx, quiet_prefix='cleaning "%s": ' % (ctx['paths']['base'], )):
        for path in ctx['paths']['base'].listdir():
            if path == ctx['paths']['real']:
                continue
            if path == ctx['paths']['active']:
                continue
            info(ctx, 'removing "%s"...', path)
            try:
                path.remove(ignore_errors=True)
            except Exception as ex:
                warn(ctx, 'failed to remove "%s": %s', path, ex)
# }}}


def cleanup_skycore(ctx):
    info(ctx, 'Cleaning up "%s"', ctx['paths']['varskycore'])
    oldstyle_log_re = re.compile(r'^.*\.log\.\d+$')
    old_log_re = re.compile(r'^.*\.log\.(\d{4})-(\d{2})-(\d{2})$')
    today = datetime.date.today()

    with indent(ctx, quiet_prefix='cleaning skycore: '):
        dirs = [ctx['paths']['varskycore'].join('log')]
        while dirs:
            directory = dirs.pop(0)
            if not directory.check(dir=1):
                continue

            try:
                paths = directory.listdir()
            except Exception as e:
                warn(ctx, "listdir %r failed: %s", directory.strpath, e)
                continue

            for path in paths:
                if path.check(dir=1, link=0):
                    dirs.append(path)
                    continue

                p = path.basename
                if oldstyle_log_re.match(p):
                    debug(ctx, 'Removing old-style logfile %s', path.strpath)
                    try:
                        path.remove()
                    except Exception as e:
                        warn(ctx, "remove %r failed: %s", path.strpath, e)
                    continue

                match = old_log_re.match(p)
                if match and (
                    today - datetime.date(int(match.group(1)), int(match.group(2)), int(match.group(3)))
                ).days > 15:
                    debug(ctx, 'Removing old logfile %s', path.strpath)
                    try:
                        path.remove()
                    except Exception as e:
                        warn(ctx, "remove %r failed: %s", path.strpath, e)


def cleanup_deprecated(ctx):
    info(ctx, "Removing deprecated paths")
    with indent(ctx, quiet_prefix='remove deprecated: '):
        for name in (
            'deprecated_uppy',
            'deprecated_initd_script',
        ):
            if name not in ctx['paths']:
                continue
            path = ctx['paths'][name]
            if path.check(exists=1):
                debug(ctx, 'remove: %r', path.strpath)
                try:
                    path.remove()
                except Exception as e:
                    warn(ctx, "remove %r failed: %s", path.strpath, e)

        if 'deprecated_initd_script' in ctx['paths']:
            initd_script = ctx['paths']['deprecated_initd_script']
            with indent(ctx, quiet_prefix='runlevels for %r' % (initd_script,)):
                updatercd = py.path.local().sysfind('update-rc.d')
                if not updatercd:
                    debug(ctx, 'update-rc.d was not found')
                else:
                    runproc(ctx, [updatercd.strpath, '-f', initd_script.basename, 'remove'], timeout=60)


def skycore_changed(ctx):
    oldrev = ctx['prev']['skycore rev']
    newrev = ctx['next']['skycore rev']
    if oldrev != newrev:
        info(
            ctx,
            'skycore revision changed %s => %s - restart skycore',
            oldrev,
            newrev,
        )
        return True

    return False


def skycore_migration_needed(ctx):
    old_var = ctx['paths']['var'].join('skycore')
    if old_var.check(dir=1, exists=1) and old_var.join('skycore.state').check(exists=1):
        info(ctx, "skycore migration is required => stop all skycore services")
        return True

    return False


def stop_skycore(ctx, error_message, timeout=None, full=True, fail=True):
    global FAIL_REASON, RESTART_SKYCORE_SERVICES

    if not Path(ctx['paths']['skyctl']).check(exists=1):
        return

    timeout = timeout or ctx['args'].stop_timeout
    no_lock = ['--no-lock'] if NO_LOCK_SKYCORE else []

    if full:
        for ns in _get_skycore_namespaces(ctx):
            if manage_skynet(
                ctx, no_lock + ['stop', ns],
                timeout=timeout, binname='skyctl', keep_autostart=True
            ) is False:
                error(ctx, error_message)
                RESTART_SKYCORE_SERVICES = 'restart_all_services\n'

    # we have to wait if skyctl.lock is busy
    deadline = time.time() + timeout
    stopped_okay = False
    skycore_bind_host = '127.0.0.1'
    skycore_bind_port = 10005

    while True:
        timeout = deadline - time.time()
        if timeout <= 0:
            raise Exception('skyctl shutdown timed out')

        result, stdout, stderr = manage_skynet(
            ctx, no_lock + ['shutdown'],
            binname='skyctl',
            timeout=timeout,
            nice=ctx['nice'],
            getresult=True,
            keep_autostart=True,
        )

        if result == 0:
            stopped_okay = True
            break

        elif result == 2:
            # lock is busy
            time.sleep(1)

        else:
            if fail:
                FAIL_REASON = error_message
                raise SystemExit(1)
            else:
                error(ctx, error_message)
                raise Exception(error_message)

    if not stopped_okay:
        error_message = 'skycore stop timed out'
        if fail:
            FAIL_REASON = error_message
            raise SystemExit(1)
        else:
            error(ctx, error_message)
            raise Exception(error_message)
    else:
        try:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            s.bind((skycore_bind_host, skycore_bind_port))
            s.close()
        except:
            error_message = 'skycore stopped, but port %d is still used' % (skycore_bind_port, )
            if fail:
                FAIL_REASON = error_message
                raise SystemExit(1)
            else:
                error(ctx, error_message)
                raise Exception(error_message)


def remove_srvmngr_services(ctx):  # STEP {{{
    info(ctx, 'Dropping services under srvmngr...')
    with indent(ctx, quiet_prefix='drop old services'):
        for path in ctx['paths']['services'].listdir():
            info(ctx, path)
            path.remove()

        for path in ctx['paths']['managed'].listdir():
            info(ctx, path)
            path.remove()

    kqueue_targ_path = ctx['paths']['real'].join('kqueue')
    info(ctx, 'Removing old kqueue if any (no kqueue should be here)')
    if kqueue_targ_path.check(exists=1):
        kqueue_targ_path.remove()
# }}}


def install_global_symlink(ctx, link_name):  # STEP {{{
    link = ctx['paths'][link_name]
    link_value = ctx['links'][link_name]

    if link.check(link=1):
        if link.readlink() == link_value:
            info(ctx, 'Symlink "%s" looks sane', link)
            return
        else:
            link.remove()

    info(ctx, 'Installing symlink "%s" -> "%s"...', link, link_value)
    link.mksymlinkto(link_value)
# }}}


def install_var_log_skynet_symlink(ctx):  # STEP {{{
    loglink = Path('/').join('var', 'log', 'skynet')
    loglink_value = ctx['paths']['var'].join('log')

    skycoreloglink = Path('/').join('var', 'log', 'skycore')
    skycoreloglink_value = ctx['paths']['varskycore'].join('log')

    for link, link_value in ((loglink, loglink_value), (skycoreloglink, skycoreloglink_value)):
        info(ctx, 'Checking log symlink at "%s"', link)
        with indent(ctx):
            if not link.dirpath().check(dir=1, exists=1):
                warn(ctx, 'Parent path "%s" is not a dir or not exists', link.dirpath())
                continue

            if link.check(link=1):
                if link.readlink() != link_value.strpath:
                    warn(ctx, 'Link exists, but points to "%s", removing...', link.readlink())
                    link.remove()
                else:
                    info(ctx, 'Link exists and looks sane (points to "%s")', link)
                    continue
            elif link.check(exists=1):
                warn(ctx, 'Path "%s" already exists but is not a symlink', link)
                continue

            info(ctx, 'Making symlink "%s" -> "%s"...', link, link_value)
            link.mksymlinkto(link_value)
# }}}


def install_init_script(ctx, src, target, update_rcd=True, executable=True):  # STEP {{{
    global FAIL_REASON

    assert not ctx['args'].local

    info(ctx, 'Installing or updating init script')

    with indent(ctx):
        initd_script = ctx['paths'][target]
        initd_script_src = ctx['paths'][src]

        cur_hash = None

        info(ctx, 'checking "%s"', initd_script)

        with indent(ctx, quiet_prefix='init script "%s": ' % (initd_script, )):
            if initd_script.check(link=1):
                warn(ctx, 'is a link, removing')
                initd_script.remove()
            elif initd_script.check(exists=1):
                if initd_script.check(file=1):
                    cur_hash = initd_script.computehash()
                else:
                    error(ctx, 'is not a file')
                    FAIL_REASON = 'Fail to install init script: not a file'
                    raise SystemExit(1)

            if cur_hash is not None:
                debug(ctx, 'already exists, current hash is %s', cur_hash)
                new_hash = initd_script_src.computehash()
                if new_hash == cur_hash:
                    info(ctx, 'up to date, do not need to update')
                else:
                    info(ctx, 'updating (new hash is %s)', new_hash)
                    with initd_script.open(mode='wb') as fp:
                        fp.write(initd_script_src.read(mode='rb'))
                        if hasattr(os, 'fsync'):
                            os.fsync(fp.fileno())
            else:
                info(ctx, 'installing')
                with initd_script.open(mode='wb') as fp:
                    fp.write(initd_script_src.read(mode='rb'))
                    if hasattr(os, 'fsync'):
                        os.fsync(fp.fileno())

            # Since we are installing init scripts only if in global mode, it is
            # safe to ensure root perms here.
            initd_script.ensure_perms(0, 0, 0o777 if executable else 0o666, mask=0o22)

        if update_rcd:
            info(ctx, 'adding "%s" to default runlevels', initd_script)
            with indent(ctx, quiet_prefix='runlevels for "%s"' % (initd_script, )):
                updatercd = py.path.local().sysfind('update-rc.d')
                if not updatercd:
                    debug(ctx, 'update-rc.d was not found')
                else:
                    runproc(ctx, [updatercd.strpath, initd_script.basename, 'defaults'], timeout=60)
# }}}


def install_launchd_scripts(ctx):  # STEP {{{
    global FAIL_REASON

    info(ctx, 'Installing launchd daemons')
    with indent(ctx):
        launchctl = py.path.local().sysfind('launchctl')
        if not launchctl:
            error(ctx, 'systemctl was not found')
            FAIL_REASON = 'Fail to install launchd daemons: no launchctl found'
            raise SystemExit(1)

        runproc(ctx, [launchctl.strpath, 'bootout', 'system/ru.yandex-team.skynetd-autostart'], timeout=60)
        runproc(ctx, [launchctl.strpath, 'bootout', 'system/ru.yandex-team.skynetd'], timeout=60)
        runproc(ctx, [launchctl.strpath, 'bootout', 'system/ru.yandex-team.skycored-autostart'], timeout=60)
        runproc(ctx, [launchctl.strpath, 'bootout', 'system/ru.yandex-team.skycored'], timeout=60)

        for source in (
            ctx['paths']['launchd_skycored'],
            ctx['paths']['launchd_skycored_autostart'],
        ):
            target = ctx['paths']['launchd'].join(source.basename)
            with target.open(mode='wb') as fp:
                fp.write(source.read(mode='rb'))
                if hasattr(os, 'fsync'):
                    os.fsync(fp.fileno())

            runproc(ctx, [launchctl.strpath, 'bootstrap', 'system', target.strpath], timeout=60)
# }}}


def install_cron_tab(ctx):  # STEP {{{
    global FAIL_REASON

    assert not ctx['args'].local

    info(ctx, 'Installing autostart crontab')

    if 'crontab' in ctx['paths']:
        pattern_skycored = re.compile('^.*skycored\s+autostart.*$')
        pattern_skyctl = re.compile('^.*skyctl\s+autoinit.*$')
        pattern_init = re.compile('^.*skynetd\s+autostart.*$')
        pattern_uppy = re.compile('^.*up.py\s+autostart.*$')

        crontab = ctx['paths']['crontab']

        with indent(ctx, quiet_prefix='crontab "%s": ' % (crontab, )):
            if crontab.check(exists=1):
                data = ''
                for line in crontab.open(mode='rb'):
                    if pattern_init.match(line) or pattern_uppy.match(line):
                        info(ctx, 'old init script invokation found')
                    elif pattern_skycored.match(line) or pattern_skyctl.match(line):
                        if line.lstrip().startswith('#'):
                            warn(ctx, 'commented in crontab')
                            return
                        elif ctx['platform'].startswith('darwin'):
                            info(ctx, 'removing from crontab, we are migrating to launchd')
                        else:
                            info(ctx, 'already installed')
                            return
                    else:
                        data += line
            else:
                data = ''

            info(ctx, 'installing new crontab')

            if not data.endswith('\n'):
                data += '\n'

            if 'initd_skycore_script' in ctx['paths']:
                init = ctx['paths']['initd_skycore_script']
                init_cmd = 'autostart'
            else:
                init = ctx['paths']['skyctl']
                init_cmd = 'autoinit --start-services'

            if not ctx['platform'].startswith('darwin'):
                crontab_txt = '*/10\t*\t*\t*\t*\troot\t'

                data += (
                    crontab_txt +
                    '[ -x %(init)s ] && '
                    '%(init)s %(init_cmd)s >/dev/null 2>&1\n' % {
                        'init': init,
                        'init_cmd': init_cmd,
                    }
                )

            try:
                with crontab.open(mode='wb') as fp:
                    fp.write(data)
                    if hasattr(os, 'fsync'):
                        os.fsync(fp.fileno())
            except py.error.EPERM:
                if ctx['platform'].startswith('darwin'):
                    warn(ctx, "Darwin forbids us to update %r, skipping" % (crontab.strpath,))
                else:
                    raise

            info(ctx, 'done')
    elif 'crond_skynetd' in ctx['paths']:
        _check_and_copy_file(ctx, 'crond', 'crond_skynetd', 'crond_skynetd_src')
# }}}


def _check_and_copy_file(ctx, dest_dir_name, dest_name, src_name):
    global FAIL_REASON

    dest_dir = ctx['paths'][dest_dir_name]
    dest = ctx['paths'][dest_name]
    src = ctx['paths'][src_name]

    cur_hash = None
    dest_dir.ensure(dir=1)

    with indent(ctx, quiet_prefix='crontab: '):
        info(ctx, 'checking "%s"', dest)
        with indent(ctx, quiet_prefix='script "%s": ' % (dest,)):
            if dest.check(link=1):
                warn(ctx, 'is a link, removing')
                dest.remove()
            elif dest.check(exists=1):
                if dest.check(file=1):
                    cur_hash = dest.computehash()
                else:
                    error(ctx, 'is not a file')
                    FAIL_REASON = 'Fail to install cron tab: not a file'
                    raise SystemExit(1)

        if cur_hash is not None:
            debug(ctx, 'already exists, current hash is "%s"', cur_hash)
            new_hash = src.computehash()
            if new_hash == cur_hash:
                info(ctx, 'up to date, do not need to update')
            else:
                info(ctx, 'updating (new hash is "%s")', new_hash)
                with dest.open(mode='wb') as fp:
                    fp.write(src.read(mode='rb'))
                    if hasattr(os, 'fsync'):
                        os.fsync(fp.fileno())
        else:
            info(ctx, 'installing')
            with dest.open(mode='wb') as fp:
                fp.write(src.read(mode='rb'))
                if hasattr(os, 'fsync'):
                    os.fsync(fp.fileno())

        # Since we are installing cron scripts only in global mode, it is
        # safe to ensure root perms here
        dest.ensure_perms(0, 0, 0o666, mask=0o22)


def install_bash_completion(ctx):  # STEP {{{
    info(ctx, 'Installing bash completion for sky')
    _check_and_copy_file(ctx, 'bash_completiond', 'sky_bash_completion', 'sky_bash_completion_src')
    info(ctx, 'Installing bash completion for skyctl')
    _check_and_copy_file(ctx, 'bash_completiond', 'skyctl_bash_completion', 'skyctl_bash_completion_src')


def fix_var_tmp_skynet(ctx):  # STEP {{{
    vartmpskynet = Path('/').join('var', 'tmp', 'skynet')
    info(ctx, 'Fixing permissions of "%s" and descendants...', vartmpskynet)
    try:
        with indent(ctx):
            if vartmpskynet.check(link=1):
                debug(ctx, 'is a link, removing...')
                vartmpskynet.remove()

            if vartmpskynet.check(exists=1) and not vartmpskynet.check(dir=1):
                debug(ctx, 'is not a dir, removing...')
                vartmpskynet.remove()

            if not vartmpskynet.check(exists=1):
                debug(ctx, 'creating...')
                vartmpskynet.ensure(dir=1)

            debug(ctx, 'fix all perms...')

            def _flt(p):
                # Ignore /var/tmp/skynet/cqueue/* and /var/tmp/skynet itself
                return (
                    p != vartmpskynet.join('cqueue') and
                    p != vartmpskynet.join('cqudp')
                )

            vartmpskynet.chown('root', 'root')
            vartmpskynet.chmod(0o1777)

            for path in itertools.chain(vartmpskynet.visit(rec=_flt, fil=_flt)):
                path.ensure_perms(ctx['perm'][0], ctx['perm'][1], (0o777, 0o666), mask=ctx['perm'][2])

    except Exception as ex:
        warn(ctx, 'Failed to fix perms on "%s": %s', vartmpskynet, ex)
# }}}


def check_skydeps_changed(ctx):
    # If deps version (python, etc.) was changed -- restart everything
    prev_deps_info = ctx['prev']['deps']
    next_deps_info = ctx['next']['deps']

    prev_deps = prev_deps_info and prev_deps_info.get('Version', None)
    next_deps = next_deps_info and next_deps_info.get('Version', None)

    if prev_deps != next_deps or next_deps is None:
        info(ctx, 'Deps version changes:')
        with indent(ctx, quiet_prefix='deps version changes: '):
            info(ctx, 'Old: %s', prev_deps)
            info(ctx, 'New: %s', next_deps)
            info(ctx, 'Will restart all services')
        return True
    return False


def migrate(ctx):  # STEP {{{
    global FAIL_REASON

    version_file = ctx['paths']['runtime'].join('.layout')
    try:
        version = int(version_file.read())
    except Exception:
        version = 0

    info(ctx, 'Migrating data paths...')
    with indent(ctx, quiet_prefix='migrate: '):
        debug(ctx, 'Current layout version: %r' % (version, ))

        if version == 3:
            info(ctx, 'Migrating 3 -> 2')
            with indent(ctx, quiet_prefix='3 -> 2: '):
                _migrate3to2(ctx)
                version_file.write('2')
                version = 2

        if version <= 0:
            info(ctx, 'Migrating 0 -> 1')
            with indent(ctx, quiet_prefix='0 -> 1: '):
                version_file.write('1')
                version = 1

        if version <= 1:
            info(ctx, 'Migrating 1 -> 2')
            with indent(ctx, quiet_prefix='1 -> 2: '):
                _migrate1to2(ctx)
                version_file.write('2')
                version = 2

        if hasattr(os, 'fsync'):
            with version_file.open(mode=fsync_read_binary_mode) as fp:
                os.fsync(fp.fileno())

        if version > 3:
            error(ctx, 'Dont know how to migrate from 3+ layout version')
            FAIL_REASON = 'Unable to migrate %r => 2' % (version, )
            raise SystemExit(1)

        _migrate_skycore(ctx)


def _migrate_skycore(ctx):
    old_var = ctx['paths']['var'].join('skycore')
    new_var = ctx['paths']['varskycore']

    if new_var.check(dir=1) and new_var.join('skycore.state').check(exists=1):  # not fresh migration
        if old_var.check(dir=1):
            debug(ctx, "just removing %r remaining from old installation", old_var.strpath)
            old_var.remove()
        else:
            debug(ctx, "old-style skycore var %r doesn't exist", old_var.strpath)
        return
    elif not old_var.check(dir=1):
        debug(ctx, "old-style skycore var %r doesn't exist", old_var.strpath)
        # if no old version, do nothing
        return

    info(ctx, "relocating old skycore data from %r to %r", old_var.strpath, new_var.strpath)
    if new_var.check(exists=1):
        new_var.remove()

    old_var.move(new_var)
    renames = (
        (new_var.join('namespaces'), new_var.join('ns')),
        (new_var.join('downloads'), new_var.join('dl')),
    )

    for old, new in renames:
        if old.check(exists=1):
            old.move(new)
            debug(ctx, "renamed %r to %r", old.strpath, new.strpath)

    old_state = new_var.join('var', 'skycore.state')
    new_state = new_var.join('skycore.state')
    if old_state.check(exists=1):
        debug(ctx, "fixing state file")
        with open(old_state.strpath, 'r') as old, open(new_state.strpath, 'w') as new:
            oldns = old_var.join('namespaces')
            oldrealns = oldns.realpath().strpath
            oldns = oldns.strpath

            oldrundir = old_var.join('var')
            oldrealrundir = oldrundir.realpath().strpath
            oldrundir = oldrundir.strpath

            newns = new_var.join('ns').strpath
            newrundir = new_var.join('var').strpath
            for line in old:
                new.write(
                    line.replace(oldns, newns)
                        .replace(oldrealns, newns)
                        .replace(oldrundir, newrundir)
                        .replace(oldrealrundir, newrundir)
                )
        old_state.remove()


def _migrate3to2(ctx):
    global FAIL_REASON

    if manage_skynet(ctx, ['--no-lock', 'stop'], timeout=ctx['args'].stop_timeout, binname='deprecated_uppy') is False:
        # TODO: kill skynet?  # noqa
        FAIL_REASON = 'Fail to stop skynet before migration (3 => 2)'
        raise SystemExit(1)
    services = ctx['paths']['services']

    if not services.check(dir=1):
        info(ctx, "Service dir %s not found", services)
        return
    if ctx['prev']['skycore rev']:
        stop_skycore(ctx, 'Fail to stop skycore services before migration (3 => 2)')

    ns = services.join('skynet')
    if ns.check():
        info(ctx, "Remove namespace directory: %s", ns)
        ns.remove()


def _migrate1to2(ctx):
    runtime = ctx['paths']['runtime']
    var = ctx['paths']['var']

    if not runtime.check(dir=1):
        return

    if not var.check(dir=1):
        var.ensure(dir=1)

    for pth in runtime.listdir():
        if (
            pth.basename in ('base', 'etc', 'var', 'managed', 'services', 'up.py.lock', 'skycore') or
            pth.basename.startswith('.')
        ):
            debug(ctx, 'Ignoring %s', pth)
            continue

        if pth.basename == 'cache':
            newpth = var.join('cacher')
        elif pth.basename == 'logger':
            newpth = var.join('log')
        elif pth.basename == 'sock':
            debug(ctx, 'Migrating sockets')

            with indent(ctx, quiet_prefix='sockets: '):
                for sock in pth.listdir():
                    if sock.basename == 'semaphorer.sock':
                        newpth = var.join('semaphorer').ensure(dir=1).join('socket')
                    else:
                        newpth = var.join(sock.basename)
                    debug(ctx, 'Migrating %s -> %s', sock, newpth)
                    if newpth.check(exists=1):
                        newpth.remove()
                    newpth.dirpath().ensure(dir=1)
                    sock.rename(newpth)

            continue
        elif pth.basename == 'cfg':
            bcfg = pth.join('batch.cfg')
            if bcfg.check(exists=1):
                newpth = ctx['paths']['etc'].join('batch.cfg')
                debug(ctx, 'Migrating %s -> %s', bcfg, newpth)
                if newpth.check(exists=1):
                    newpth.remove()
                newpth.dirpath().ensure(dir=1)
                bcfg.rename(newpth)
            continue
        else:
            newpth = var.join(pth.basename)

        debug(ctx, 'Migrating %s -> %s', pth, newpth)
        if newpth.check(exists=1):
            newpth.remove()
        newpth.dirpath().ensure(dir=1)
        pth.rename(newpth)

    kqueue = var.join('kqueue')
    if kqueue.check(exists=1, dir=1):
        authkeys = kqueue.join('authkeys')
        if authkeys.check(exists=1):
            newpth = ctx['paths']['etc'].join('auth')
            debug(ctx, 'Migrating %s -> %s', authkeys, newpth)
            if newpth.check(exists=1):
                newpth.remove()
            newpth.dirpath().ensure(dir=1)
            authkeys.rename(newpth)
# }}}


def hackhackhack(ctx):
    # Extra hacks before symlink switch
    oldcoplink = ctx['paths']['active'].join('tools', 'skybone-ctl')
    newcoplink = ctx['paths']['real'].join('tools', 'skybone-ctl')

    if oldcoplink.check(link=1):
        newcoplink.mksymlinkto(oldcoplink.readlink())

    oldmdslink = ctx['paths']['active'].join('tools', 'skybone-mds-ctl')
    newmdslink = ctx['paths']['real'].join('tools', 'skybone-mds-ctl')

    if oldmdslink.check(link=1):
        newmdslink.mksymlinkto(oldmdslink.readlink())


def unsafe_main(ctx):
    global FAIL_REASON
    global NO_LOCK_SKYCORE

    systemd_found = False

    if ctx['args'].dest == '/':
        ctx['args'].dest = None

    result = 0
    if ctx['args'].made_tmpdir:
        ctx['tmpdir'] = Path(ctx['args'].made_tmpdir)
        if not ctx['tmpdir'].check(exists=1, dir=1):
            error(ctx, 'Tempdir should already be made, but it is not!')
            FAIL_REASON = 'TMPDIR was not made'
            raise SystemExit(1)
    else:
        tmpdir = ctx['args'].tmpdir
        if tmpdir is None:
            tmpdir = os.environ.get('TMPDIR', None)
        if tmpdir is None:
            tmpdir = '/tmp'

        tmpdir = Path(tmpdir)

        if not tmpdir.check(exists=1, dir=1):
            error(ctx, 'TMPDIR "%s" not exists or not directory', tmpdir)
            FAIL_REASON = 'TMPDIR not exists or not a directory'
            raise SystemExit(1)

        if not tmpdir.check(changeable=1, listable=1):
            error(ctx, 'TMPDIR "%s" not listable or not changeable', tmpdir)
            FAIL_REASON = 'TMPDIR not listable or not changeable'
            raise SystemExit(1)

        ctx['tmpdir'] = Path(tempfile.mkdtemp(prefix='skynet.install.', dir=tmpdir.strpath))
        ctx['tmpdir_made_here'] = True

    # Start logging if asked
    if ctx['args'].log:
        ctx['log'] = ctx['tmpdir'].join('skyinstall.log')
        ctx['logfp'] = ctx['log'].open(mode='wb')

    # Grab file permissions we will need to set
    get_perms(ctx)

    # If we started logging in root mode -- make proper rights on log file
    if not ctx['args'].local and ctx['args'].log:
        ctx['log'].ensure_perms(ctx['perm'][0], ctx['perm'][1], 0o666, mask=ctx['perm'][0])

    # Detect hostname and grab basic config from genisys
    get_base_config(ctx)

    # Check misc system features (niceness, arch, etc.)
    check_system(ctx)

    try:
        check_paths(ctx)
    except py.error.Error as ex:
        error(ctx, 'Error while checking paths: %s', ex)
        FAIL_REASON = 'Fail while checking paths: %s' % (ex, )
        raise SystemExit(1)

    if ctx['log']:
        ctx['logfp'].close()
        oldlog = ctx['log']
        ctx['log'] = ctx['paths']['runtime'].join('var', 'log').ensure(dir=1).join('install.log')
        if ctx['log'].check(exists=1):
            logbackup = ctx['log'].dirpath().join(ctx['log'].basename + '.bak')
            if logbackup.check(exists=1):
                logbackup.remove()
            ctx['log'].move(logbackup)
        oldlog.move(ctx['log'])

        if not ctx['args'].local:
            ctx['log'].ensure_perms(ctx['perm'][0], ctx['perm'][1], 0o666, mask=ctx['perm'][2])
        ctx['logfp'] = ctx['log'].open(mode='ab')

    check_old_skynet(ctx)

    try:
        prepare_tarballs(ctx, sys.stdin)
        fix_all_perms(ctx, path=ctx['paths']['base'])
        fix_shebang(ctx, path=ctx['paths']['base'])

        if fsync_threads:
            info(ctx, 'Waiting for fsync all files...')
            for _ in fsync_threads:
                fsync_queue.put(None)
            [t.join() for t in fsync_threads]

        skydeps_changed = check_skydeps_changed(ctx)
        info(ctx, 'Services state change list: ')
        with indent(ctx, quiet_prefix='Services state change list: '):
            info(ctx, 'explicitly stop:  ALL')

        if ctx['paths']['deprecated_uppy'].check(exists=1):
            uppylock = lock_up_py_lock(ctx, ctx['paths']['runtime'].join('up.py.lock'))

        code, stdout, stderr = manage_skynet(
            ctx, ['shutdown', '--help'],
            timeout=30,
            binname='skyctl',
            getresult=True
        )
        if code is not None and stdout and '--no-lock' in stdout:
            # it's fun but it's really False in this very case
            NO_LOCK_SKYCORE = False

        skyctllock = lock_up_py_lock(ctx, ctx['paths']['varskycore'].join('skyctl.lock'))

        if ctx['paths']['deprecated_uppy'].check(exists=1):
            code, stdout, stderr = manage_skynet(
                ctx, ['--no-lock', 'stop'],
                timeout=ctx['args'].stop_timeout,
                getresult=True
            )

            if (
                (code == 1 and 'unknown mode --no-lock' in stderr) or           # up.py
                (code == 1 and 'unknown directive \'--no-lock\'' in stderr) or  # freebsd initd
                (code == 3 and 'Usage' in stderr)                               # linux initd
            ):
                # Skynet we want to update dont support lockless up.py operations
                # So, close our lock and make old style simple stopping
                fcntl.flock(uppylock, fcntl.LOCK_UN)
                uppylock.close()
                if manage_skynet(ctx, ['stop'], timeout=ctx['args'].stop_timeout) is False:
                    # TODO: kill skynet?  # noqa
                    FAIL_REASON = 'Unable to stop skynet'
                    raise SystemExit(1)
                else:
                    uppylock = lock_up_py_lock(ctx, ctx['paths']['runtime'].join('up.py.lock'))  # reacquire lock quickly!
                    code = 0

            # code == 0: up.py stop succeeded
            # code > 0: up.py failed
            # code is None: up.py canceled (not exists)
            if code is not None and code != 0:
                FAIL_REASON = 'Unable to stop skynet'
                raise SystemExit(1)

        migrate_skycore = skycore_migration_needed(ctx)

        if ctx['prev']['skycore rev'] and (migrate_skycore or not ctx['next']['skycore rev']):
            stop_skycore(ctx, 'skycore shutdown failed')  # skycore has been removed from distro
        elif (skycore_changed(ctx) or ctx['paths']['deprecated_uppy'].check(exists=1)) and ctx['prev']['skycore rev']:
            stop_skycore(ctx, 'skycore shutdown failed', full=False, fail=False)

        migrate(ctx)
        hackhackhack(ctx)  # any more dirty hacks before symlink switch
        update_symlinks(ctx)
        cleanup_prefix(ctx)
        cleanup_skycore(ctx)
        remove_srvmngr_services(ctx)
        fix_all_perms(ctx, descend_base=False)

        if not ctx['args'].local:
            fix_var_tmp_skynet(ctx)

    except (Exception, SystemExit) as ex:
        exc_info = sys.exc_info()

        # Cleanup unpacked stuff if any, so we dont pollute free space
        try:
            if ctx['paths']['real'].check(exists=1):
                info(ctx, 'Removing "%s"...', ctx['paths']['real'])
                ctx['paths']['real'].remove()
        except Exception as ex:
            warn(ctx, 'Failed to remove "%s": %s', ctx['paths']['real'], ex)

        six.reraise(*exc_info)

    srvmngryaml = ctx['paths']['etc'].join('ya.skynet.srvmngr.yaml')
    libconfyaml = ctx['paths']['etc'].join('ya.skynet.library.config.yaml')
    if ctx['args'].local:
        srvmngryaml.write('', mode='wb')
        libconfyaml.write('local: true\n', mode='wb')
    else:
        srvmngryaml.write('user: %s\n' % (ctx['args'].user, ), mode='wb')
        libconfyaml.write('', mode='wb')

    if hasattr(os, 'fsync'):
        with srvmngryaml.open(mode=fsync_read_binary_mode) as fp:
            os.fsync(fp.fileno())
        with libconfyaml.open(mode=fsync_read_binary_mode) as fp:
            os.fsync(fp.fileno())

    if (
        'initd' in ctx['paths']
        and 'initd_skycore_script_src' in ctx['paths']
        and ctx['paths']['initd_skycore_script_src'].check(exists=1)
    ):
        install_init_script(ctx, 'initd_skycore_script_src', 'initd_skycore_script')
    if 'upstart' in ctx['paths'] and ctx['paths']['upstart'].check(exists=1, dir=1):
        install_init_script(ctx, 'upstart_skycore_job_src', 'upstart_skycore_job', update_rcd=False)
    if 'systemd' in ctx['paths'] and ctx['paths']['systemd'].check(exists=1, dir=1):
        install_init_script(ctx, 'systemd_skycore_job_src', 'systemd_skycore_job', update_rcd=False,
                            executable=False)
        with indent(ctx, quiet_prefix='systemd unit for skycore'):
            systemctl = py.path.local().sysfind('systemctl')
            if not systemctl:
                debug(ctx, 'systemctl was not found')
            else:
                systemd_found = True
                runproc(ctx, [systemctl.strpath, 'enable', ctx['paths']['systemd_skycore_job'].basename], timeout=60)
    if 'launchd' in ctx['paths']:
        install_launchd_scripts(ctx)

    result = 0
    if ctx['args'].start:
        if ctx['next']['skycore rev']:
            skyctllock.seek(0)
            skyctllock.write('autostart: enabled\nautoservices: enabled\n' + RESTART_SKYCORE_SERVICES)
            skyctllock.flush()

            # FIXME (torkve) create an unified start/stop process for all supported init-systems
            if 'launchd' in ctx['paths']:
                launchctl = py.path.local().sysfind('launchctl')
                result = runproc(ctx, [launchctl.strpath, 'start', 'ru.yandex-team.skycored'], timeout=60)

                # force starting all services
                manage_skynet(
                    ctx, ['--no-lock', 'force-update'],
                    timeout=ctx['args'].start_timeout,
                    binname='skyctl'
                )
            else:
                # we have to wait if skyctl.lock is busy
                max_wait = ctx['args'].start_timeout
                while max_wait:
                    result, stdout, stderr = manage_skynet(
                        ctx,
                        ['--no-lock', 'init', '--start-services'] + (
                            ['--restart-all-services'] if migrate_skycore else []
                        ) + (
                            ['--skydeps-changed'] if skydeps_changed else []
                        ),
                        binname='skyctl',
                        timeout=ctx['args'].start_timeout, nice=ctx['nice'],
                        getresult=True
                    )
                    if result == 2:
                        # lock is busy
                        max_wait -= 1
                        time.sleep(1)
                    else:
                        if result > 0:
                            result = 12
                        break

                # force starting all services
                manage_skynet(
                    ctx, ['--no-lock', 'force-update'],
                    timeout=ctx['args'].start_timeout,
                    binname='skyctl'
                )

                if systemd_found:
                    runproc(ctx, [systemctl.strpath, 'start', ctx['paths']['systemd_skycore_job'].basename], timeout=60)

    else:
        # we have to call skyctl init --oneshot to install all services but do not start them
        if ctx['next']['skycore rev']:
            # we have to wait if skyctl.lock is busy
            max_wait = ctx['args'].start_timeout
            while max_wait:
                result, stdout, stderr = manage_skynet(
                    ctx, ['--no-lock', 'init', '--oneshot'],
                    binname='skyctl',
                    timeout=ctx['args'].start_timeout, nice=ctx['nice'],
                    getresult=True
                )
                if result == 2:
                    # lock is busy
                    max_wait -= 1
                    time.sleep(1)
                else:
                    if result > 0:
                        result = 12
                    break

        skyctllock.seek(0)
        skyctllock.write('autostart: disabled\nautoservices: disabled\n' + RESTART_SKYCORE_SERVICES)
        skyctllock.flush()

    cleanup_bases(ctx)
    cleanup_deprecated(ctx)

    if not ctx['args'].local:
        install_global_symlink(ctx, 'skylink')
        install_global_symlink(ctx, 'skyctllink')
        if sys.platform == 'cygwin':
            install_global_symlink(ctx, 'cygpslink')

    if 'crontab' in ctx['paths'] or 'crond_skynetd' in ctx['paths']:
        install_cron_tab(ctx)

    if not ctx['args'].local:
        install_var_log_skynet_symlink(ctx)

    if 'bash_completiond' in ctx['paths']:
        install_bash_completion(ctx)

    # TODO:  # noqa
    # 1) loick supervisor/autostart

    info(ctx, 'All tasks completed.')
    FAIL_REASON = 'Failed to start some services. Details will follow soon...' if result == 12 else None
    update_info(ctx)
    raise SystemExit(result)


def update_info(ctx):
    try:
        with ctx['paths']['active'].join('.info').open('rb') as f:
            data = f.readlines()

        with ctx['paths']['active'].join('.info').open('wb') as f:
            for line in data:
                if line.startswith('state: '):
                    line = 'state: ok\n'
                f.write(line)

            if hasattr(os, 'fsync'):
                os.fsync(f.fileno())

    except Exception as ex:
        warn(ctx, 'Failed to update Skynet installation metadata: %s', str(ex))


def report(ctx, message=None, exitcode=None):
    global FAIL_REASON

    if message is not None:
        error(ctx, message)
    elif FAIL_REASON:
        error(ctx, 'Fail reason: %s', FAIL_REASON)
        message = FAIL_REASON

    if not ctx['args'].start:
        # do not send heartbeat report if 'no start' selected
        return

    name = 'SkyInstall'
    debug(ctx, 'Sending %r HeartBeat report.', name)
    data = dict.fromkeys(['version', 'url', 'revision', 'timestamp'])
    try:
        data = yaml.load(ctx['paths']['active'].join('.info').open('rb'))
    except Exception as ex:
        warn(ctx, 'Failed to fetch current Skynet installation metadata: %s', str(ex))

    data = {
        'exitcode': exitcode,
        'reason': message,
        'version': data['version'],
        'installed': data['timestamp'],
        'svn': {'url': data['url'], 'revision': data['revision']},
    }

    try:
        _eval(
            ctx,
            '''
                from api.heartbeat import client
                client.scheduleReport(
                    {0!r}, {1!r},
                    discards=['SkyInfo'],
                    incremental=False,
                    compress=False,
                    sendDelay={2},
                )
            '''.format(name, data, HB_REPORT_SEND_TIMEOUT),
            critical=False,
            active=True,
            attempts=2,
        )
        info(ctx, 'Report has been scheduled successfully.')
    except Exception as ex:
        msg = [m for m in str(ex).split('\n') if m][-1]
        warn(ctx, 'Failed to send a report via current Skynet heartbeat: %s', msg)

        import json
        import urllib2
        data = json.dumps({
            'name': name,
            'report': data,
            'source': ctx['hostname']
        })
        url = HB_REST_API_BASE_URL + '/report'
        info(ctx, 'Sending the report of %d bytes directly via %r', len(data), url)
        try:
            req = urllib2.Request(url, data, {'Content-Type': 'application/json'})
            resp = json.loads(urllib2.urlopen(req, None, HB_REPORT_SEND_TIMEOUT).read())
            if resp:
                warn(ctx, 'Server %r respond: %r', resp.info().get(HB_REST_API_SERVER_ID_HDR), resp)
        except Exception as ex:
            error(ctx, 'Unable to sent a report directly: %s', str(ex))
            if isinstance(ex, urllib2.HTTPError):
                error(ctx, 'Server %r response: %s', ex.info().get(HB_REST_API_SERVER_ID_HDR), ex.read())


def main():  # {{{
    sys.stderr = sys.stdout

    os.umask(0o022)  # Reset umask

    parser = argparse.ArgumentParser(
        prog='skynet.bin',
        usage='%(prog)s [options] dest',
        formatter_class=HelpFormatter,
        description=textwrap.dedent('''
            This is help for rich installation script which sits inside
            skynet.bin. You can launch it directly, but all arguments passed
            to skynet.bin are also forwarded to this script.

            Return codes are:

              0: everything goes fine
              1: generic error, nothing was installed probably. Old skynet may be
                 partially stopped in this case (if failed during stop and --kill
                 not specified)
              2: installed ok, but failed to start some services
        '''),
    )

    output_group = parser.add_argument_group('Output control', description=textwrap.dedent('''
        Customize output. We *never* print anything to stderr. Even errors will reside in
        stdout. You can make output more verbose or quiet by options below
    '''))

    output_group.add_argument(
        '-q', '--quiet', dest='quiet', action='store_const', const=True, default=False,
        help='Be quiet. Only warnings and errors will be printed'
    )
    output_group.add_argument(
        '-b', '--batch', dest='batch', action='store_true', default=False,
        help=textwrap.dedent('''
            Batch mode. This will switch output to one more suited
            for running on multiple servers and combining their output.
            For example, no date/time will be printed, no any "random" stuff, etc.
        ''')
    )
    output_group.add_argument(
        '--debug', dest='debug', action='store_true', default=False,
        help='Print debug info as well'
    )
    output_group.add_argument(
        '--log', dest='log', action='store_true', default=False,
        help='Write full log of operation'
    )

    generic_group = parser.add_argument_group('Generic options')
    generic_group.add_argument(
        '-t', '--tmpdir', dest='tmpdir', action='store', metavar='DIR',
        help='Temp dir to use. By default - TMPDIR env variable'
    )
    generic_group.add_argument(
        '-T', dest='made_tmpdir', action='store', metavar='DIR',
        help='Use this dir as temp (no other path will be made inside it)'
    )
    generic_group.add_argument(
        '-a', '--arch', dest='arch', action='store',
        help='Architecture to use. Will autodetect if not specified'
    )
    generic_group.add_argument(
        '-k', '--keeptemp', dest='keeptemp', action='store_const', const=True, default=False,
        help='Dont clean temp path before exiting'
    )

    install_group = parser.add_argument_group('Install customization')
    install_group.add_argument(
        '--drop-custom-services', dest='drop_custom_services', action='store_true', default=False,
        help='deprecated. Not used anymore'
    )
    install_group.add_argument(
        '--user', dest='user', action='store', default='skynet', metavar='USER',
        help='Target user. Works only in root mode.'
    )
    install_group.add_argument(
        '--user-home', dest='user_home', action='store', default='/skynet', metavar='DIR',
        help='Check user home. Works only in root mode.'
    )
    install_group.add_argument(
        '--user-class', dest='user_class', action='store', default='skynet', metavar='CLS',
        help='User class for applying limits on FreeBSD. Works only in root mode.'
    )
    install_group.add_argument(
        '--local', dest='local', action='store_true', default=False,
        help='Local install, this will install skynet under local user.'
    )
    install_group.add_argument(
        '--group', dest='group', action='store', default='skywheel', metavar='GRP',
        help='Group to set to privileged files (e.g. code).'
    )
    install_group.add_argument(
        '--prefix', dest='prefix', metavar='DIR',
        help='Use specific prefix (by default value from genisys used, usually /Berkanavt)'
    )

    startstop_group = parser.add_argument_group('Starting and stopping')
    startstop_group.add_argument(
        '-s', '--start', dest='start', action='store_true', default='auto',
        help='Force starting after installation, even if *was not* started before'
    )
    startstop_group.add_argument(
        '-r', '--force-restart', dest='restart', action='store_true', default=False,
        help='Force all services restart during the installation, even they are not changed'
    )
    startstop_group.add_argument(
        '--no-start', dest='start', action='store_false', default='start',
        help='Force not starting after installation, even if *was* started before'
    )
    startstop_group.add_argument(
        '--stop-timeout', dest='stop_timeout', action='store', type=int, default=600, metavar='N',
        help='Stop skynet timeout, in seconds'
    )
    startstop_group.add_argument(
        '--start-timeout', dest='start_timeout', action='store', type=int, default=900, metavar='N',
        help='Start skynet timeout, in seconds'
    )

    system_group = parser.add_argument_group('These options cant be used directly')
    system_group.add_argument('--version', dest='version', action='store')
    system_group.add_argument('--revision', dest='revision', type=int, action='store')
    system_group.add_argument('--skycore-rev', dest='skycore_rev', action='store', default='')
    system_group.add_argument('--url', dest='url', action='store')
    system_group.add_argument('dest', default=None, action='store', type=Path, nargs='?')

    ctx = {
        'args': parser.parse_args(),
        'quiet_prefix': [],
        'indent': 0,
        'log': None
    }
    if not ctx['args'].local:
        if os.getuid() != 0:
            error(ctx, 'You must be root to install skynet globally (or use local install via --local flag)')
            raise SystemExit(1)
        info(ctx, 'Running in root mode', color='green')
    else:
        warn(ctx, 'Running NOT in root mode, local install')

    try:
        unsafe_main(ctx)
    except SystemExit as ex:
        report(ctx, None, ex.args[0])
        raise
    except Exception:
        exc_info = sys.exc_info()
        try:
            import traceback
            trace = traceback.format_exc().strip()
            if ctx['args'].batch:
                install_group.add_argument(
                    '--prefix', dest='prefix', metavar='DIR',
                    help='Use specific prefix (by default value from genisys used, usually /Berkanavt)'
                )
                trace = trace.replace(ctx['args'].made_tmpdir, '${TMPDIR}')
            ctx['indent'] = 0  # reset indent
            report(ctx, trace, 1)
            raise SystemExit(1)
        except:
            six.reraise(*exc_info)
    finally:
        try:
            if ctx.get('tmpdir_made_here', False):
                ctx['tmpdir'].remove()
        except:
            pass
# }}}


if __name__ == '__main__':
    main()
