TMPDIR=${TMPDIR:-/tmp}
KEEPTEMP=0
QUIET=0
NOMOTD=0
BATCH=0
SELFTEST=0
GETVERSION=0
SHOWHELP=0

help() {
    echo "Usage: $0 [options] [dest]"
    echo "Options:"
    echo "  -a, --arch          Use specified arch (detected ${ARCH})"
    echo "  -n, --no-motd       Dont show motd :-/"
    echo "  -V, --version       Output version and exit"
    echo " -V2, --versionext    Output version + URL + revision and exit"
    echo "      --log           Write full logfile"
    echo "      --selftest      Run selftest instead of installation"
    echo "      --nocheck       Dont check md5 checksum of $0 (ourselves)"
    echo ""
    echo "Note: if running by root, dest defaults to \"/\""
    echo "      if running by other user it will be \".\" by default"
    echo ""

    if tty >/dev/null; then
        local ttysets=$(stty -g)
        stty raw min 0 time 50
        echo -n "Do you want to see rich script help? [yn] "
        input=$(dd if=/dev/tty bs=1 count=1 2>/dev/null)
        stty "$ttysets"
        echo ""

        if [ "$input" = "y" ]; then
            SHOWHELP=1
            NOMOTD=1
            echo ""
        elif [ "$input" = "" ]; then
            echo "You are soooo slllooowwwww!!"
            exit 1
        else
            exit 1
        fi
    else
        echo "Running in non-interactive mode -- rich help cant be shown"
        exit 1
    fi
}


parseargs() {
    ARGS=( )
    while true; do
        case $1 in
            -t | --tmpdir)      TMPDIR=$2;    shift 2 ;;
            -a | --arch)        ARCH=$2;      shift 2 ;;
            -k | --keeptemp)    KEEPTEMP=1;   ARGS[${#ARGS[@]}]="$1"; shift 1 ;;
            -q | --quiet)       QUIET=1;      ARGS[${#ARGS[@]}]="$1"; shift 1 ;;
            -n | --no-motd)     NOMOTD=1;     shift 1 ;;
            -b | --batch)       BATCH=1;      ARGS[${#ARGS[@]}]="$1"; shift 1 ;;
                 --selftest)    SELFTEST=1;   shift 1 ;;
                 --nocheck)     NO_SELFCHECK=1; shift 1;;
            -V | --version)     GETVERSION=1; shift 1 ;;
            -V2| --versionext)  GETVERSION=2; shift 1 ;;
            -h | --help)        help;         break   ;;
            '')                 break                 ;;
            *)
                ARGS[${#ARGS[@]}]="$1"; shift 1
                ;;
        esac
    done
}


log() {
    local color=$1
    local stream=$2
    local severity=$3
    local msg=""
    shift 3

    if [ "${stream}" = "stdout" ]; then
        local fd=1
    elif [ "${stream}" = "stderr" ]; then
        local fd=2
    else
        local fd=1
    fi

    reset="\033[0m"
    if [ "${color}" = "white" ]; then
        color="\033[37m"
    elif [ "${color}" = "boldred" ]; then
        color="\033[31;1m"
    elif [ "${color}" = "boldyellow" ]; then
        color="\033[33;1m"
    elif [ "${color}" = "green" ]; then
        color="\033[32m"
    elif [ "${color}" = "boldgreen" ]; then
        color="\033[32;1m"
    elif [ "${color}" = "boldblack" ]; then
        color="\033[30;1m"
    fi

    [ "${BATCH}" -eq 0 ] && msg="[$(date +%Y-%m-%d\ %H:%M:%S)]  "

    msg="${msg}[${severity}]  $*"

    if [ -t $fd ]; then
        msg="${color}${msg}${reset}"
    fi

    echo -e "${msg}" >&${fd}
}

info() {
    [ "${QUIET}" -eq 1 ] && return
    log white stdout INFO "$@"
}

warn() {
    log boldyellow stdout WARN "$@"
}

error() {
    log boldred stdout ERRR "$@"
}

byebye() {
    [ "${QUIET}" -eq 1 ] && return
    log boldblack stdout "BEER" "$@"
}

motd() {
    if [ -t 1 ]; then
        echo -ne ""\
"\033[40m\033[31;1m      _______. __  ___ ____    ____ .__   __.  _______ .___________. \033[0m\n"\
"\033[40m\033[31;1m     /       ||  |/  / \   \  /   / |  \ |  | |   ____||           | \033[0m\n"\
"\033[40m\033[31;1m    |   (----\`|  '  /   \   \/   /  |   \|  | |  |__   \`---|  |----\` \033[0m\n"\
"\033[40m\033[31;1m     \   \    |    <     \_    _/   |  . \`  | |   __|      |  |      \033[0m\n"\
"\033[40m\033[31;1m .----)   |   |  .  \      |  |     |  |\   | |  |____     |  |      \033[0m\n"\
"\033[40m\033[31;1m |_______/    |__|\__\     |__|     |__| \__| |_______|    |__|      \033[0m\n"\
"\033[40m\033[31;1m                                                                     \033[0m\n"
    else
        echo -ne ""\
"      _______. __  ___ ____    ____ .__   __.  _______ .___________.\n"\
"     /       ||  |/  / \   \  /   / |  \ |  | |   ____||           |\n"\
"    |   (----\`|  '  /   \   \/   /  |   \|  | |  |__   \`---|  |----\n"\
"     \   \    |    <     \_    _/   |  . \`  | |   __|      |  |\n"\
" .----)   |   |  .  \      |  |     |  |\   | |  |____     |  |\n"\
" |_______/    |__|\__\     |__|     |__| \__| |_______|    |__|\n\n"
    fi
}

cleantemp() {
    [ $KEEPTEMP -eq 0 ] || return

    # We were probably chmodding some directories to 000 in tests...
    if [ -e "${TMP}/selftest" ]; then
        find "${TMP}/selftest" -type f -or -type d -exec chmod 777 {} \;
    fi

    if [ "${BATCH}" -eq 0 ]; then
        info "Executing: rm -rf / ${TMP#/}..."  # Yep.. Trollmode.
    else
        info "Cleaning temp dir..."
    fi
    rm -rf "${TMP}"
}


trap_with_arg() {
    local func=$1
    shift 1
    for sig; do
        trap "$func $sig" "$sig"
    done
}


trapped() {
    local ec=$?
    cleantemp
    error "Terminated with signal: $1"
    exit $ec
}


function timer()
{
    if [[ $# -eq 0 ]]; then
        echo $(date '+%s')
    else
        local  stime=$1
        local etime=$(date '+%s')

        if [[ -z "$stime" ]]; then stime=$etime; fi

        local dt=$((etime - stime))
        local ds=$((dt % 60))
        local dm=$(((dt / 60) % 60))
        local dh=$((dt / 3600))

        if [ $dh -eq 0 ]; then
            if [ $dm -eq 0 ]; then
                printf "%02ds" $ds
            else
                printf "%02dm%02ds" $dm $ds
            fi
        else
            printf '%dh%02dm%02ds' $dh $dm $ds
        fi
    fi
}


main() {
    started=$(timer)
    detectarch
    parseargs "$@"

    [ "$NO_SELFCHECK" = "" ] && __MD5_CHECK__ "$0"

    if [ $QUIET -eq 0 ] && [ $NOMOTD -eq 0 ]; then
        motd
        info "Skycore revision: '${RELEASE_SKYCORE_REV}'"
        if [ -t 1 ]; then
            local V="Release: \033[32m${RELEASE_VERSION}\033[30m r${RELEASE_REVISION}"
            local VL=${#V}

            echo -en "\033[40m\033[30;1m ${V} \033[0m"
            for ((i=$VL; i<60; i++)); do
                echo -n " "
            done
            echo -e "\033[40m\033[30;1m (c) Cyberdyne Systems \033[0m"
        else
            local V=" Release: ${RELEASE_VERSION} r${RELEASE_REVISION}"
            local VL=${#V}

            echo -n "$V"
            for ((i=$VL; i<46; i++)); do
                echo -n " "
            done

            echo " (c) Cyberdyne Systems"
        fi
        echo -en "\n\n"
    fi

    SKIP=$(awk "/^__XTR__/ { print NR + 1; exit 0; }" $0)
    THIS=$(which $0 2>/dev/null)
    if [ ! $? -eq 0 ]; then
        error "Unable to find ourselves!"
        error
        error "This could be caused by running skynet.bin"
        error "using external sh/bash like this:"
        error "sh path/to/skynet.bin"
        error "AND if skynet.bin has no executable flag"
        error
        error "This is unsupported anyway, so set +x on skynet.bin"
        error "and run it directly"
        exit 1
    fi

    if [ ${GETVERSION} -gt 0 ]; then
        [ ${GETVERSION} -eq 3 ] && echo "Skycore revision: ${RELEASE_SKYCORE_REV}"
        [ ${GETVERSION} -lt 3 ] && echo "Version: ${RELEASE_VERSION}"
        [ ${GETVERSION} -eq 2 ] && echo "URL: ${RELEASE_URL}:${RELEASE_REVISION}"
        exit 0
    fi

    tail -n +$SKIP $THIS | tar -tf - skynet-deps_${ARCH}.tgz >/dev/null 2>&1
    ret=$?

    if [ ! $ret -eq 0 ]; then
        error "Sorry, arch '${ARCH}' not supported by this universal binary!"
        error "Supported architectures are:"
        tail -n +$SKIP $THIS | tar -tf - | grep skynet-deps | while read i; do
            tmp1=${i#*_}
            tmp2=${tmp1%.tgz}
            error "  - $tmp2"
        done
        exit 2
    fi

    local TMP
    TMP=$(mktemp -d "${TMPDIR}/skynet.install.XXXXXX" 2>&1)
    if [ ! $? -eq 0 ]; then
        error "Error: failed to create temp directory: ${TMP}"
        error "Use --tempdir switch to choose another temporary directory"
        exit 3
    fi

    trap_with_arg trapped SIGINT SIGTERM SIGHUP SIGALRM SIGABRT SIGKILL SIGPIPE SIGUSR1 SIGUSR2 SIGBUS SIGFPE SIGSEGV

    [ "${BATCH}" -eq 0 ] && info "Using tempdir: ${TMP}"

    info "Unpacking python..."
    tail -n +$SKIP $THIS | tar -xf - -O skynet-deps_${ARCH}.tgz | tar -xzf - -C "${TMP}"
    if [ ! $? -eq 0 ]; then
        error "Error: failed to unpack python"
        cleantemp
        exit 4
    fi

    info "Unpacking installation script..."
    tail -n +$SKIP $THIS | tar -C "${TMP}" -xf - skyinstall.py selftest.py tests
    if [ ! $? -eq 0 ]; then
        error "Error: failed to unpack skyinstall.py and selftest.py"
        cleantemp
        exit 5
    fi

    if [ ${SHOWHELP} -eq 0 ]; then
        if [ ${SELFTEST} -eq 0 ]; then
            info "Running rich installation by python..."
            local INSTALLER=skyinstall.py
        else
            info "Running selftest..."
            local INSTALLER=selftest.py
        fi
    else
        info "Running rich help..."
        local INSTALLER=skyinstall.py
        ARGS=( "--help" )
    fi

    [ $QUIET -eq 0 ] && echo
    tail -n +$SKIP $THIS | "${TMP}/python/bin/python" "${TMP}/${INSTALLER}" "${ARGS[@]}" \
        -T "${TMP}" -a "${ARCH}" \
        --version="${RELEASE_VERSION}" --revision="${RELEASE_REVISION}" --url="${RELEASE_URL}" \
        --skycore-rev="${RELEASE_SKYCORE_REV}"
    ret=$?
    [ $QUIET -eq 0 ] && echo

    if [ ${SHOWHELP} -eq 0 ]; then
        if [ ${SELFTEST} -eq 0 ]; then
            cleantemp
            if [ $ret -eq 0 ]; then
                info "Installation completed"
            else
                error "Installation failed with exit code ${ret}!"
            fi
        else
            cleantemp
        fi

        if [ ${BATCH} -eq 0 ]; then
            elapsed=$(timer $started)
            info "Elapsed time: $elapsed"
        fi

        if [ $ret -eq 0 ]; then
            byebye "That's all. Bye-bye ;-)"
        fi
    else
        cleantemp
        ret=1
    fi

    exit $ret
}

main "$@"
