import sys
import os
import subprocess as subproc


def runInstaller(args=[], script='skyinstall.py', env={}):
    if os.getuid() != 0:
        args = ['--local'] + args

    args = [str(arg) for arg in args]
    environment = os.environ.copy()
    environment.update(env)

    with open('data.tar', 'rb') as datafp:
        proc = subproc.Popen(
            [sys.executable, script] + args,
            stdin=datafp.fileno(), stdout=subproc.PIPE, stderr=subproc.STDOUT,
            env=environment
        )
        stdout = proc.communicate()[0]
    return proc.returncode, stdout


def testShowHelp(tmpdir):
    code, out = runInstaller(args=['-h', '-T', tmpdir.strpath])
    print(out)
    assert code == 0
    assert 'show this help message and exit' in out


def testPrefixIsALink(tmpdir):
    dest = tmpdir.join('destlink')
    dest.mksymlinkto('some/symlink')
    code, out = runInstaller(args=['-b', dest], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 0
    assert 'which actually pointing to "%s"' % (tmpdir.join('some', 'symlink'), ) in out


def testPrefixIsAFile(tmpdir):
    dest = tmpdir.join('destfile').ensure(file=1)
    code, out = runInstaller(args=['-b', dest], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 1
    assert 'Path (%-14s) "%s" is a file, but we expect dir' % ('prefix', dest, ) in out


def testCantCreateDest(tmpdir):
    inner = tmpdir.join('some/inner')
    dest = inner.join('path/to/dest')
    inner.ensure(dir=1).chmod(0500)
    code, out = runInstaller(args=['-b', dest], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 1
    assert 'We are not able to create "path" dir in "%s": is not changeable by us' % (inner, ) in out


def testNotListableDest(tmpdir):
    dest = tmpdir.join('dst').ensure(dir=1)
    dest.chmod(0)
    code, out = runInstaller(args=['-b', dest], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 1
    assert 'We are not able to list contents of %s directory' % (dest, ) in out


def testPrefixIsNotChangeableButOk(tmpdir):
    prefix = tmpdir.join('dst')
    dest = prefix.join('Berkanavt').ensure(dir=1)
    prefix.join('skynet').mksymlinkto('Berkanavt/supervisor/base/active')
    prefix.chmod(0500)
    code, out = runInstaller(args=['-b', dest], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 0


def testInvalidLightPrefixLink(tmpdir):
    lp = tmpdir.join('skynet')
    lp.mksymlinkto('invalid')

    code, out = runInstaller(args=['-b', tmpdir], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 0


def testInvalidLightPrefixLinkUnfixable(tmpdir):
    tmpdir.join('Berkanavt').ensure(dir=1)
    tmpdir.join('etc').ensure(dir=1)

    lp = tmpdir.join('skynet')
    lp.mksymlinkto('invalid')

    tmpdir.chmod(0500)
    code, out = runInstaller(args=['-b', tmpdir], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 1
    assert 'We are not able to modify "skynet" symlink' in out


def testValidLightPrefixLinkUnfixable(tmpdir):
    tmpdir.join('supervisor').ensure(dir=1)

    lp = tmpdir.join('skynet')
    lp.mksymlinkto('supervisor/base/active')

    tmpdir.chmod(0500)
    code, out = runInstaller(args=['-b', tmpdir], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)
    assert code == 0


def testInvalidActiveLinkUnfixable(tmpdir):
    bd = tmpdir.join('supervisor/base')
    bd.ensure(dir=1)

    lp = bd.join('active')
    lp.mksymlinkto('invalid')

    bd.chmod(0500)
    code, out = runInstaller(args=['-b', tmpdir], env={'STOP_AFTER_CHECKING_PATHS': '1'})
    print(out)

    assert code == 1
    assert 'We are not able to modify "active" symlink' in out
